/*! \file connection_fsm.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */


#include"ut.h"
#include"vapi.h"
#include"dmgr.h"
#include"vcore.h"
#include"msp.h"
#include "vcore_voip.h"
#include "appitf.h"

extern SToneGeneric astToneGeneric_g[];

/****************************************************************************
 * VFSM_CreateConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent\n
 *          -#  SPU_FEATURES_CONTROL\n
 *          -#  SUPVSR_CREATE_CHANNEL\n
 *          -#  VCEOPT\n
 *          -#  ECHO_CANCEL\n
 *          -#  JBOPT\n
 *          -#  DGAIN\n
 *          -#  DTMFOPT\n
 *          -#  VOPENA - is sent by default\n
 *          It depends on the field\n
 *          stVAPIConfig_g.pstVoIPChnlParams->stVopena.uVoiceOperation status\n 
 *          in global structure for default params SVAPIConfig stVAPIConfig_g.\n
 *          If it is set to VOIP_VOPENA_MODE_DISABLE (0x00) then VOPENA is ignored and\n
 *          not issued, else (>= 0x01) VOPENA command is issued.\n
 *      -#  When new channel is created using SUPVSR_CREATE_CHANNEL\n
 *          -#  Initialize it using VCORE_InitChannel\n
 *          -#  Add the request to the new channel and remove it from 
 *              Supervisory channel\n
 *          -#  Launch the pending request on Supvisory channel\n
 *      -#  All of the rest of MSP request are sent on the new channel\n
 *      -#  When request is completed give a calback or signal request 
 *          completion semaphore\n
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings are done by APPITF\n.
 *
 *
 *  \return   None
 *
 *  \param pstChnl  Firstly this will be a supervisory channel on which 
 *                  calls to this processing function will have newly
 *                  created channel as this parameter.
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */
VSTATUS VFSM_CreateConnection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	gtl_msg_t *pstMsgMulti;
	VSTATUS Status;
	SVoIPChnlParams *pstChnlParams = NULL;
	SChnl *pstSupvChnl;
	SVapiReq *pstChildVapiReq;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_CreateConnection: Entering conn(%u) state(%u) \n", ConnId, pstVapiReq->usReqState);

	pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case CREATE_CONNECTION_INIT:
		/* Check if the multi cmd mode is enabled (only for CSME and POS); if not get out */
		if ((pstChnl->pstDev->eItfType != ePCI_ITF) && (!pstChnl->pstDev->bMultiCmdEnabled))
		{
			Status = VAPI_ERR_MULTI_CMD_NOT_INITIALIZED;
			goto finish;
		}

		/* This allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SChnl *));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_CREATECONN_FAIL;
			goto finish;
		}
		/* User Data for child request is comming from the parent request */
		UT_MemCopy(pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, sizeof(SChnl *));

		pstVapiReq->usReqState = CREATE_CONNECTION_CONFIG;
		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_CreateChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VFSM_CreateChannel,	/* Child Request Handler */
				CREATE_CHANNEL_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case CREATE_CONNECTION_CONFIG:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		/* The device channel now exists.
		add the connection structure to the AVL tree
		The channel ID has been added to the device array by VFSM_CreateChannel*/
		Status = VCORE_AddConnection(pstChnl);
		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}
		/* from here new requests can be posted to the connection */

		UT_Log(VCORE, DEBUG, "VFSM_CreateConnection: Device channel(%u) created on dev(%u) conn(%u) pstChnl 0x%x\n",
			pstChnl->usMSPChnlId, pstChnl->pstDev->DevId, ConnId, pstChnl);

		/* Now initialize the channel */
		pstVapiReq->usReqState = CREATE_CONNECTION_FINISHED;

		switch (pstChnl->usConnType)
		{
		case eVOIP:
			pstChnlParams = pstChnl->pstChnlParams;;
			pstMsgMulti = UT_Calloc(1, sizeof(gtl_msg_t));
			if (pstMsgMulti == NULL)
			{
				Status = VAPI_ERR_NOMEM;
				goto finish;
			}

			pstMsgMulti->fifo = UT_Calloc(1, pstSupvChnl->pstDev->MaxMsgSize);	/* alloc the max fifo length   */
			if (pstMsgMulti->fifo == NULL)
			{
				UT_FreeMem(pstMsgMulti);
				Status = VAPI_ERR_NOMEM;
				goto finish;
			}

			/* init msg fields to defautl values*/
			VCORE_InitMsg(pstMsgMulti, pstMsgMulti->fifo, pstSupvChnl->pstDev->MaxMsgSize);

			/*Add VCEOPT with packet generation disabled */
			Status = VDEV_SetVceopt(pstMsgMulti, &(pstChnlParams->stVoiceOpt));
			if (Status != SUCCESS)
				goto finish_err;

			/*Add ECHO CANCELER depending on choseen option */
			if ( (pstChnl->pstDev->usSpuFeatureMask & SPU_FEATURE_DFEC_MASK) == SPU_FEATURE_DFEC_MASK )
			{
				Status = VDEV_SetVoipDfeCancel(pstMsgMulti, &(pstChnlParams->stDfeCan));
			}
			else
			{
				Status = VDEV_SetVoipEchoCancel(pstMsgMulti, &(pstChnlParams->stEchoCan));
			}

			if (Status != SUCCESS)
				goto finish_err;
 
			/*Add Jitter buffer Option */
			Status = VDEV_SetVoipJbopt(pstMsgMulti, &(pstChnlParams->stJbopt));
			if (Status != SUCCESS)
				goto finish_err;

			/*Add DGAIN */
			Status = VDEV_SetVoipDgain(pstMsgMulti, &(pstChnlParams->stDgain));
			if (Status != SUCCESS)
				goto finish_err;

			/*Add Tone control */
			Status = VDEV_SetToneCtrlOpt(pstMsgMulti, &(pstChnlParams->stToneCtrl));
			if (Status != SUCCESS)
				goto finish_err;

			/*Add DTMFOPT */
			Status = VDEV_SetVoipDtmfOpt(pstMsgMulti, &(pstChnlParams->stDtmfOpt));
			if (Status != SUCCESS)
				goto finish_err;

			/*Add Tonerelay */
			Status = VDEV_SetToneRelayOpt(pstMsgMulti, &(pstChnlParams->stToneRelay));
			if (Status != SUCCESS)
				goto finish_err;

			/*Add VOPENA */
			if (pstChnlParams->stVopena.mode == VOIP_VOPENA_MODE_ENABLE_RTP)
			{
				Status = VDEV_SetVoipVopena(pstMsgMulti, &(pstChnlParams->stVopena));
				if (Status != SUCCESS)
					goto finish_err;
			}

			pstVapiReq->pvExecData = pstMsgMulti;	/* Save the multilist structure */

			pstMsgMulti->channel = pstChnl->usMSPChnlId;
			VCORE_SendMSPReq(pstChnl, pstMsgMulti);
 
			break;

		case eFOIP:

			pstMsgMulti = UT_Calloc(1, sizeof(gtl_msg_t));
			if (pstMsgMulti == NULL)
			{
				Status = VAPI_ERR_NOMEM;
				goto finish;
			}

			pstMsgMulti->fifo = UT_Calloc(1, pstSupvChnl->pstDev->MaxMsgSize);	/* alloc the max fifo length   */
			if (pstMsgMulti->fifo == NULL)
			{
				UT_FreeMem(pstMsgMulti);
				Status = VAPI_ERR_NOMEM;
				goto finish;
			}

			/* init msg fields to defautl values*/
			VCORE_InitMsg(pstMsgMulti, pstMsgMulti->fifo, pstSupvChnl->pstDev->MaxMsgSize);

			/* send the following commands */
			/* Send FAXOPTS */
			Status = VDEV_SetFaxOpt(pstMsgMulti, &(pstChnl->pstChnlParams->stFaxOpts));
			if (Status != SUCCESS)
				goto finish_err;

			Status = VDEV_SetFaxLevel(pstMsgMulti, &(pstChnl->pstChnlParams->stFaxTxLevel));

			if (Status != SUCCESS)
				goto finish_err;

			pstVapiReq->pvExecData = pstMsgMulti;	/* Save the multilist structure */

			VCORE_SendMSPReq(pstChnl, pstMsgMulti);

			break;

		default:
			/* the connections is created but no initialization performed for this type of connection */
			UT_Log(VCORE, DEBUG, "VFSM_CreateConnection: conn(%u) type(%u) not initialized\n",
					ConnId, pstChnl->usConnType);

			goto finish;
			break;
		}

		break;

	case CREATE_CONNECTION_FINISHED:
 
		Status = VCORE_CheckStatusMulti(pstChnl, (gtl_msg_t *) (pstVapiReq->pvExecData), Event, pstMsg);
		UT_FreeMem(((gtl_msg_t *) (pstVapiReq->pvExecData))->fifo);
		UT_FreeMem(pstVapiReq->pvExecData);
		pstVapiReq->pvExecData = NULL;

		 /* save the Src and Dest MAC addresses of the device as the default for channel*/
		UT_MemCopy( &(pstChnl->stEthMacAddresses), &(pstChnl->pstDev->stEthMacAddresses), sizeof(SSetEthMAc));

		goto finish;	
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_CreateConnection: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	}

	return SUCCESS;
 
finish_err:
	UT_FreeMem(pstMsgMulti->fifo);
	UT_FreeMem(pstMsgMulti);

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_CreateConnection: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_CreateConnection: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	/* In these 2 cases we need to free the connection structure, the channel is not created */
	if ((Status == VAPI_ERR_CREATECONN_FAIL) || (Status == VAPI_ERR_MULTI_CMD_NOT_INITIALIZED))
	{
		/* we fall is this situation when the device channel has not been created. So we need to free the Connection stucture */ 
		VCORE_CloseConnection(pstChnl);
		UT_FreeMem(pstChnl);
	}

	/* In case of SUCCESS or error else than VAPI_ERR_CONNECTION_FAIL VAPI_ERR_MULTI_CMD_NOT_INITIALIZED the connection exists */
	UT_Log(VCORE, INFO, "VFSM_CreateConnection: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_AllocateConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent\n
 *          -#  SUPVSR_CREATE_CHANNEL\n
 *      -#  All of the rest of MSP request are sent on the new channel\n
 *      -#  When request is completed give a calback or signal request 
 *          completion semaphore\n
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings are done by APPITF\n.
 *
 *
 *  \return   None
 *
 *  \param pstChnl  Firstly this will be a supervisory channel on which 
 *                  calls to this processing function will have newly
 *                  created channel as this parameter.
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */
VSTATUS VFSM_AllocateConnection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SChnl *pstSupvChnl;
	SVapiReq *pstChildVapiReq;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_AllocateConnection: Entering conn(%u) state(%u) \n", ConnId, pstVapiReq->usReqState);

	pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case ALLOCATE_CONNECTION_INIT:
		/* This allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SChnl *));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_CREATECONN_FAIL;
			goto finish;
		}
		/* User Data for child request is comming from the parent request */
		UT_MemCopy(pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, sizeof(SChnl *));

		pstVapiReq->usReqState = ALLOCATE_CONNECTION_FINISHED;
		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_CreateChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VFSM_CreateChannel,	/* Child Request Handler */
				CREATE_CHANNEL_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case ALLOCATE_CONNECTION_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		/* The device channel now exists.
		add the connection structure to the AVL tree
		The channel ID has been added to the device array by VFSM_CreateChannel*/
		Status = VCORE_AddConnection(pstChnl);
		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}
		/* from here new requests can be posted to the connection */

		UT_Log(VCORE, DEBUG, "VFSM_AllocateConnection: Device channel(%u) created on dev(%u) conn(%u) pstChnl 0x%x\n",
			pstChnl->usMSPChnlId, pstChnl->pstDev->DevId, ConnId, pstChnl);

		 /* save the Src and Dest MAC addresses of the device as the default for channel*/
		UT_MemCopy( &(pstChnl->stEthMacAddresses), &(pstChnl->pstDev->stEthMacAddresses), sizeof(SSetEthMAc));

		goto finish;	
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_AllocateConnection: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	}

	return SUCCESS;
 
finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_AllocateConnection: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_AllocateConnection: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	if ((Status != SUCCESS))
	{
		/* we fall is this situation when the device channel has not been created. So we need to free the Connection stucture */ 
		VCORE_CloseConnection(pstChnl);
		UT_FreeMem(pstChnl);
	}

	UT_Log(VCORE, INFO, "VFSM_AllocateConnection: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_DestroyConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Stop first the connection (VOPENA 0)
 *      -#  The destroy the channel or the participant accordingly
 *      -#  Cleanup the channel ot the Participant from device manager
 *
 *  - Assumptions
 *      -#  This FSM handler can be only called as a child request from a another FSM handler
 *	processing a request on a SUPERVISOR channels
 *
 *  \return 
 *  VAPI_ERR_INVALID_PARAM Invalid parameter supplied
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_DestroyConnection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SChnl *pstSupvChnl;
	SVapiReq *pstChildVapiReq;
	U16 * pusChannelId;
	SFsmParticipantInfo *ParticipantInfo;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_DestroyConnection: Entering conn(%u) state(%u) \n", ConnId, pstVapiReq->usReqState);

	pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);

	switch (pstVapiReq->usReqState)
	{
	case DESTROY_CONN_INIT:

		/*Wait the child to finish on DESTROY_STOP_CONN_WAIT_CHILD state */
		if (pstChnl->ePartType == eNOTPART)
			pstVapiReq->usReqState = DESTROY_CONN_DESTROY_CHANNEL;
		else
			pstVapiReq->usReqState = DESTROY_CONN_DESTROY_PART;

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_StopConnection,	/* Child Request Handler */
				STOP_CONN_INIT);	/* Child Request handler state */
		
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case DESTROY_CONN_DESTROY_CHANNEL:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DESTROY_CONN_FINISHED;
		/* This allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(U16));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest)
		The channel id is used by the DestroyChannel command */
		pusChannelId = pstChildVapiReq->pvUserData;

		*(pusChannelId) = pstChnl->usMSPChnlId;
		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_DestroyChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VFSM_DestroyChannel,	/* Child Request Handler */
				DESTROY_CHANNEL_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case DESTROY_CONN_DESTROY_PART:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DESTROY_CONN_FINISHED;
		/* This allocate a new request (and UserData memory in this new request*/ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SFsmParticipantInfo));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest)
		It contains the channel and conference Id */
		ParticipantInfo = pstChildVapiReq->pvUserData;

		ParticipantInfo->DevPartId = pstChnl->usMSPChnlId;
		ParticipantInfo->DevConfId = pstChnl->pstConf->usMSPConfId;

		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */
				VFSM_DestroyParticipant,	/* Child Request Handler */
				DESTROY_PARTICIPANT_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case DESTROY_CONN_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_DestroyConnection: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_DestroyConnection: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_DestroyConnection: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);
	if (Status == SUCCESS)
	{
		/* We finished handling the user request, now clear it's request list ... */
		VCORE_CloseConnection(pstChnl);
		/* free the channel */
		UT_FreeMem(pstChnl);
	}
	else
		/* put back connection to the AVL tree to be able to re-issue commands*/
		VCORE_AddConnection(pstChnl);

	UT_Log(VCORE, INFO, "VFSM_DestroyConnection: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;

}

/****************************************************************************
 * VFSM_DestroyOldConnection : The function does the following things -
 ***************************************************************************/
VSTATUS VFSM_DestroyOldConnection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SChnl *pstSupvChnl;
	SVapiReq *pstChildVapiReq;
	U16 * pusChannelId;
	SFsmParticipantInfo *ParticipantInfo;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_DestroyOldConnection: Entering conn(%u) state(%u) \n", ConnId, pstVapiReq->usReqState);

	pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);

	switch (pstVapiReq->usReqState)
	{
	case DESTROY_CONN_INIT:

		/*Wait the child to finish on DESTROY_STOP_CONN_WAIT_CHILD state */
		if (pstChnl->ePartType == eNOTPART)
			pstVapiReq->usReqState = DESTROY_CONN_DESTROY_CHANNEL;
		else
			pstVapiReq->usReqState = DESTROY_CONN_DESTROY_PART;

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VFSM_StopConnection,	/* Child Request Handler */
				STOP_CONN_INIT);	/* Child Request handler state */
		
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case DESTROY_CONN_DESTROY_CHANNEL:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DESTROY_CONN_FINISHED;
		/* This allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(U16));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest)
		The channel id is used by the DestroyChannel command */
		pusChannelId = pstChildVapiReq->pvUserData;

		*(pusChannelId) = pstChnl->usMSPChnlId;
		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_DestroyChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VFSM_DestroyChannel,	/* Child Request Handler */
				DESTROY_CHANNEL_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case DESTROY_CONN_DESTROY_PART:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DESTROY_CONN_FINISHED;
		/* This allocate a new request (and UserData memory in this new request*/
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SFsmParticipantInfo));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest)
		It contains the channel and conference Id */
		ParticipantInfo = pstChildVapiReq->pvUserData;

		ParticipantInfo->DevPartId = pstChnl->usMSPChnlId;
		ParticipantInfo->DevConfId = pstChnl->pstConf->usMSPConfId;

		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */
				VFSM_DestroyParticipant,	/* Child Request Handler */
				DESTROY_PARTICIPANT_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case DESTROY_CONN_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_DestroyOldConnection: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_DestroyOldConnection: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_DestroyOldConnection: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_DestroyOldConnection: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_LoopbackInter : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Enable/disable and sends VOIP_LOOPBACK message on both the channels.
 *
 *  \return 
 *  None
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_LoopbackInter(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SLpbackUserData *pstLoopbackUserData;
	VSTATUS Status;
	SChnl * pstChnl2nd; 
	SVapiReq * pstChildVapiReq;
	SVapiReq **ppstLockVapiReq;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_LoopbackInter: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstLoopbackUserData = (SLpbackUserData *) pstVapiReq->pvUserData;

	/* retrieve the 2nd channel involved in the loopback session*/
	pstChnl2nd = DMGR_GetConnection(pstLoopbackUserData->ConnId2);
	/* Make sure the 2nd channel is still there */
	if (pstChnl2nd == NULL)
	{
		UT_ErrorLog(APPITF, "VFSM_LoopbackInter: Connection Id(%u) doesn't exist\n", pstLoopbackUserData->ConnId2);
		Status = VAPI_ERR_INVALID_CONNID;
		goto finish;
	}

	switch (pstVapiReq->usReqState)
	{
	case LOOPBACK_INTER_INIT:
		/* now we want to make sure no other request is processed to the 2nd channel 
		before this one is ended. Proceed with a inter connection locking */
		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}
		/* save the pointer of this request in execdata*/
		pstVapiReq->pvExecData = (U8 *) UT_AllocMem(sizeof(SVapiReq *));
		if (!pstVapiReq->pvExecData)
		{
			UT_FreeMem(pstChildVapiReq);
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		ppstLockVapiReq = pstVapiReq->pvExecData;
		*ppstLockVapiReq = pstChildVapiReq;

		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
			pstVapiReq,			/* Parent Request */ 
			VFSM_LockConnection,		/* Child Request Handler */
			LOCK_CONN_INIT);		/* Child Request handler state */

		pstVapiReq->usReqState = LOOPBACK_INTER_CHNL1;
		/*add it to the queue of the current 2nd connection for process */
		Status = VCORE_ProcessLockRequest(pstChnl2nd, pstChildVapiReq);

		break;

	case LOOPBACK_INTER_CHNL1:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
		if (Status != SUCCESS)
			goto finish;

		if (Event != FSM_EVENT_LOCKED)
			UT_ErrorLog(APPITF, "VFSM_LoopbackInter: Something is going wrong in inter Locking\n");

		/* The 2nd channel has now a request posted in its queue
		no other request posted from the user application will be processed before 
		we remove the request.
		We can also use the VCORE_StartChannelChildRequest() for this channel */

		/**********************************************************************************/
		/* FROM HERE WE NEED TO MAKE SURE TO UNLOCK THE SECOND CHANNEL IN CASE OF FAILURE */
		/* ONLY CHILD REQUEST CAN BE POSTED TO THE LOCKED CONNECTION			 */
		/* NO DIRECT CALL TO VDEV_xxx functions						 */
		/**********************************************************************************/

		pstVapiReq->usReqState = LOOPBACK_INTER_CHNL2;
		/* Enable Interchannel loopback for the 1st connection */
		Status = VDEV_VoIPLoopback(pstChnl, pstChnl2nd->usMSPChnlId, pstLoopbackUserData->eLoopback);
		if (Status != SUCCESS)
			goto unlock_finish;

		break;

	case LOOPBACK_INTER_CHNL2:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_LOOPBACK);
		if (Status != SUCCESS)
			goto unlock_finish;

		/* The second channel is under interlocking;
		we need to run the request using child request mechanism */
		/* this allocate a new request  */ 
		pstChildVapiReq = VCORE_AllocateRequest(sizeof(SLpbackUserData));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto unlock_finish;
		}

		/* pass the userdata from parent to child request */
		((SLpbackUserData *) pstChildVapiReq->pvUserData)->eLoopback = pstLoopbackUserData->eLoopback;
		((SLpbackUserData *) pstChildVapiReq->pvUserData)->ConnId2 = pstChnl->usMSPChnlId;

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */ 
				VFSM_InterChannelLoopback,	/* Child Request Handler */
				INTERCHANNEL_INIT);		/* Child Request handler state */

		pstVapiReq->usReqState = LOOPBACK_INTER_FINISHED;
		/*add it to the front of the queue the 2 connection for process */
		VCORE_StartChannelChildRequest(pstChnl2nd, pstChildVapiReq);

		break;

	case LOOPBACK_INTER_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto unlock_finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_LoopbackInter: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

unlock_finish:
	/* we have finished with the 2nd channel
	 so remove the locking request from its queue by calling the FSM lock handler directly
	(stored in pstVapiReq->pvExecData) */
	ppstLockVapiReq = pstVapiReq->pvExecData;

	VFSM_LockConnection(pstChnl2nd, *(ppstLockVapiReq), FSM_EVENT_UNLOCK, NULL);

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_LoopbackInter: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_LoopbackInter: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_LoopbackInter: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;
}

/****************************************************************************
 * VFSM_LoopbackSingle : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *
 *  \return 
 *  None
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_LoopbackSingle(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SLpbackUserData *pstLoopbackUserData;
	VSTATUS Status;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_LoopbackSingle: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstLoopbackUserData = (SLpbackUserData *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case SINGLE_LOOPBACK_INIT:

		pstVapiReq->usReqState = SINGLE_LOOPBACK_FINISHED;
		/* Destination id not used for single loopback */
		Status = VDEV_VoIPLoopback(pstChnl, 0, pstLoopbackUserData->eLoopback);
		if (Status != SUCCESS)
			goto finish;

		break;

	case SINGLE_LOOPBACK_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_LOOPBACK);

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_LoopbackSingle: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_LoopbackSingle: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_LoopbackSingle: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_LoopbackSingle: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;
}

/****************************************************************************
 * VFSM_LoopbackTHC : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# For THC loopback, create and send THC_MODE_ENABLE message 
 *          on both the channels. Then THC_REDIRECT_RX message is sent
 *          on the first channel.  
 *
 *  \return 
 *  None
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_LoopbackTHC(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SLpbackUserData *pstLoopbackUserData;
	VSTATUS Status;
	IN SChnl * pstChnl2nd;
	SVapiReq **ppstLockVapiReq;
	struct _VOIP_VOPENA *pstVopena1, *pstVopena2;
	SVapiReq * pstChildVapiReq;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_LoopbackTHC: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstLoopbackUserData = (SLpbackUserData *) pstVapiReq->pvUserData;

	/* retrieve the 2nd channel involved in the loopback session*/
	pstChnl2nd = DMGR_GetConnection(pstLoopbackUserData->ConnId2);
	/* Make sure the 2nd channel is still there */
	if (pstChnl2nd == NULL)
	{
		UT_ErrorLog(APPITF, "VFSM_LoopbackTHC: Connection Id(%u) doesn't exist\n", pstLoopbackUserData->ConnId2);
		Status = VAPI_ERR_INVALID_CONNID;
		goto finish;
	}

	if ((pstChnl->usConnType != eVOIP) && (pstChnl2nd->usConnType != eVOIP) &&
		(pstChnl->usConnType != eFOIP) && (pstChnl2nd->usConnType != eFOIP))
	{
		UT_ErrorLog(APPITF, "VFSM_LoopbackTHC: Invalid channel type %d-%d\n", pstChnl->usConnType, pstChnl2nd->usConnType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto finish;
	}

	switch (pstVapiReq->usReqState)
	{
	case THC_ENABLE_LOOPBACK_INIT:

		pstVopena1 = &(pstChnl->pstChnlParams->stVopena);
		pstVopena2 = &(pstChnl2nd->pstChnlParams->stVopena);
		/*Validate if vopena settings are OK for THC mode */
		if ((pstVopena1->mode == VOIP_VOPENA_MODE_DISABLE) || 
			(pstVopena2->mode == VOIP_VOPENA_MODE_DISABLE))
		{
			UT_ErrorLog(APPITF, "VFSM_LoopbackTHC: Failed connection disabled,"
					    "ConnId1 = %u, ConnId2 = %u, Status = %d\n",
					    ConnId, pstChnl2nd->ConnId,
					    VAPI_ERR_INVALID_OPERATION);
			Status = VAPI_ERR_INVALID_OPERATION;
			goto finish;
		}

		if (pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.compander !=
			pstChnl2nd->pstChnlParams->stVoiceOpt.param_4.bits.compander)
		{
			UT_ErrorLog(APPITF, "VFSM_LoopbackTHC: Failed encoding Std",
				    "do not match,ConnId1 = %u, ConnId2 = %u\n",
				    ConnId, pstChnl2nd->ConnId);
			Status = VAPI_ERR_INVALID_OPERATION;
			goto finish;
		}

		/*Check for G.711 mode */
		if (((pstVopena1->param_5.bits.rtp_pt != PT_G711_U_LAW) && 
			(pstVopena2->param_5.bits.rtp_pt != PT_G711_A_LAW)) || 
			((pstVopena1->param_5.bits.rtp_pt != PT_G711_U_LAW) && 
			(pstVopena2->param_5.bits.rtp_pt != PT_G711_A_LAW)))
		{
			UT_ErrorLog(APPITF, "VFSM_LoopbackTHC: Failed payload type"
				    " is not G711 ,ConnId1 = %u, ConnId2 = %u\n",
				    ConnId, pstChnl2nd->ConnId);
			Status = VAPI_ERR_INVALID_OPERATION;
			goto finish;
		}

		/* now we want to make sure no other request is processed to the 2nd channel 
		before this one is ended. Proceed with a inter connection locking */
		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}
		/* save the pointer of this request in execdata*/
		pstVapiReq->pvExecData = (U8 *) UT_AllocMem(sizeof(SVapiReq *));
		if (!pstVapiReq->pvExecData)
		{
			UT_FreeMem(pstChildVapiReq);
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		ppstLockVapiReq = pstVapiReq->pvExecData;
		*ppstLockVapiReq = pstChildVapiReq;

		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
			pstVapiReq,			/* Parent Request */ 
			VFSM_LockConnection,		/* Child Request Handler */
			LOCK_CONN_INIT);		/* Child Request handler state */

		pstVapiReq->usReqState = THC_ENABLE_LOOPBACK_ENABLE_1;
		/*add it to the queue of the current 2nd connection for process */
		Status = VCORE_ProcessLockRequest(pstChnl2nd, pstChildVapiReq);

		break;

	case THC_ENABLE_LOOPBACK_ENABLE_1:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
		if (Status != SUCCESS)
			goto finish;

		if (Event != FSM_EVENT_LOCKED)
			UT_ErrorLog(APPITF, "VFSM_LoopbackInter: Something is going wrong in inter Locking\n");

		/* The 2nd channel has now a request posted in its queue
		no other request posted from the user application will be processed before 
		we remove the request.
		We can also use the VCORE_StartChannelChildRequest() for this channel */

		/**********************************************************************************/
		/* FROM HERE WE NEED TO MAKE SURE TO UNLOCK THE SECOND CHANNEL IN CASE OF FAILURE */
		/* ONLY CHILD REQUEST CAN BE POSTED TO THE LOCKED CONNECTION			 */
		/* NO DIRECT CALL TO VDEV_xxx functions						 */
		/**********************************************************************************/

		pstVapiReq->usReqState = THC_ENABLE_LOOPBACK_ENABLE_2;
		/* Enable THC fort 1st connection */
		Status = VDEV_ThcModeEnable(pstChnl);
		if (Status != SUCCESS)
			goto unlock_finish;

		break;

	case THC_ENABLE_LOOPBACK_ENABLE_2:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_THC_MODE_ENABLE);

		if (Status != SUCCESS)
			goto unlock_finish;

		/* The second channel is under interlocking;
		we need to run the request using child request mechanism*/
		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto unlock_finish;
		}

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_EnableTHC,		/* Child Request Handler */
				ENABLE_THC_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = THC_ENABLE_LOOPBACK_ENABLE_REDIRECT;
		/*add it to the front of the queue the 2 connection for process */
		VCORE_StartChannelChildRequest(pstChnl2nd, pstChildVapiReq);

		break;

	case THC_ENABLE_LOOPBACK_ENABLE_REDIRECT:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
		if (Status != SUCCESS)
			goto unlock_finish;

		/* retrieve the 2nd channel involved in the loopback session*/
		pstVapiReq->usReqState = THC_ENABLE_LOOPBACK_FINISHED;
		Status = VDEV_ThcRedirectRx(pstChnl, pstChnl2nd->usTimeSlot);
		if (Status != SUCCESS)
			goto unlock_finish;

		break;

	case THC_ENABLE_LOOPBACK_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_THC_REDIRECT_RX);

		goto unlock_finish;

		break;

	/* we get directly to this state when we want to stop the loopback */
	case THC_DISABLE_LOOPBACK_INIT:

		/* now we want to make sure no other request is processed to the 2nd channel 
		before this one is ended. Proceed with a inter connection locking */
		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}
		/* save the pointer of this request in execdata*/
		pstVapiReq->pvExecData = (U8 *) UT_AllocMem(sizeof(SVapiReq *));
		if (!pstVapiReq->pvExecData)
		{
			UT_FreeMem(pstChildVapiReq);
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		ppstLockVapiReq = pstVapiReq->pvExecData;
		*ppstLockVapiReq = pstChildVapiReq;

		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
			pstVapiReq,			/* Parent Request */ 
			VFSM_LockConnection,		/* Child Request Handler */
			LOCK_CONN_INIT);		/* Child Request handler state */

		pstVapiReq->usReqState = THC_DISABLE_LOOPBACK_1;
		/*add it to the queue of the current 2nd connection for process */
		Status = VCORE_ProcessLockRequest(pstChnl2nd, pstChildVapiReq);

		break;

	case THC_DISABLE_LOOPBACK_1:

		if (Event != FSM_EVENT_LOCKED)
			UT_ErrorLog(APPITF, "VFSM_LoopbackInter: Something is going wrong in inter Locking\n");

		/* The 2nd channel has now a request posted in its queue
		no other request posted from the user application will be processed before 
		we remove the request.
		We can also use the VCORE_StartChannelChildRequest() for this channel */

		/**********************************************************************************/
		/* FROM HERE WE NEED TO MAKE SURE TO UNLOCK THE SECOND CHANNEL IN CASE OF FAILURE */
		/* ONLY CHILD REQUEST CAN BE POSTED TO THE LOCKED CONNECTION			 */
		/* NO DIRECT CALL TO VDEV_xxx functions						 */
		/**********************************************************************************/

		pstVapiReq->usReqState = THC_DISABLE_LOOPBACK_2;
		Status = VDEV_VopDisable(pstChnl);
		if (Status != SUCCESS)
			goto unlock_finish;

		break;

	case THC_DISABLE_LOOPBACK_2:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);

		if (Status != SUCCESS)
			goto unlock_finish;

		/* Channel succesfully disabled update the internal variable */
		pstChnl->pstChnlParams->stVopena.mode = VOIP_VOPENA_MODE_DISABLE;

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto unlock_finish;
		}

		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
			pstVapiReq,			/* Parent Request */ 
			VFSM_StopConnection,		/* Child Request Handler */
			STOP_CONN_INIT);		/* Child Request handler state */

		pstVapiReq->usReqState = THC_DISABLE_LOOPBACK_FINISHED;
		/*add it to the queue of the current connection for process */
		VCORE_StartChannelChildRequest(pstChnl2nd, pstChildVapiReq);

		break;

	case THC_DISABLE_LOOPBACK_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto unlock_finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_LoopbackTHC: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

unlock_finish:
	/* we have finished with the 2nd channel
	 so remove the locking request from its queue by calling the FSM lock handler directly
	(stored in pstVapiReq->pvExecData) */
	ppstLockVapiReq = pstVapiReq->pvExecData;

	VFSM_LockConnection(pstChnl2nd, *(ppstLockVapiReq), FSM_EVENT_UNLOCK, NULL);

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_LoopbackTHC: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_LoopbackTHC: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_LoopbackTHC: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;
}

/****************************************************************************
 * VFSM_StartCallerId : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent\n
 *          -#  FC_VOIP_CNDSETP\n
 *          -#  FC_VOIP_CNDONGEN\n
 *          -#  FC_VOIP_CNDOFFGEN\n
 *      -#  When request is completed give a calback or signal request 
 *          completion semaphore\n
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings are done by APPITF\n.
 *
 *
 *  \return   SUCCESS or Failure code.
 *
 *  \param pstChnl  This will be a channel on which 
 *                  MSP commands will be sent. 
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */
VSTATUS VFSM_StartCallerId(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SCallerIdParams *pstUserData;
	VSTATUS Status;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_StartCallerId: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SCallerIdParams *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case START_CID_INIT:

		pstVapiReq->usReqState = START_CID_GEN;
		Status = VDEV_CNDSetup(pstChnl, pstUserData);
		if (Status != SUCCESS)
			goto finish;

		break;

	case START_CID_GEN:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CND_SET_PARAMS);
		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = START_CID_FINISHED;

		if (pstUserData->ucOnhookOffhookSelect)
			Status = VDEV_CNDOnGen(pstChnl, pstUserData);
		else
			Status = VDEV_CNDOffGen(pstChnl, pstUserData);
		
		if (Status != SUCCESS)
			goto finish;

		break;

	case START_CID_FINISHED:

		if (pstUserData->ucOnhookOffhookSelect)
		{
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CND_ONHOOK_GEN);
		}
		else
		{
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CND_OFFHOOK_GEN);
		}
		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_StartCallerId: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_StartCallerId: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG,"VFSM_StartCallerId: Completing req state(%u) status(%d) conn(%u)\n",
		       pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, DEBUG, "Exiting VFSM_StartCallerId: Status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_PlayCid : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent\n
 *          -#  FC_CND_SET_PARAMS\n
 *          -#  FC_VOIP_CNDONGEN or \n
 *          -#  FC_VOIP_CNDOFFGEN\n
 *      -#  When request is completed give a calback or signal request 
 *          completion semaphore\n
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings are done by APPITF\n.
 *
 *
 *  \return   SUCCESS or Failure code.
 *
 *  \param pstChnl  This will be a channel on which 
 *                  MSP commands will be sent. 
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */
VSTATUS VFSM_PlayCid(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SVcoreCidInfo *pstUserData;
	VSTATUS Status;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_PlayCid: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SVcoreCidInfo *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case START_CID_INIT:

		pstVapiReq->usReqState = START_CID_GEN;
		Status = VDEV_SetCidParams(pstChnl);
		if (Status != SUCCESS)
			goto finish;

		break;

	case START_CID_GEN:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CND_SET_PARAMS);
		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = START_CID_FINISHED;

		VDEV_CidGen(pstChnl, pstUserData);
		if (Status != SUCCESS)
			goto finish;

		break;

	case START_CID_FINISHED:

		if (pstUserData->bIsOnHook)
		{
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CND_ONHOOK_GEN);
		}
		else
		{
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CND_OFFHOOK_GEN);
		}
		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_PlayCid: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_PlayCid: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG,"VFSM_PlayCid: Completing req state(%u) status(%d) conn(%u)\n",
		       pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, DEBUG, "Exiting VFSM_PlayCid: Status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_StopCallerId : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent\n
 *          -#  FC_VOIP_CNDSTOP\n
 *      -#  When request is completed give a calback or signal request 
 *          completion semaphore\n
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings are done by APPITF\n.
 *
 *
 *  \return   SUCCESS or Failure code.
 *
 *  \param pstChnl  This will be a channel on which 
 *                  MSP commands will be sent..
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */
VSTATUS VFSM_StopCallerId(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_StopCallerId: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case STOP_CID_INIT:
		pstVapiReq->usReqState = STOP_CID_FINISHED;
		Status = VDEV_CNDStop(pstChnl);
		if (Status != SUCCESS)
			goto finish;

		break;

	case STOP_CID_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CND_STOP);
		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_StopCallerId: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;
finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_StopCallerId: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_StopCallerId: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_StopCallerId: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;
}

/****************************************************************************
 * VFSM_StartTone : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent\n
 *          -#  FC_VOIP_TONEGEN\n
 *      -#  When request is completed give a calback or signal request 
 *          completion semaphore\n
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings of Tone are done by APPITF\n.
 *
 *
 *  \return   SUCCESS or Failure code.
 *
 *  \param pstChnl  This will be a channel on which 
 *                  MSP commands will be sent..
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 *
 */
VSTATUS VFSM_StartTone(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SToneGenParams *pstUserData;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_StartTone: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SToneGenParams *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case START_TONE_INIT:

		pstVapiReq->usReqState = START_TONE_FINISHED;
		Status = VDEV_StartTonGen(pstChnl, pstUserData);
		if (Status != SUCCESS)
			goto finish;

		break;

	case START_TONE_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_TONEGEN);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_StartTone: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_StartTone: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_StartTone: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_StartTone: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;
}

/****************************************************************************
 * VFSM_PlayTone : The function does the following things -
 ***************************************************************************/
VSTATUS VFSM_PlayTone(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	CONNID ConnId = pstChnl->ConnId;
	void *pstUserData = pstVapiReq->pvUserData;
	EToneType eToneType;

	UT_Log(VCORE, DEBUG, "VFSM_PlayTone: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	/*lock the mutex to lock the access to astToneGeneric_g array*/
	UT_MutexLock(&stMutexToneGenArr_g);
	eToneType = astToneGeneric_g[((SPlayToneFlags *) pstUserData)->eToneId].eToneType;

	switch (pstVapiReq->usReqState)
	{
	case PLAY_TONE_INIT:
		pstVapiReq->usReqState = PLAY_TONE_FINISHED;

		if (eToneType == eDualTone)
			Status = VDEV_PlayDualTone(pstChnl, pstUserData);
		else
			Status = VDEV_PlayQuadTone(pstChnl, pstUserData);

		if (Status != SUCCESS)
			goto finish;

		break;

	case PLAY_TONE_FINISHED:
		if (eToneType == eDualTone)
		{
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_TONEGEN);
		}
		else
		{
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_QUADTONEGEN);
		}

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_PlayTone: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;
	}

	UT_MutexUnLock(&stMutexToneGenArr_g);
	return SUCCESS;

finish:
	UT_MutexUnLock(&stMutexToneGenArr_g);
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_PlayTone: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_PlayTone: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_PlayTone: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;
}

/****************************************************************************
 * VFSM_StopTone : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent\n
 *          -#  FC_VOIP_TONEOFF\n
 *      -#  When request is completed give a calback or signal request 
 *          completion semaphore\n
 *
 *  \return   SUCCESS or Failure code.
 *
 *  \param pstChnl  This will be a channel on which 
 *                  MSP commands will be sent..
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 *
 */
VSTATUS VFSM_StopTone(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SStopToneData *pstUserData;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_StopTone: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SStopToneData *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case STOP_TONE_INIT:

		pstVapiReq->usReqState = STOP_TONE_FINISHED; 
		Status = VDEV_ToneOff(pstChnl, pstUserData);
		if (Status != SUCCESS)
			goto finish;
		break;

	case STOP_TONE_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_TONEOFF);
		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_StopTone: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_StopTone: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_StopTone: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_StopTone: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SetCodecType : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Check the Codec passed by the user\n
 *      -# Now as codec type is directly indexed with the codec info\n
 *         struct,so use the Payload value of the struct and issue Vopena\n
 *      -# Check status of Vopena if it fails set the old payload type value\n
 *
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_SetCodecType(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	ECodecType *peUserData = NULL;
	CONNID ConnId = pstChnl->ConnId;
	U8 ucPtVal;

	UT_Log(VCORE, DEBUG, "VFSM_SetCodecType: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	if ((pstChnl->usConnType != eVOIP) && (pstChnl->usConnType != eFOIP))
	{
		UT_ErrorLog(APPITF, "VFSM_SetCodecType: Invalid channel type %d\n", pstChnl->usConnType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto finish;
	}

	peUserData = (ECodecType *) pstVapiReq->pvUserData;

	if (pstChnl->MediaSubLayer == eAudioSubLayer)
		ucPtVal = pstChnl->pstChnlParams->ucAudioPt[(*peUserData)];
	else
		ucPtVal = pstChnl->pstChnlParams->ucVideoPt[(*peUserData)];

	if (ucPtVal == 0xff)
	{
		Status = VAPI_ERR_PT_NOT_INITIALIZED;
		UT_ErrorLog(VCORE, "VFSM_SetCodecType: connid (%u) PT (%d) not initialized\n", ConnId, (*peUserData));
		goto finish;
	}

	switch (pstVapiReq->usReqState)
	{
	case SET_CODEC_INIT:

		/* We cant handle this function on non VOIP connetion
		Changing codec of T38 is not allowed */
		if (pstChnl->usConnType != eVOIP)
		{
			Status = VAPI_ERR_UNSUPP_FEATURE;
			goto finish;
		}
		/* Changing codec of a local partcipant doesn't make sense 
		Note it is not prohibited */
		if (pstChnl->ePartType == eLSP)
		{
			Status = VAPI_ERR_INVALID_OPERATION;
			goto finish;
		}

		pstVapiReq->pvExecData = UT_AllocMem(sizeof(SCodecData));
		if (pstVapiReq->pvExecData == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		UT_MemSet(pstVapiReq->pvExecData, 0, sizeof(SCodecData));
		((SCodecData *) (pstVapiReq->pvExecData))->Status = SUCCESS;

		/* Backup of current payload type */
		((SCodecData *) (pstVapiReq->pvExecData))->uiCurrRTPHeaderPT = pstChnl->pstChnlParams->stVopena.param_5.bits.rtp_pt;

		/* update the internal variable according the codec index */
		pstChnl->pstChnlParams->stVopena.param_5.bits.rtp_pt = ucPtVal;

		/* Now if the channel is not active we want to stop the process here
		(no command sent to the channel).
		The new codec will be used on next channel enable  */
		if (pstChnl->pstChnlParams->stVopena.mode == VOIP_VOPENA_MODE_DISABLE)
		{
			/* update the internal variable codec index */
			pstChnl->pstChnlParams->eCodecVal = (*peUserData);

			Status = SUCCESS;
			goto finish;
		}

		/* The channel is active we want to disable it first */

		/*Fill the message with vop disable */
		pstChnl->pstChnlParams->stVopena.mode = VOIP_VOPENA_MODE_DISABLE;

		pstVapiReq->usReqState = SET_CODEC_WAIT_VOPDIS;
		/* send the command to the channel */
		Status = VDEV_SendVoipVopena(pstChnl, &(pstChnl->pstChnlParams->stVopena));

		if (Status != SUCCESS)
			goto finish;

		break;
 
	case SET_CODEC_WAIT_VOPDIS:
		/* received response from the device */
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);

		if (Status != SUCCESS)
			goto finish;

		/*Fill the message with vop enable */
		pstChnl->pstChnlParams->stVopena.mode = VOIP_VOPENA_MODE_ENABLE_RTP;

		pstVapiReq->usReqState = SET_CODEC_WAIT_VOPENA;
		Status = VDEV_SendVoipVopena(pstChnl, &(pstChnl->pstChnlParams->stVopena));

		if (Status != SUCCESS)
			goto finish;

		break;

	case SET_CODEC_WAIT_VOPENA:
		/* received response from the device */
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);
		if (Status == SUCCESS)
		{
			/* the codec is succesfully changed, stop the process
			update the codec index variable  */
			pstChnl->pstChnlParams->eCodecVal = (*peUserData);
			goto finish;
		}
		else
		{
			/* retrieve the previous codec */
			pstChnl->pstChnlParams->stVopena.param_5.bits.rtp_pt =
			    ((SCodecData *) (pstVapiReq->pvExecData))->uiCurrRTPHeaderPT;

			pstVapiReq->usReqState = SET_CODEC_BACK;
			/* and send again vopen with the previous codec */
			Status = VDEV_SendVoipVopena(pstChnl, &(pstChnl->pstChnlParams->stVopena));

			if (Status != SUCCESS)
				goto finish;

		}
		break;

	case SET_CODEC_BACK:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);
 
		/* whatever success or not, the change codec operation failed */
		/* in any case the process is finished ... */
		/* ... and the codec is not changed */
		Status = VAPI_ERR_UNDEFINED;
			goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetCodecType: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	}; /* end of switch request state */

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_SetCodecType: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_SetCodecType: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);
 
	UT_Log(VCORE, INFO, "VFSM_SetCodecType: conn(%u) status (%d)\n", ConnId, Status);
 
	return Status;
 }

/****************************************************************************
 * VFSM_SetPacketInterval : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Check if the pkt generation is enabled or disabled\n
 *      -# If enabled first disable it by calling disable vopena\n
 *         and then enable it after setiing Pkt interval\n
 *      -# Check for return Status in each case
 *
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_SetPacketInterval(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	U8 *pucUserData;
	struct _VOIP_VOPENA *pstVopena;
	struct _VOIP_VCEOPT *pstVoiceParams;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_SetPacketInterval: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	switch(pstChnl->usConnType)
	{
	case eVOIP:
	case eFOIP:
		pstVopena = &(pstChnl->pstChnlParams->stVopena);
		pstVoiceParams = &(pstChnl->pstChnlParams->stVoiceOpt);
		break;

	default:
		UT_Log(VCORE, DEBUG, "VFSM_SetPacketInterval: Channel Type %d not supported yet \n", pstChnl->usConnType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto finish;
	}

	pucUserData = (U8 *) pstVapiReq->pvUserData;

	/* FIXME all the following commands will be sent in multicmd mode in a later version */
	switch (pstVapiReq->usReqState)
	{
	case PACKET_INTERVAL_INIT:

		/* save the current VOPENA operation */
		pstVapiReq->pvExecData = (U8 *) UT_AllocMem(3*sizeof(U8));
		if (!pstVapiReq->pvExecData)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}
		((U8 *) pstVapiReq->pvExecData)[0] = pstVopena->mode; 
		/*Store old value*/
		((U8 *) pstVapiReq->pvExecData)[1] = pstVoiceParams->param_4.bits.packet_interval;
		pstVoiceParams->param_4.bits.packet_interval = *pucUserData;

		((U8 *) pstVapiReq->pvExecData)[2] = pstChnl->bIsActive;

		/* Disable the channel */
		pstVapiReq->usReqState = PACKET_INTERVAL_VCEOPT;
		Status = VDEV_VopDisable(pstChnl);
		if (Status != SUCCESS)
			goto finish;

		break;

	case PACKET_INTERVAL_VCEOPT:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);

		if (Status != SUCCESS)
			goto finish;

		pstChnl->bIsActive = False;
		pstVopena->mode = VOIP_VOPENA_MODE_DISABLE;
		pstVapiReq->usReqState = PACKET_INTERVAL_RENABLE;
		Status = VDEV_SendVceopt(pstChnl, pstVoiceParams);
		if (Status != SUCCESS)
			goto finish;

		break;

	case PACKET_INTERVAL_RENABLE:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VCEOPT);
		if (Status != SUCCESS)
                {
			/*return old settings*/
			pstVapiReq->usReqState = PACKET_INTERVAL_RENABLE;
			pstVoiceParams->param_4.bits.packet_interval = ((U8 *) pstVapiReq->pvExecData)[1];
			Status = VDEV_SendVceopt(pstChnl, pstVoiceParams);
			if (Status != SUCCESS)
				goto finish;
		}	
		else
		{
			/* retrieve the original vopena operation */
			pstVopena->mode = ((U8 *) pstVapiReq->pvExecData)[0];

			/* if the connection was originally disabled, it is not required 
			to redisable it*/ 
			pstChnl->bIsActive = ((U8 *) pstVapiReq->pvExecData)[2];
			if(pstChnl->bIsActive == False)
				goto finish;

			pstVapiReq->usReqState = PACKET_INTERVAL_FINISHED;
			Status = VDEV_SendVoipVopena(pstChnl, pstVopena);
			if (Status != SUCCESS)
				goto finish;
                }
		break;

	case PACKET_INTERVAL_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);

		if (Status != SUCCESS)
			pstChnl->bIsActive = False;

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetPacketInterval: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	};

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_SetPacketInterval: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_SetPacketInterval: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetPacketInterval: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SetPayloadType : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# If direction is eTx then call VoipPTset command\n
 *      -# If direction is eRx then send VoIP_PTSETRXOV command\n
 *      -# If direction is eBoth then call VoipPTset first and then\n
 *         call VoIP_PTSETRXOV command\n
 *      -# Check for return Status in each case
 *
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_SetPayloadType(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SPtmngUserData *pstUserData;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_SetPayloadType: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SPtmngUserData *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case SET_PAYLOAD_INIT:

		if (pstChnl->MediaSubLayer == eVideoSubLayer)
		{
			pstVapiReq->usReqState = SET_PAYLOAD_VIDEO_FINISHED;
			/*Video doesn't support asymetric payload */
			pstUserData->ucDir = eTx;
		}
		else
		{
			switch (pstUserData->ucDir)
			{
			case eTx:
				pstVapiReq->usReqState = SET_PAYLOAD_FINISHED;
				break;
	
			case eRx:
				pstVapiReq->usReqState = SET_PAYLOAD_FINISHED;
				break;
	
			case eBoth:
				pstVapiReq->usReqState = SET_PAYLOAD_BOTH_DIR;
				break;
	
			};
		}
		Status = VDEV_PLTypeSet(pstChnl, pstUserData->usPTIndex, pstUserData->ucPlTypVal, pstUserData->ucDir);
		if (Status != SUCCESS)
			goto finish;

		break;

	case SET_PAYLOAD_BOTH_DIR:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_PTSET);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = SET_PAYLOAD_FINISHED;
		Status = VDEV_PLTypeSet(pstChnl, pstUserData->usPTIndex, pstUserData->ucPlTypVal, eRx);
		if (Status != SUCCESS)
			goto finish;

		break;

	case SET_PAYLOAD_FINISHED:

		switch (pstUserData->ucDir)
		{
		case eTx:
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_PTSET);
			break;

		case eRx:
		case eBoth:
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_PTSETRXOVR);
			break;

		};

		/* update the VAPI variable only if the PT has been succesfully set for TX direction*/
		if ((Status == SUCCESS) && (pstUserData->ucDir != eRx))
		{
			pstChnl->pstChnlParams->ucAudioPt[(pstUserData->eCodec)] = pstUserData->ucPlTypVal;
		}

		goto finish;

		break;

	case SET_PAYLOAD_VIDEO_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VIDEO_PTSET);

		if (Status == SUCCESS)
			pstChnl->pstChnlParams->ucVideoPt[(pstUserData->eCodec)] = pstUserData->ucPlTypVal;

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetPayloadType: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	};

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_SetPayloadType: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_SetPayloadType: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetPayloadType: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_EchoCancellerReset : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Reset all the parameters to be passed\n
 *      -# Call Echo cancellor reset command and check its status\n
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_EchoCancellerReset(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_EchoCancellerReset: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case ECHOCAN_RESET_INIT:
		if (pstChnl->usConnType == eVOIP)
		{
			/*Send ECHO_CANCEL */
			pstVapiReq->usReqState = ECHOCAN_RESET_FINISHED;
			Status = VCORE_SetEchoCan(pstChnl);
		}
		else
		{
			Status = VAPI_ERR_UNSUPP_FEATURE;
			goto finish;
		}
		break;

	case ECHOCAN_RESET_FINISHED:
		if (pstChnl->pstDev->usSpuFeatureMask & SPU_FEATURE_DFEC_MASK)
		{
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_DFECAN);
		}
		else
		{
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_ECHOCAN);
		}

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_EchoCancellerReset: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	};

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_EchoCancellerReset: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_EchoCancellerReset: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_EchoCancellerReset: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SendNte : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Call SendNteEvent command and check its status\n
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_SendNte(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SNteEventUserData *pstUserData = NULL;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_SendNte: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SNteEventUserData *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case SEND_NTE_INIT:

		if (pstChnl->usConnType != eVOIP)
		{
			Status = VAPI_ERR_UNSUPP_FEATURE;
			goto finish;
		}

		pstVapiReq->usReqState = SEND_NTE_FINISHED;
		Status = VDEV_SendNteCmd(pstChnl, pstUserData->uiNtePyLd,
					  pstUserData->uiSsrc, pstUserData->usOverRideBitField,
					  pstUserData->usPyLdType, pstUserData->usRedundancyInterval);

		if (Status != SUCCESS)
			goto finish;

		break;

	case SEND_NTE_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_SENDNTE);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SendNte: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	};

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_SendNte: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_SendNte: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SendNte: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SetRtpSsrcHeader : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Set the value of Ssrc header as specified by user\n
 *      -# Call Vopena command and check its status\n
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_SetRtpSsrcHeader(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	U32 *puiUserData;
	struct _VOIP_VOPENA *pstVopena;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_SetRtpSsrcHeader: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	puiUserData = (U32 *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case RTP_SSRC_INIT:
		switch(pstChnl->usConnType)
		{
		case eVOIP:
		case eFOIP:
			/* FIXME We will need to handle this command for FoIPoRTP */
			pstVopena = &(pstChnl->pstChnlParams->stVopena);
			break;
	
		default:
			UT_Log(VCORE, DEBUG, "VFSM_SetRtpSsrcHeader: Channel Type %d not supported yet \n", pstChnl->usConnType);
			Status = VAPI_ERR_UNSUPP_FEATURE;
			goto finish;
		}

		/* update the VAPI internal variable */
		pstVopena->rtp_ssrc = *puiUserData;

		/* if the channel is currently not enbled just set the variable and exit */
		if (!(pstChnl->bIsActive))
		{
			Status = SUCCESS;
			goto finish;
		}

		/* else issue VOPENA with the new SSRC*/
		pstVapiReq->usReqState = RTP_SSRC_FINISHED;
		Status = VDEV_SendVoipVopena(pstChnl, pstVopena);
		if (Status != SUCCESS)
			goto finish;

		break;

	case RTP_SSRC_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetRtpSsrcHeader: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	};

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_SetRtpSsrcHeader: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_SetRtpSsrcHeader: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetRtpSsrcHeader: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SwitchToT38: The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent\n
 *          -#  VOPENA\n
 *          -#  SUPVSR_DESTROY_CHANNEL \n
 *          -#  SUPVSR_CREATE_CHANNEL \n
 *          -#  FAXOPT \n
 *          -#  FAXLVL \n
 *          -#  VCEOPT \n
 *          -#  SET_ETH_HDR_CHNL \n
 *          -#  SET_IP_HDR_CHNL \n
 *      -#  When new channel is created using SUPVSR_CREATE_CHANNEL\n
 *          -#  Replace MSP channel info in old channel strucure with \n
 *              newly created channel info.\n
 *          -#  Remove request from Supervisory channel\n
 *          -#  Launch the next request on Supvisory channel\n
 *      -#  The rest of MSP request are sent on the new channel\n
 *      -#  When request is completed give a calback or signal request 
 *          completion semaphore\n
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings are done by APPITF\n.
 *
 *  \return   None
 *
 *  \param pstChnl  Firstly this will be the channel on which switchover is
 *                  to take place. Then the request will be copied to the 
 *                  supervisory channel and SUPV_DESTROY_CHANNEL and 
 *                  SUPV_CREATE_CHANNEL requests will be sent. Subsequent 
 *                  calls to this processing function will have original
 *                  channel as this parameter.
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */
VSTATUS VFSM_SwitchToT38(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SVapiReq *pstChildVapiReq;
	VSTATUS Status = SUCCESS;
	SVoIPChnlParams *pstChnlParams;
	SInitChannel *pstInitChannel;
	SFsmFaxOptions *pstFsmFaxOptions;
	SChnl **ChannelInfo;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_SwitchToT38: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstFsmFaxOptions = (SFsmFaxOptions *) pstVapiReq->pvUserData;

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{

	case MODIFY_T38_INIT:
		/* FoIP switch is not allowed if the channel is not VoIP */
		if (eVOIP != pstChnl->usConnType)
		{
			UT_ErrorLog(VCORE, "VFSM_SwitchToT38: Channel Type Invalid\n");
			Status = VAPI_ERR_INVALID_CONN_TYPE;
			goto finish;
		}

		/* FoIP switch is not allowed if the channel is a conerence participant */
		if (eNOTPART != pstChnl->ePartType)
		{
			UT_ErrorLog(VCORE, "VFSM_SwitchToT38: Fax Switch not allowed on Conference participants\n");
			Status = VAPI_ERR_INVALID_CONN_TYPE;
			goto finish;
		}

		/* this allocate a new request (and User Data required) */ 
		pstChildVapiReq = VCORE_AllocateRequest(sizeof(SChnl *));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Init the UserData for the ChangeChannelType FSM handler */
		ChannelInfo = pstChildVapiReq->pvUserData;
		ChannelInfo[0] = pstChnl;

		/* Set the channel type we want */
		pstChnl->usConnType = eFOIP;

		/* wait the request child to be finished in this state*/
		pstVapiReq->usReqState = MODIFY_T38_CHANGE_TYPE_DONE;
		/* initialise the Child request 
		The pstChnl pointer is the data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */ 
				pstVapiReq,		/* Parent Request */	
				VFSM_ChangeChannelType,	/* Child Request Handler */
				CHANGE_TYPE_INIT);	/* Child Request handler state */

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case MODIFY_T38_CHANGE_TYPE_DONE:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstChnl->ucMsgIndex = 1;
		/* The IP header is not initialized yet */
		pstChnl->bIPHdrSet = False;

		pstChnlParams = pstChnl->pstChnlParams;

		if (pstChnlParams->ucAudioPt[eIFP] != 0xFF)
		{
			/* override the payload type with the FoIP PT*/
			pstChnlParams->stVopena.param_5.bits.rtp_pt = pstChnlParams->ucAudioPt[eIFP];
		}

		/* For Fax Channel limit the minimun packet interval to 20ms */
		if(pstChnlParams->stVoiceOpt.param_4.bits.packet_interval < 20)
			pstChnlParams->stVoiceOpt.param_4.bits.packet_interval = 20;

		/* this allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SInitChannel));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		pstInitChannel = pstChildVapiReq->pvUserData;
		/* For now we force full initialization up to the transport config (ETh + IP header) 
		The user will be able to configure it in the future */
		pstInitChannel->stFsmModifyChannelInfo.OpMode = VAPI_T38_OPMODE_NETWORK;

		if (pstInitChannel->stFsmModifyChannelInfo.OpMode == VAPI_T38_OPMODE_NONE)
		{
			Status = SUCCESS;
			goto finish;
		}

		/* indicate we want to initialise a Fax channel */
		pstInitChannel->stFsmModifyChannelInfo.DevChannelType = eFOIP;

		if (pstFsmFaxOptions->bUseIt == True)
			UT_MemCopy(&pstInitChannel->stFsmFaxOptions, pstFsmFaxOptions, sizeof(SFsmFaxOptions));

		UT_MemCopy(pstChildVapiReq->pvUserData, pstInitChannel, sizeof(SInitChannel));

		/* initialise the Child request */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */ 
				pstVapiReq,		/* Parent Request */	
				VFSM_InitChannel,	/* Child Request Handler */
				INIT_FAX_CHANNEL_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = MODIFY_T38_INIT_TRANSPORT;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case MODIFY_T38_INIT_TRANSPORT:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
		if (Status != SUCCESS)
			goto finish;

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* for the M821xx device we must not issue the SET_ETH_HDR_CHANNEL command
		so start the child request from SET_IP_HDR state */
		if((pstChnl->pstDev->ucDevType == DEV_TYPE_M821XX) ||
			(pstChnl->pstDev->ucDevType == DEV_TYPE_M83XXX))

			/* initialise the Child request */ 
			VCORE_SetChildRequest(pstChildVapiReq,			/* Child Request */
					pstVapiReq,				/* Parent Request */
					VFSM_InitChannelTransport,		/* Child Request Handler */
					INIT_CHANNEL_TRANSPORT_IP_PARAMS_SEND);	/* Child Request handler state */

		else
			/* initialise the Child request */ 
			VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */ 
					pstVapiReq,			/* Parent Request */	
					VFSM_InitChannelTransport,	/* Child Request Handler */
					INIT_CHANNEL_TRANSPORT_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = MODIFY_T38_FINISHED;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case MODIFY_T38_ENABLED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
		if (Status != SUCCESS)
			goto finish;

		/* if the channel has to be started continue, else finish now */
		/* for this relese we finish here the T38 switch, the channel is not enabled */
		/* the user needs to call VAPI_EnabledConnection to start UDPTL traffic */
		/*For future release the user will be able to choose if he wants to start FoIPoUDPTL or
		FoIPoRTP traffic */
		break;

	case MODIFY_T38_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SwitchToT38: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	}


	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_SwitchToT38: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_SwitchToT38: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SwitchToT38: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SetT38Options : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Stores the Fax switchover parameters in the channel structure.
 *  \param pstChnl  pointer to channel for which FAX params to be configured.
 *  \param pstMsg   Will be NULL always. Kept to preserve existing calling 
 *                  syntax from ProcessRequest.
 *  \return 
 *  SUCCESS 
 *  
 */
VSTATUS VFSM_SetT38Options(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SFsmFaxOptions *pstFaxConfigOpts;
	VSTATUS Status;
	struct _FAXOPT *pstFaxOptions;
	struct _FAXLVL *pstFaxLevel;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_SetT38Options: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	/*Extract request arguments */
	pstFaxConfigOpts = (SFsmFaxOptions *) pstVapiReq->pvUserData;
	pstFaxOptions = &(pstChnl->pstChnlParams->stFaxOpts);
	pstFaxLevel = &(pstChnl->pstChnlParams->stFaxTxLevel);

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case T38_OPTIONS_INIT:

		switch(pstChnl->usConnType)
		{
		case eVOIP:
			/* in VOIP type just copy the user passed options to the internal channel structure
			They will be used later on once the channel will be in FOIP type*/

			UT_MemCopy(pstFaxOptions, &pstFaxConfigOpts->stFaxOpts, sizeof(struct _FAXOPT));
			UT_MemCopy(pstFaxLevel, &pstFaxConfigOpts->stFaxLevel, sizeof(struct _FAXLVL));
			Status = SUCCESS;
			goto finish;
			break;

		case eFOIP:
			break;
	
		default:
			UT_Log(VCORE, DEBUG, "VFSM_SetT38Options: Channel Type %d not supported yet \n", pstChnl->usConnType);
			Status = VAPI_ERR_INVALID_PARAM;
			goto finish;
		}

		pstVapiReq->usReqState = T38_OPTIONS_FAX_LEVEL;
		/* On FoIP channel send the new options to the channel */
		Status = VDEV_SendFaxOpt(pstChnl, &pstFaxConfigOpts->stFaxOpts);
		if (Status != SUCCESS)
			goto finish;

		break;

	case T38_OPTIONS_FAX_LEVEL:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId, CMD_CLASS_CONF_CHANNEL,
					CMD_TYPE_CONF_RESP, FC_FAXOPT);
		if (Status != SUCCESS)
			goto finish;

		/* If ack to FaxOptions command update the VAPI variable*/
		UT_MemCopy(pstFaxOptions, &pstFaxConfigOpts->stFaxOpts, sizeof(struct _FAXOPT));
		pstVapiReq->usReqState = T38_OPTIONS_FAX_FINISHED;
		/* Now set the Fax Level */
		Status = VDEV_SendFaxLevel(pstChnl, &pstFaxConfigOpts->stFaxLevel);
		if (Status != SUCCESS)
			goto finish;

		break;

	case T38_OPTIONS_FAX_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId, CMD_CLASS_CONF_CHANNEL,
					CMD_TYPE_CONF_RESP, FC_FAXLVL);

		/* If ack to FaxLevel command update the VAPI variable*/
		if (Status == SUCCESS)
			UT_MemCopy(pstFaxLevel, &pstFaxConfigOpts->stFaxLevel, sizeof(struct _FAXLVL));

		goto finish;

		break;

	case T38_OPTIONS_FAX_SKIP:
		/* we get here from VAPI_ConfigureT38Options if the bUseExistingOpts is set*/
		Status = SUCCESS;
		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetT38Options: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	};

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_SetT38Options: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_SetT38Options: Completing req state(%u) status(%d) conn(%u)\n",
				pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetT38Options: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SetConnectionIPParams : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Send the SetIpHdrChnl command 
 *
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */

VSTATUS VFSM_SetConnectionIPParams(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	struct _SET_IP_HDR_CHANNEL *pstUserData;
	struct _SET_IP_HDR_CHANNEL *pstIpParams;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_SetConnectionIPParams: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (struct _SET_IP_HDR_CHANNEL *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case IP_PARAMS_INIT:

		if ((eVOIP != pstChnl->usConnType) && (eFOIP != pstChnl->usConnType) && (eIMSOTDM != pstChnl->usConnType))
		{
			UT_Log(VCORE, DEBUG, "VFSM_SetConnectionIPParams: Channel Type %d not supported yet \n", pstChnl->usConnType);
			Status = VAPI_ERR_UNSUPP_FEATURE;
			goto finish;
		}

		if ((pstChnl->pstDev->eEthHdrMode != eMODE_IP) && (pstChnl->pstDev->eEthHdrMode != eMODE_ALT))
		{
			UT_ErrorLog(APPITF, "VFSM_SetConnectionIPParams: Device ETH header not set\n");
			Status = VAPI_ERR_INVALID_OPERATION;
			goto finish;
		}

		/* The device IP address must be set when in single or multiple limited mode */
		if (((pstChnl->pstDev->eSrcIpMode == eSINGLE_IP) || (pstChnl->pstDev->eSrcIpMode == eMULTIPLE_IP))
			&& (pstChnl->pstDev->uiSrcIpAddr == 0))
		{
			UT_ErrorLog(APPITF, "VFSM_SetConnectionIPParams: Call VAPI_SetDeviceIPAddr first\n");
			Status = VAPI_ERR_DEVIP_NOTSET;
			goto finish;
		}

		pstVapiReq->usReqState = IP_PARAMS_FINISHED;
		Status = VDEV_SendIpHdrChnl(pstChnl, pstUserData);
		if (Status != SUCCESS)
			goto finish;

		break;

	case IP_PARAMS_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_SET_IP_HDR_CHANNEL);
		if (Status != SUCCESS)
			goto finish;

		switch(pstChnl->usConnType)
		{
		case eVOIP:
		case eFOIP:
		case eIMSOTDM:
			pstIpParams = &(pstChnl->pstChnlParams->stIpHdrParams);

			break;
	
		default:
			UT_Log(VCORE, DEBUG, "VFSM_SetConnectionIPParams: Channel Type %d not supported yet \n", pstChnl->usConnType);
			Status = VAPI_ERR_INVALID_PARAM;
			goto finish;
		}

		pstChnl->bIPHdrSet = True;
		/* save the IP/UDP params to the connection params */
		pstIpParams->ip_dst = pstUserData->ip_dst;
		pstIpParams->ip_src = pstUserData->ip_src;

		/*Do not save the udp port for another service than RTP */
		if(pstUserData->param_4.bits.serviceid == SET_IP_HDR_CHANNEL_SERVICEID_DEFAULTPKT)
		{
			pstIpParams->uh_sport = pstUserData->uh_sport;
			pstIpParams->uh_dport = pstUserData->uh_dport;
		}

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetConnectionIPParams: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	};

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_SetConnectionIPParams: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_SetConnectionIPParams: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetConnectionIPParams: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}


/****************************************************************************
 * VFSM_TranscodingSession : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Stop 1st channel (if required) 
 *      -# Stop 2ndt channel  (if required) 
 *      -# Send transcoding command to supervisor 
 *      -# Start 1st channel (if required) 
 *      -# Start 2ndt channel  (if required) 
 *
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl 1st channel of transcoding session.
 *  \param pstMsg  Message obtained from MSP.
 */

VSTATUS VFSM_TranscodingSession(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	STranscodingParams *TranscodingParams;	/* Params provided by the user */
	STransConnectParams *pstTransConnectParams;/* Params intended for the device command */
	VSTATUS Status;
	SChnl * pstChnl2nd;
	SChnl *pstSupvChnl;
	U16 *pusVoiceOperation;
	U16 FrameSize1, FrameSize2;
	SVapiReq *pstChildVapiReq;
	SVapiReq **ppstLockVapiReq;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_TranscodingSession: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	TranscodingParams = (STranscodingParams *) pstVapiReq->pvUserData;

	/* retrieve the 2nd channel involved in the transcoding session*/
	pstChnl2nd = DMGR_GetConnection(TranscodingParams->Connection2Id);
	/* Make sure the 2nd channel is still there */
	if (pstChnl2nd == NULL)
	{
		UT_ErrorLog(APPITF, "VFSM_TranscodingSession: Connection Id(%u) doesn't exist\n",
			TranscodingParams->Connection2Id);
		Status = VAPI_ERR_INVALID_CONNID;
		goto finish;
	}

	/*allow transcoding on VoIP or FoIP channels only */
	if (((pstChnl->usConnType != eVOIP) && (pstChnl->usConnType != eFOIP)) ||
		((pstChnl2nd->usConnType != eVOIP) && (pstChnl2nd->usConnType != eFOIP)))
	{
		UT_ErrorLog(APPITF, "VFSM_TranscodingSession: transcoding only allowed on VoIP or FoIP channel\n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto finish;
	}

	switch (pstVapiReq->usReqState)
	{

	case TRANSCODING_INIT:

		/* retrieve the packet interval of 2 current channels 
		to check if the connections can be transcoded */
		FrameSize1 = pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.packet_interval;
		FrameSize2 = pstChnl2nd->pstChnlParams->stVoiceOpt.param_4.bits.packet_interval;
	
		/*Virtual Transcoding allows any packet size*/
		if (TranscodingParams->ucVirtualTranscoding != 1)
		{
			if (FrameSize1 >= FrameSize2)
			{
				if (FrameSize1 % FrameSize2)
				{
					UT_ErrorLog(APPITF, "VFSM_TranscodingSession: packet sizes need to be integer multiples of each other\n");
					Status = VAPI_ERR_INVALID_PARAM;
					goto finish;
				}
			}
			else
			{
				if (FrameSize2 % FrameSize1)
				{
					UT_ErrorLog(APPITF, "VFSM_TranscodingSession: packet sizes need to be integer multiples of each other\n");
					Status = VAPI_ERR_INVALID_PARAM;
					goto finish;
				}
			}
		}
		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}
		/* save the pointer of this request in execdata*/
		pstVapiReq->pvExecData = (U8 *) UT_AllocMem(sizeof(SVapiReq *));
		if (!pstVapiReq->pvExecData)
		{
			UT_FreeMem(pstChildVapiReq);
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		ppstLockVapiReq = pstVapiReq->pvExecData;
		*ppstLockVapiReq = pstChildVapiReq;

		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
			pstVapiReq,			/* Parent Request */ 
			VFSM_LockConnection,		/* Child Request Handler */
			LOCK_CONN_INIT);		/* Child Request handler state */

		pstVapiReq->usReqState = TRANSCODING_STOP_CHANNEL_1;
		/*add it to the queue of the current 2nd connection for process */
		Status = VCORE_ProcessLockRequest(pstChnl2nd, pstChildVapiReq);

		break;

	case TRANSCODING_STOP_CHANNEL_1:

		/* retrieve the child status */
		Status = pstVapiReq->Status;
		if (Status != SUCCESS)
			goto finish;

		if (Event != FSM_EVENT_LOCKED)
			UT_ErrorLog(APPITF, "VFSM_TranscodingSession: Something is going wrong in inter Locking\n");

		/* The 2nd channel has now a request posted in its queue
		no other request posted from the user application will be processed before 
		we remove the request.
		We can also use the VCORE_StartChannelChildRequest() for this channel */

		/**********************************************************************************/
		/* FROM HERE WE NEED TO MAKE SURE TO UNLOCK THE SECOND CHANNEL IN CASE OF FAILURE */
		/* ONLY CHILD REQUEST CAN BE POSTED TO THE LOCKED CONNECTION			 */
		/* NO DIRECT CALL TO VDEV_xxx functions						 */
		/**********************************************************************************/

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto unlock_finish;
		}

		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
			pstVapiReq,			/* Parent Request */ 
			VFSM_StopConnection,		/* Child Request Handler */
			STOP_CONN_INIT);		/* Child Request handler state */

		pstVapiReq->usReqState = TRANSCODING_STOP_CHANNEL_2;
		/*add it to the queue of the current connection for process */
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case TRANSCODING_STOP_CHANNEL_2:

		/* retrieve the child status */
		Status = pstVapiReq->Status;
		if (Status != SUCCESS)
			goto unlock_finish;

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto unlock_finish;
		}

		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_StopConnection,	/* Child Request Handler */
				STOP_CONN_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = TRANSCODING_START;
		/*add it to the front of the queue the 2 connection for process */
		VCORE_StartChannelChildRequest(pstChnl2nd, pstChildVapiReq);

		break;

	case TRANSCODING_START:

		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto unlock_finish;
		/* The 2nd channel has been succesfully stopped*/
		/* We now want to send the trans connect command to the supervisor */
		/* process the request on supervisor */
		pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);

		/* this allocate a new request  */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(STransConnectParams));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto unlock_finish;
		}

		pstTransConnectParams = pstChildVapiReq->pvUserData;

		pstTransConnectParams->bTranscodingEnable = TranscodingParams->TranscodingEnable;
		pstTransConnectParams->ucDSPMode = TranscodingParams->ucDSPMode;
		pstTransConnectParams->ucVirtualTranscoding = TranscodingParams->ucVirtualTranscoding;
		pstTransConnectParams->Channel1Id = pstChnl->usMSPChnlId;
		pstTransConnectParams->Channel2Id = pstChnl2nd->usMSPChnlId;

		/*If it is virtual transcoding force the payload type to 0xff*/
		if (TranscodingParams->ucVirtualTranscoding == 1)
		{
			pstTransConnectParams->Channel1PT = 0xff;
			pstTransConnectParams->Channel2PT = 0xff;
		}
		/*else used the current configured payload type*/
		else
		{
			pstTransConnectParams->Channel1PT = pstChnl->pstChnlParams->stVopena.param_5.bits.rtp_pt;
			pstTransConnectParams->Channel2PT = pstChnl2nd->pstChnlParams->stVopena.param_5.bits.rtp_pt;
		}

		/* initialise the Child request */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_SetTransConnect,	/* Child Request Handler */
				SET_TRANSCONNECT_INIT);	/* Child Request handler state */

		/* If the user wants to enable the session, start the RTP traffic*/
		/* Else keep the connections stopped */ 
		if (TranscodingParams->TranscodingEnable)
			pstVapiReq->usReqState = TRANSCODING_START_CHANNEL2;
		else
			pstVapiReq->usReqState = TRANSCODING_FINISHED;

		/*add it to the queue of the supervisor channel for process */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case TRANSCODING_START_CHANNEL2:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto unlock_finish;
		/* The Trans Connect command has been succesfully sent to the supervisor
		we now want to start both channels if transcoding enable*/

		/* this allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(U16));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto unlock_finish;
		}

		/* initialise the Child request */
		pusVoiceOperation = pstChildVapiReq->pvUserData;
		*(pusVoiceOperation) = VOIP_VOPENA_MODE_ENABLE_RTP; 

		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
			pstVapiReq,			/* Parent Request */ 
			VFSM_StartConnectionPacket,	/* Child Request Handler */
			START_CONN_INIT);		/* Child Request handler state */

		pstVapiReq->usReqState = TRANSCODING_START_CHANNEL1;
		/*add it to the front of the queue the 2 connection for process */
		VCORE_StartChannelChildRequest(pstChnl2nd, pstChildVapiReq);

		break;

	case TRANSCODING_START_CHANNEL1:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto unlock_finish;

		/* this allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(U16));

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto unlock_finish;
		}

		/* initialise the Child request */
		pusVoiceOperation = pstChildVapiReq->pvUserData;
		*(pusVoiceOperation) = VOIP_VOPENA_MODE_ENABLE_RTP; 

		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
			pstVapiReq,			/* Parent Request */ 
			VFSM_StartConnectionPacket,	/* Child Request Handler */
			START_CONN_INIT);		/* Child Request handler state */

		pstVapiReq->usReqState = TRANSCODING_FINISHED;
		/*add it to the front of the queue of the current connection for process */
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case TRANSCODING_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto unlock_finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_TranscodingSession: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

unlock_finish:
		/* we have finished with the 2nd channel
		 so remove the locking request from its queue by calling the FSM lock handler directly
		(stored in pstVapiReq->pvExecData) */

		ppstLockVapiReq = pstVapiReq->pvExecData;

		VFSM_LockConnection(pstChnl2nd, *(ppstLockVapiReq), FSM_EVENT_UNLOCK, NULL);

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_TranscodingSession: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_TranscodingSession: Completing req state(%u) status(%d) conn(%u)\n",
				pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_TranscodingSession: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}


/****************************************************************************
 * VFSM_EnableCIDDetection : The function does the following things -
 ***************************************************************************/
/*!
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#  Following MSP messages will be sent\n
 *          -#  FC_VOIP_CNDDETSP
 *          -#  FC_VOIP_CNDDETCTL\n
 *      -#  When request is completed give a calback or signal request
 *          completion semaphore\n
 *
 *  Assumptions:\n
 *      -#  Default Parameter settings are done by APPITF\n.
 *
 *
 *  \return   SUCCESS or Failure code.
 *
 *  \param pstChnl  This will be a channel on which
 *                  MSP commands will be sent.
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */

VSTATUS VFSM_EnableCIDDetection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SCIDDetectionParams *pstUserData = NULL;
	VSTATUS Status;
	U8 ucEnableType;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_EnableCIDDetection: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SCIDDetectionParams *) pstVapiReq->pvUserData;
	ucEnableType = pstUserData->ucCIDDEnableType;

	switch (pstVapiReq->usReqState)
	{
	case CID_DET_ENABLE_INIT:
		pstVapiReq->usReqState = CID_DET_ENABLE_CTRL; 
		Status = VDEV_CIDDetectionSetup(pstChnl, pstUserData);
		if (Status != SUCCESS)
			goto finish;
		break;

	case CID_DET_ENABLE_CTRL:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CNDDETSP);
		if (Status != SUCCESS)
			goto finish;

		switch(ucEnableType)
		{
		case CIDD_ENABLE_ONHK:
			pstVapiReq->usReqState = CID_DET_ENABLE_FINISHED; 
			Status = VDEV_CNDDETCtrlSend(pstChnl, CIDD_ENABLE_ONHK);
			break;

		case CIDD_ENABLE_OFFHK:
			pstVapiReq->usReqState = CID_DET_ENABLE_FINISHED; 
			Status = VDEV_CNDDETCtrlSend(pstChnl, CIDD_ENABLE_OFFHK);
			break;

		case CIDD_ENABLE_BOTH:
			pstVapiReq->usReqState = CID_DET_ENABLE_BOTH; 
			Status = VDEV_CNDDETCtrlSend(pstChnl, CIDD_ENABLE_ONHK);
		}

		if (Status != SUCCESS)
			goto finish;

		break;

	case CID_DET_ENABLE_BOTH:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CNDDETCTL);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = CID_DET_ENABLE_FINISHED; 
		Status = VDEV_CNDDETCtrlSend(pstChnl, CIDD_ENABLE_OFFHK);

		if (Status != SUCCESS)
			goto finish;

		break;

	case CID_DET_ENABLE_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CNDDETCTL);

		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_EnableCIDDetection: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_EnableCIDDetection: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_EnableCIDDetection: Completing state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_EnableCIDDetection: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

VSTATUS VFSM_SetCIDDetection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SCidDetCtrl *pstUserData = NULL;
	VSTATUS Status;
	CONNID ConnId = pstChnl->ConnId;
	struct _CNDDETCTL cid_det_ctrl;

	UT_Log(VCORE, DEBUG, "VFSM_SetCIDDetection: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SCidDetCtrl *) pstVapiReq->pvUserData;

	UT_MemSet(&cid_det_ctrl, 0 , sizeof(struct _CNDDETCTL));

	cid_det_ctrl.param_4.bits.cid_enable = pstUserData->usCidEnable;

	switch (pstVapiReq->usReqState)
	{
	case CID_DET_ENABLE_INIT:
		pstVapiReq->usReqState = CID_DET_ENABLE_CTRL; 
		Status = VDEV_SetCidDetParams(pstChnl);
		if (Status != SUCCESS)
			goto finish;
		break;

	case CID_DET_ENABLE_CTRL:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CNDDETSP);
		if (Status != SUCCESS)
			goto finish;

		switch(pstUserData->usHookStatus)
		{
		case CIDD_ENABLE_ONHK:
			pstVapiReq->usReqState = CID_DET_ENABLE_FINISHED; 
			cid_det_ctrl.param_4.bits.hook_status = CNDDETCTL_HOOK_STATUS_ONHOOK;
			Status = VDEV_SetCidCtrl(pstChnl, &cid_det_ctrl);
			break;

		case CIDD_ENABLE_OFFHK:
			pstVapiReq->usReqState = CID_DET_ENABLE_FINISHED; 
			cid_det_ctrl.param_4.bits.hook_status = CNDDETCTL_HOOK_STATUS_OFFHOOK;
			Status = VDEV_SetCidCtrl(pstChnl, &cid_det_ctrl);
			break;

		/*In case on both detection of Onhook and offhook CID enable first onhook*/
		case CIDD_ENABLE_BOTH:
			pstVapiReq->usReqState = CID_DET_ENABLE_BOTH; 
			cid_det_ctrl.param_4.bits.hook_status = CNDDETCTL_HOOK_STATUS_ONHOOK;
			Status = VDEV_SetCidCtrl(pstChnl, &cid_det_ctrl);
		}

		if (Status != SUCCESS)
			goto finish;

		break;

	case CID_DET_ENABLE_BOTH:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CNDDETCTL);

		if (Status != SUCCESS)
			goto finish;

		/*In case on both detection now enable offhook*/
		pstVapiReq->usReqState = CID_DET_ENABLE_FINISHED; 
		cid_det_ctrl.param_4.bits.hook_status = CNDDETCTL_HOOK_STATUS_OFFHOOK;
		Status = VDEV_SetCidCtrl(pstChnl, &cid_det_ctrl);

		if (Status != SUCCESS)
			goto finish;

		break;

	case CID_DET_ENABLE_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_CNDDETCTL);

		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetCIDDetection: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_SetCIDDetection: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_SetCIDDetection: Completing state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetCIDDetection: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_StopConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  If the VOICE is enable send VOPENA DISABLE
 *      -#  else nothing
 *
 *  \b Inputs-Outputs: \n
 *	\li pstChnl The channel on which the request is going.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_StopConnection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	struct _VOIP_VOPENA *pstVopena = NULL;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_StopConnection: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case STOP_CONN_INIT:

		/* Send the command only if the channel enabled */
		if (pstChnl->bIsActive)
		{
			pstVapiReq->usReqState = STOP_CONN_FINISHED;
			Status = VDEV_VopDisable(pstChnl);
			if (Status != SUCCESS)
				goto finish;
		}
		else
		{
			/* if the channel is already disabled do nothing */
			Status = SUCCESS;
			goto finish;
		}

		break;

	case STOP_CONN_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);

		/* If we are trying to disable a channel, but get no response.
		May be the channel does not exist anymore
		So let continue with the regular process to remove the VAPI conenction structure*/ 
		if (Status == VAPI_ERR_MSP_NO_RESPONSE)
		{
			UT_ErrorLog(VCORE, "VFSM_StopConnection: Device channel doesn't (%u) exist continue\n",pstChnl->usMSPChnlId);
			/* If a child request report SUCCESS to the parent*/
			Status = SUCCESS;
		}

		/*If success update the VAPI variable */
		if (Status == SUCCESS)
		{

			if ((pstChnl->usConnType == eVOIP) || (pstChnl->usConnType == eFOIP)) 
			{
				pstVopena = &(pstChnl->pstChnlParams->stVopena);
				pstVopena->mode = VOIP_VOPENA_MODE_DISABLE;
			}
			pstChnl->bIsActive = False;
		}

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_StopConnection: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_StopConnection: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_StopConnection: Completing state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_StopConnection: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;

}

/****************************************************************************
 * VFSM_StartConnectionPacket : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -# Reset the packetisation bit (VCEOPT)
 *
 *  \b Inputs-Outputs: \n
 *	\li pstChnl The channel on which the request is going.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_StartConnectionPacket(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	struct _VOIP_VCEOPT *pstVoiceParams;
	struct _VOIP_VOPENA *pstVopena;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_StartConnectionPacket: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);
	/*FIXME need to pass the VOPENA operation mode to allow VoIP-Without-RTP or FoIPoRTP*/


	switch(pstChnl->usConnType)
	{
	case eVOIP:
	case eFOIP:
	case eIMSOTDM:
		pstVopena = &(pstChnl->pstChnlParams->stVopena);
		pstVoiceParams = &(pstChnl->pstChnlParams->stVoiceOpt);
		break;

	default:
		UT_Log(VCORE, DEBUG, "VFSM_StartConnectionPacket: Channel Type %d not supported yet \n", 
				pstChnl->usConnType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto finish;
	}

	/*FIXME we need to enable packet reception too */
	/*enable packet transmission for eRSP on VoIP channel*/
	if (pstChnl->ePartType == eLSP)
		pstVoiceParams->param_4.bits.packet_generation = VOIP_VCEOPT_PACKET_GENERATION_DISABLE;
	else
		pstVoiceParams->param_4.bits.packet_generation = VOIP_VCEOPT_PACKET_GENERATION_ENABLE;

	switch (pstVapiReq->usReqState)
	{
	case START_CONN_INIT:

		/* Check if the device ethernet header has been correctly set */
		switch (pstChnl->pstDev->eEthHdrMode)
		{
		case eMODE_NONE:
				UT_ErrorLog(VCORE, "VFSM_StartConnectionPacket: IP or ALT Mode not defined\n");
			if (pstChnl->ePartType != eLSP)
			{
				Status = VAPI_ERR_ETH_HDR_NOT_SET;
				goto finish;
			break;
			}

		case eMODE_IP:
			/* If the channel is not a Local participant and the IP header is not set--> error
			(Local participant doesn't requires a IP header to be activated */
			if ((!pstChnl->bIPHdrSet) && (pstChnl->ePartType != eLSP))
			{
				UT_ErrorLog(VCORE, "VFSM_StartConnectionPacket: IP Header is not set for IP packets\n");
				Status = VAPI_ERR_IP_HDR_NOT_SET;
				goto finish;
			}
			break;

		case eMODE_ALT:
			/* if the device is set in alternate header mode check if the ALT_HDR_CHANNEL has been issued */
			if ((!pstChnl->bAltHdrSet) && (pstChnl->ePartType != eLSP))
			{
				UT_ErrorLog(VCORE, "VFSM_StartConnectionPacket: ALT Header is not set for packets\n");
				Status = VAPI_ERR_ALT_HDR_NOT_SET;
				goto finish;
			}
			break;

		default:
			UT_ErrorLog(VCORE, "VFSM_StartConnectionPacket: Undefined device eth header mode\n");
			Status = VAPI_ERR_ETH_HDR_NOT_SET;
			goto finish;
		}

		pstVapiReq->usReqState = START_CONN_WAIT_VCEOPT_ACK;

		Status = VDEV_SendVceopt(pstChnl, pstVoiceParams);
		if (Status != SUCCESS)
			goto finish;

		break;

	case START_CONN_WAIT_VCEOPT_ACK:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VCEOPT);

		if (Status != SUCCESS)
			goto finish;

		/* if the channel is already enable stop here */
		if (pstChnl->bIsActive)
			goto finish;

		switch(pstChnl->usConnType)
		{
		case eVOIP:
			if (pstChnl->ePartType == eLSP)
				pstVopena->mode = VOIP_VOPENA_MODE_ENABLE_LSP;	
			else
				pstVopena->mode = VOIP_VOPENA_MODE_ENABLE_RTP;	
			break;
	
		case eFOIP:
			/* if is a fax channel, only enable it if we want to generate packets */
			if (pstVoiceParams->param_4.bits.packet_generation == VOIP_VCEOPT_PACKET_GENERATION_DISABLE)	
				goto finish;

			pstVopena->mode = VOIP_VOPENA_MODE_ENABLE_UDPTL;	
			break;

		case eIMSOTDM:
			pstVopena->mode = VOIP_VOPENA_MODE_ENABLE_IP_TDM;	

			break;			
			
		}

		pstVapiReq->usReqState = START_CONN_FINISHED;
		Status = VDEV_SendVoipVopena(pstChnl, pstVopena);
		if (Status != SUCCESS)
			goto finish;

		break;

	case START_CONN_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);

		if (Status == SUCCESS)
			pstChnl->bIsActive = True;

		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_StartConnectionPacket: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_StartConnectionPacket: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_StartConnectionPacket: Completing state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_StartConnectionPacket: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;

}

/****************************************************************************
 * VFSM_StartConnectionTDM : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -# Set the packetisation bit (VCEOPT)
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData contains the Enable Type
 *	\li pstChnl The channel on which the request is going.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_StartConnectionTDM(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	struct _VOIP_VCEOPT *pstVoiceParams;
	struct _VOIP_VOPENA *pstVopena;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_StartConnectionTDM: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	switch(pstChnl->usConnType)
	{
	case eVOIP:
	case eFOIP:
	case eIMSOTDM:
		pstVopena = &(pstChnl->pstChnlParams->stVopena);
		pstVoiceParams = &(pstChnl->pstChnlParams->stVoiceOpt);
		break;

	default:
		UT_Log(VCORE, DEBUG, "VFSM_StartConnectionTDM: Channel Type %d not supported yet \n", pstChnl->usConnType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto finish;
	}

	/*disable packet transmission */
	/*FIXME we need to enable packet reception too */
	pstVoiceParams->param_4.bits.packet_generation = VOIP_VCEOPT_PACKET_GENERATION_DISABLE;

	switch (pstVapiReq->usReqState)
	{
	case START_CONN_INIT:

		pstVapiReq->usReqState = START_CONN_WAIT_VCEOPT_ACK;

		Status = VDEV_SendVceopt(pstChnl, pstVoiceParams);
		if (Status != SUCCESS)
			goto finish;

		break;

	case START_CONN_WAIT_VCEOPT_ACK:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VCEOPT);

		if (Status != SUCCESS)
			goto finish;

		/* if the channel is already enable stop here */
		if (pstChnl->bIsActive)
			goto finish;

		switch(pstChnl->usConnType)
		{
		case eVOIP:
			if (pstChnl->ePartType == eLSP)
				pstVopena->mode = VOIP_VOPENA_MODE_ENABLE_LSP;	
			else
				pstVopena->mode = VOIP_VOPENA_MODE_ENABLE_RTP;	
			break;
	
		case eFOIP:
			/* if is a fax channel only enable it if we want to generate packets */
			if (pstVoiceParams->param_4.bits.packet_generation == VOIP_VCEOPT_PACKET_GENERATION_DISABLE)	
				goto finish;

			pstVopena->mode = VOIP_VOPENA_MODE_ENABLE_UDPTL;

		case eIMSOTDM:
			pstVopena->mode = VOIP_VOPENA_MODE_ENABLE_IP_TDM;

			break;
		}

		pstVapiReq->usReqState = START_CONN_FINISHED;
		Status = VDEV_SendVoipVopena(pstChnl, pstVopena);
		if (Status != SUCCESS)
			goto finish;

		break;

	case START_CONN_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);

		if (Status == SUCCESS)
			pstChnl->bIsActive = True;

		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_StartConnectionTDM: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_StartConnectionTDM: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);


	UT_Log(VCORE, DEBUG, "VFSM_StartConnectionTDM: Completing state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_StartConnectionTDM: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;

}

VSTATUS VFSM_SetConnEthMac(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SVapiReq * pstChildVapiReq;
	SSetEthMAc *pstUserData = NULL;
	struct _VOIP_VOPENA *pstVopena;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_SetConnEthMac: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	switch(pstChnl->usConnType)
	{
	case eVOIP:
	case eFOIP:
		pstVopena = &(pstChnl->pstChnlParams->stVopena);
		break;

	default:
		UT_Log(VCORE, DEBUG, "VFSM_SetConnEthMac: Channel Type %d not supported yet \n", 
				pstChnl->usConnType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto finish;
	}

	switch (pstVapiReq->usReqState)
	{
		case CONN_ETH_MAC_INIT:
		/* We want to stop the channel first */
		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* save the current value of vopena */
		pstVapiReq->pvExecData = (U8 *) UT_AllocMem(2*sizeof(U8));
		if (!pstVapiReq->pvExecData)
		{
			UT_FreeMem(pstChildVapiReq);
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		((U8 *) pstVapiReq->pvExecData)[0] = pstChnl->bIsActive;
		((U8 *) pstVapiReq->pvExecData)[1] = pstVopena->mode;
		
		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_StopConnection,	/* Child Request Handler */
				STOP_CONN_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = CONN_ETH_MAC_STOP;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case CONN_ETH_MAC_STOP:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		/*  retrieve the original state of the vopena*/
		pstVopena->mode = ((U8 *) pstVapiReq->pvExecData)[1];

		pstUserData = (SSetEthMAc *) pstVapiReq->pvUserData;

		pstVapiReq->usReqState = CONN_ETH_MAC_CHECK;
		Status = VDEV_SendEthMacAddr(pstChnl, pstUserData->astSrcMacAddr, pstUserData->astDestMacAddr,
							pstChnl->stEthMacAddresses.usVlanId);

		if (Status != SUCCESS)
			goto finish;

		break;

	case CONN_ETH_MAC_CHECK:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_SET_ETH_HDR_CHAN);

		if (Status != SUCCESS)
			goto finish;

		/* save the Src, Dest MAC addresses and VLAN id to connection structure*/
		( (SSetEthMAc *) pstVapiReq->pvUserData)->usVlanId  = pstChnl->stEthMacAddresses.usVlanId;
		UT_MemCopy( &(pstChnl->stEthMacAddresses), (SSetEthMAc *) pstVapiReq->pvUserData, sizeof(SSetEthMAc));

		/* check if we need to renable the channel */
		if (((U8 *) pstVapiReq->pvExecData)[0]== True)
		{
			pstVapiReq->usReqState = CONN_ETH_MAC_FINISHED;
			Status = VDEV_SendVoipVopena(pstChnl, pstVopena);
			if (Status != SUCCESS)
				goto finish;
		}
		else
			goto finish;

		break;

	case CONN_ETH_MAC_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);

		if (Status == SUCCESS)
			pstChnl->bIsActive = True;

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_ConnSetEthMac: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	};

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_ConnSetEthMac: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_ConnSetEthMac: Completing req state(%u) status(%d) conn(%u)\n",
					pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_ConnSetEthMac: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_LockConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  place or remove a request in the connection queue depending on the Event
 *
 *  \b Inputs-Outputs: \n
 *	\li pstChnl The channel on which the request is going.
 *	\li pstVapiReq The processed request.
 *	\li Event EVENT_LOCk, EVENT_UNLOCK
 *	\li pstMsg NULL.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *
 */
VSTATUS VFSM_LockConnection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_LockConnection: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	if (Event == FSM_EVENT_UNLOCK)
	{
		pstVapiReq->usReqState = LOCK_CONN_FINISHED;
	}

	switch (pstVapiReq->usReqState)
	{
	case LOCK_CONN_INIT:
		pstVapiReq->pstParentReq->Status = Status;

		/* the parent (if any ) heritates the context of the child request */
		pstVapiReq->pstParentReq->ucThreadContext = pstVapiReq->ucThreadContext;

		pstVapiReq->usReqState = LOCK_CONN_WAIT;
		UT_Log(VCORE, DEBUG, "VFSM_LockConnection: locking req(0x%x) state(%u) status(%d) conn(%u)\n",
			pstVapiReq, pstVapiReq->usReqState, Status, ConnId);

		pstVapiReq->pstParentReq->pfnReqFSMHdlr(pstVapiReq->pstParentReq->pstChnl, pstVapiReq->pstParentReq, FSM_EVENT_LOCKED, NULL);

		break;

	case LOCK_CONN_WAIT:
		break;


	case LOCK_CONN_FINISHED:
		UT_Log(VCORE, DEBUG, "VFSM_LockConnection: Unlocking req(0x%x) state(%u) status(%d) conn(%u)\n",
			pstVapiReq, pstVapiReq->usReqState, Status, ConnId);
		/*Remove the request node from this list */
		UT_RemoveNode(&pstChnl->stReqList, &pstVapiReq->stNode);
		UT_FreeMem(pstVapiReq);
		/*Initiate Pending request on this channel */
		VCORE_ProcessRequestList(pstChnl);
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_LockConnection: wrong state %u\n", pstVapiReq->usReqState);

	}

	return Status;
}

/****************************************************************************
 * VFSM_EnableTHC : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  If send Enqble THC to the chqnnel
 *      -#  wait for the reponse
 *
 *  \b Inputs-Outputs: \n
 *	\li pstChnl The channel on which the request is going.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *
 */
VSTATUS VFSM_EnableTHC(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_EnableTHC: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	switch (pstVapiReq->usReqState)
	{
	case ENABLE_THC_INIT:

		pstVapiReq->usReqState = ENABLE_THC_FINISHED;
		Status = VDEV_ThcModeEnable(pstChnl);
		if (Status != SUCCESS)
			goto finish;
		break;

	case ENABLE_THC_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_THC_MODE_ENABLE);

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_EnableTHC: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_EnableTHC: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_EnableTHC: Completing state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_EnableTHC: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;

}

/****************************************************************************
 * VFSM_InterChannelLoopback : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *
 *  \b Inputs-Outputs: \n
 *	\li pstChnl The channel on which the request is going.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *
 */
VSTATUS VFSM_InterChannelLoopback(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SLpbackUserData *pstLoopbackUserData;
	VSTATUS Status;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_InterChannelLoopback: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstLoopbackUserData = (SLpbackUserData *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case INTERCHANNEL_INIT:


		pstVapiReq->usReqState = INTERCHANNEL_FINISHED;
		Status = VDEV_VoIPLoopback(pstChnl, pstLoopbackUserData->ConnId2, pstLoopbackUserData->eLoopback);
		if (Status != SUCCESS)
			goto finish;
		break;

	case INTERCHANNEL_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_LOOPBACK);

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_InterChannelLoopback: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;
	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_InterChannelLoopback: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_InterChannelLoopback: Completing state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_InterChannelLoopback: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;
}

/****************************************************************************
 * VFSM_ModConn
 ***************************************************************************/
VSTATUS VFSM_ModConn(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	gtl_msg_t *pstReqMsg;
	VSTATUS Status;
	SFullModConnInfo *pstUserData;
	gtl_msg_t *pstMsgMulti;
	U16 *pvComm;
	U16 usNumComm;
	SApiHdr *pstFifo;
	CONNID ConnId = pstChnl->ConnId;
	int tempSize;
	void *tempFifo;
	int i;

	UT_Log(VCORE, DEBUG, "VFSM_ModConn: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SFullModConnInfo *) pstVapiReq->pvUserData;
	pstReqMsg = (gtl_msg_t *) pstVapiReq->pvExecData;

	switch (pstVapiReq->usReqState)
	{
	case MOD_CONN_INIT:
		usNumComm = pstUserData->usModifyNum;
		pvComm = UT_Calloc(usNumComm, sizeof (U16));

		for (i = 0; i < usNumComm; i++)
			pvComm[i] = pstUserData->pstModConnInfo[i].usFunctionCode;

		pstMsgMulti = UT_Calloc(1, sizeof(gtl_msg_t));
		if (pstMsgMulti == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		pstMsgMulti->fifo = UT_Calloc(1, pstChnl->pstDev->MaxMsgSize);	/* alloc the max fifo length   */
		if (pstMsgMulti->fifo == NULL)
		{
			UT_FreeMem(pstMsgMulti);
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* init msg fields to defautl values*/
		VCORE_InitMsg(pstMsgMulti, pstMsgMulti->fifo, pstChnl->pstDev->MaxMsgSize);		
		
		Status = VDEV_SetQueryCommand(pstMsgMulti, pvComm, usNumComm);

		UT_FreeMem(pvComm);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = MOD_CONN_SEND_COMM;

		pstVapiReq->pvExecData = pstMsgMulti;	/* Save the multilist structure */
		Status = VCORE_SendMSPReq(pstChnl, pstMsgMulti);
		if (Status != SUCCESS)
			goto finish;

		break;
	
	case MOD_CONN_SEND_COMM:
		Status = VCORE_CheckStatusMulti(pstChnl, (gtl_msg_t *) (pstVapiReq->pvExecData), Event, pstMsg);
		UT_FreeMem(((gtl_msg_t *) (pstVapiReq->pvExecData))->fifo);
		UT_FreeMem(pstVapiReq->pvExecData);
		pstVapiReq->pvExecData = NULL;

		if (Status != SUCCESS)
			goto finish;

		/*allocate memory for the request and its UserData. If error get out */
		pstReqMsg = UT_Calloc(1, sizeof(gtl_msg_t));
		if (pstReqMsg == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* allocate buffer to handle the command parameters */
		pstReqMsg->fifo = UT_Calloc(1, pstChnl->pstDev->MaxMsgSize);
		if (pstReqMsg->fifo == NULL)
		{
			UT_FreeMem(pstReqMsg);
			UT_ErrorLog(APPITF, "No memory available\n");
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* fill the fifo with the user data */
		UT_MemCopy(pstReqMsg->fifo, pstMsg->fifo, pstMsg->fifo_size);
		pstReqMsg->fifo_size = pstMsg->fifo_size;

		((SApiHdr *) pstReqMsg->fifo)->cmd_class = CMD_CLASS_CONF_CHANNEL;
		((SApiHdr *) pstReqMsg->fifo)->cmd_type = CMD_TYPE_CONF_CHANGE;		

		/*Save the user message before checking if the message is valid (fifo and fifo len are manipulated)*/
		tempSize = pstReqMsg->fifo_size;
		tempFifo = (void *) pstReqMsg->fifo;

		Status = VCORE_ModifyQueriedCommand(pstReqMsg, pstUserData->pstModConnInfo, pstUserData->usModifyNum);

		if (Status != SUCCESS)
		{
			UT_FreeMem(pstReqMsg->fifo);
			UT_FreeMem(pstReqMsg);
			Status = VAPI_ERR_INVALID_PARAM;
			goto finish;
		}

		pstReqMsg->fifo_size = tempSize;
		pstReqMsg->fifo = tempFifo;

		Status = VCORE_CheckMessage(pstChnl->pstDev, (SMsg *) pstReqMsg, CMD_LEVEL_CONN);
		if (Status != SUCCESS)
		{
			UT_FreeMem(pstReqMsg->fifo);
			UT_FreeMem(pstReqMsg);
			Status = VAPI_ERR_INVALID_PARAM;
			goto finish;
		}

		pstReqMsg->fifo_size = tempSize;
		pstReqMsg->fifo = tempFifo;

		/*Send the GTL message */
		pstFifo = (SApiHdr *)pstReqMsg->fifo;
		pstReqMsg->channel = pstChnl->usMSPChnlId;

		pstVapiReq->usReqState = MOD_CONN_FINISHED;

		pstVapiReq->pvExecData = pstReqMsg;
		Status = VCORE_SendMSPReq(pstChnl, pstReqMsg);
		if (Status != SUCCESS)
			goto finish;

		break;

	case MOD_CONN_FINISHED:
		Status = VCORE_CheckPassThruStatus(pstChnl, pstReqMsg, Event, pstMsg);
		UT_FreeMem(((gtl_msg_t *) (pstVapiReq->pvExecData))->fifo);
		UT_FreeMem(pstVapiReq->pvExecData);
		pstVapiReq->pvExecData = NULL;

		/* in case of unexpected error pstMsg = NULL,
		nothing reported to the application */
		if ((Status == VAPI_ERR_MSP_NO_RESPONSE) ||
			(Status == VAPI_IND_STOP_REC_PLAY) ||
			(Status == VAPI_ERR_UNDEFINED_EVENT))
			goto finish;

		pstVapiReq->pvCallbackData = UT_AllocMem(pstMsg->fifo_size);

		if (!pstVapiReq->pvCallbackData)
		{
			Status = VAPI_ERR_NOMEM;
				goto finish;
		}

		UT_MemCopy(pstVapiReq->pvCallbackData, pstMsg->fifo, pstMsg->fifo_size);
		pstVapiReq->uiCallbkDataLen = pstMsg->fifo_size;
	
		goto finish;
		break;

	}

	return SUCCESS;

finish:
	/*Do request completion processing */
	UT_Log(VCORE, DEBUG, "VFSM_ModConn: Completing state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	if(Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_ModConn: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	/*If non blocking call, then clean up the user-data 
	in SYNC mode it is freed in VAPI_ModifyConnection*/
	if (!pstVapiReq->bIsSync)
	{
		UT_FreeMem(((SFullModConnInfo *) pstVapiReq->pvUserData)->pstModConnInfo);
	}

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	/*Do request completion processing */
	UT_Log(VCORE, DEBUG, "VFSM_ModConn: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_RecoverConnection : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *          -#  SUPVSR_CREATE_CHANNEL QUERY FIXME\n
 *      -#  When the channel is retrieved the id is set the the device chaannel array
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings are done by APPITF\n.
 *
 *
 *  \return   None
 *
 *  \param pstChnl  Firstly this will be a supervisory channel on which 
 *                  calls to this processing function will have newly
 *                  created channel as this parameter.
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */
VSTATUS VFSM_RecoverConnection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SChnl *pstSupvChnl;
	SVapiReq *pstChildVapiReq;
	VSTATUS RecoverStatus = SUCCESS;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_RecoverConnection: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case CREATE_CONNECTION_INIT:

		pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);

		/* Check if the multi cmd mode is enabled (only for CSME and POS); if not get out */
		if ((pstChnl->pstDev->eItfType != ePCI_ITF) && (!pstChnl->pstDev->bMultiCmdEnabled))
		{
			Status = VAPI_ERR_MULTI_CMD_NOT_INITIALIZED;
			goto finish;
		}

		/* This allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SChnl *));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* User Data for child request is comming from the parent request */
		UT_MemCopy(pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, sizeof(SChnl *));

		pstVapiReq->usReqState = CREATE_CONNECTION_FINISHED;
		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_recoverChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VFSM_RecoverChannel,	/* Child Request Handler */
				CREATE_CHANNEL_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case CREATE_CONNECTION_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
                RecoverStatus = Status;

                /* if the channel has been recovered, but with the wrong type
                make the connection valid to be able to at least destroy it */
                if ((Status != SUCCESS) && (Status != VAPI_ERR_RECOVER_WRONG_TYPE))
                        goto finish;

		/* The device channel now exists.
		add the connection structure to the AVL tree */
		Status = VCORE_AddConnection(pstChnl);
		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}
		/* from here new requests can be posted to the connection */

		UT_Log(VCORE, DEBUG, "VFSM_RecoverConnection: Device channel(%u) created on dev(%u) conn(%u)\n",
			pstChnl->usMSPChnlId, pstChnl->pstDev->DevId, ConnId);

		 /* save the Src and Dest MAC addresses of the device as the default for channel*/
		UT_MemCopy( &(pstChnl->stEthMacAddresses), &(pstChnl->pstDev->stEthMacAddresses), sizeof(SSetEthMAc));

		/* in case of VAPI_ERR_RECOVER_WRONG_TYPE the connection is valid
		but we report to indicate the error to the application*/
		Status = RecoverStatus;

		goto finish;	
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_RecoverConnection: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;
	};

	return SUCCESS;
 
finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_RecoverConnection: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_RecoverConnection: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);
 
	UT_Log(VCORE, INFO, "VFSM_RecoverConnection: Exiting status(%d) conn(%u)\n", Status, ConnId);
	return Status;
}


/****************************************************************************
 * VFSM_ChangeConnection
 ***************************************************************************/
VSTATUS VFSM_ChangeConnection(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SVapiReq *pstChildVapiReq;
	VSTATUS Status = SUCCESS;
	SFullChangeConnInfo *pstFullChangeConnInfo;
	SChnl *pstSupvChnl;
	SConference *pstConf;
	SChnl **ppstUserData;
	int i;

	UT_Log(VCORE, DEBUG, "VFSM_ChangeConnection: Entering conn(%u) state(%u)\n", pstChnl->ConnId, pstVapiReq->usReqState);

	pstFullChangeConnInfo = (SFullChangeConnInfo *) pstVapiReq->pvUserData;

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case CHANGE_CONN_TYPE_INIT:
		/*allocate memory for the request and its UserData (if any). If error get out */ 
		pstChildVapiReq = VCORE_AllocateRequest(0);

		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* wait the request child to be finished in this state*/
		if (pstFullChangeConnInfo->stChangeConnInfo.ePartType == eNOTPART)
			pstVapiReq->usReqState = CHANGE_CONN_TYPE_CREATE_CHANNEL;
		else
			pstVapiReq->usReqState = CHANGE_CONN_TYPE_CREATE_PART;

		/* initialise the Child request
		The pstChnl pointer is the data are required for the child fsm_handler */
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */
				VFSM_DestroyOldConnection,	/* Child Request Handler */
				DESTROY_CONN_INIT);		/* Child Request handler state */

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case CHANGE_CONN_TYPE_CREATE_CHANNEL:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		/*allocate memory for the request and its UserData. If error get out */
		pstChildVapiReq = VCORE_AllocateRequest(sizeof(SChnl *));
		if (pstChildVapiReq == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* initialise the connection structure */
		pstChnl->ucMsgIndex = 1;
		pstChnl->bIPHdrSet = False;
		pstChnl->bIsActive = False;
		pstChnl->usConnType = pstFullChangeConnInfo->stChangeConnInfo.eConnType;
		pstChnl->usMode = pstFullChangeConnInfo->stChangeConnInfo.eConnMode;
		pstChnl->ePartType = eNOTPART;

		/* Init default internal variables to match the MSP default values 
		in case of VoIP or FoIP channel type*/
		if (pstChnl->usConnType == eVOIP || pstChnl->usConnType == eFOIP)
		{
			DMGR_SetVoIPDevDefaults(pstChnl->pstChnlParams);
		}

		/*check for new params, else reuse old ones*/
		if (pstFullChangeConnInfo->pusParams != NULL)
		{
			pstChnl->usTimeSlot = pstFullChangeConnInfo->pusParams[1];

			if (pstChnl->usMode == eWideBand)
			{
				/*limit the number of TS to 4 (in case of user mistake)*/
				if (pstFullChangeConnInfo->pusParams[0] > MAX_WB_TS)
					pstFullChangeConnInfo->pusParams[0] = MAX_WB_TS;

				for (i = 0; i < pstFullChangeConnInfo->pusParams[0] - 1; i ++)
					pstChnl->usExtTimeSlot[i] = pstFullChangeConnInfo->pusParams[i + 2];
			}
		}
		
		/* pass the connection structure pointer to the request*/
		ppstUserData = pstChildVapiReq->pvUserData;
		*ppstUserData = pstChnl;

		pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);

		if (pstFullChangeConnInfo->stChangeConnInfo.eChangeOpMode == eOpModeNone)
			pstVapiReq->usReqState = CHANGE_CONN_TYPE_FINISHED;
		else
			pstVapiReq->usReqState = CHANGE_CONN_TYPE_SET_OP_MODE;

		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_CreateChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VFSM_CreateChannel,	/* Child Request Handler */
				CREATE_CHANNEL_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case CHANGE_CONN_TYPE_CREATE_PART:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;
		
		Status = VCORE_ConferenceGenericCheck(pstFullChangeConnInfo->pusParams[0], "APPITF_CreateParticipant", &pstConf);

		if (Status != SUCCESS)
			goto finish;

		if ((pstFullChangeConnInfo->stChangeConnInfo.ePartType == eTRANSP) && (pstConf->bIsTranscoding == False))
		{
			UT_ErrorLog(APPITF, "VFSM_ChangeConnection: Transcoding not enabled for conference(%u) \n", 
						pstFullChangeConnInfo->pusParams[0]);
			Status = VAPI_ERR_INVALID_PARAM;
			goto finish;
		}

		/*allocate memory for the request and its UserData. If error get out */
		pstChildVapiReq = VCORE_AllocateRequest(sizeof(SChnl *));
		if (pstChildVapiReq == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		pstChnl->pstConf = pstConf;
		pstChnl->ePartType = pstFullChangeConnInfo->stChangeConnInfo.ePartType;
		pstChnl->usMode = pstFullChangeConnInfo->stChangeConnInfo.eConnMode;
		pstChnl->usConnType = eVOIP;	/* This field is only applicable for LSP participant */
		pstChnl->usTimeSlot = pstFullChangeConnInfo->pusParams[2];
		pstChnl->ucMsgIndex = 1;
		pstChnl->bIPHdrSet = False;
		pstChnl->bIsActive = False;

		if (pstChnl->usMode == eWideBand)
		{
			/*limit the number of TS to 4 (in case of user mistake)*/
			if (pstFullChangeConnInfo->pusParams[1] > MAX_WB_TS)
				pstFullChangeConnInfo->pusParams[1] = MAX_WB_TS;

			for (i = 0; i < pstFullChangeConnInfo->pusParams[1] - 1; i ++)
				pstChnl->usExtTimeSlot[i] = pstFullChangeConnInfo->pusParams[i + 3];
		}		

		/* pass the connection structure pointer to the request*/
		ppstUserData = pstChildVapiReq->pvUserData;
		*ppstUserData = pstChnl;

		if (pstFullChangeConnInfo->stChangeConnInfo.eChangeOpMode == eOpModeNone)
			pstVapiReq->usReqState = CHANGE_CONN_TYPE_FINISHED;
		else
			pstVapiReq->usReqState = CHANGE_CONN_TYPE_SET_OP_MODE;

		/* initialise the Child request */ 
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */
				VFSM_AllocateParticipant,	/* Child Request Handler */
				ALLOCATE_CONF_PARTICIPANT_INIT);/* Child Request handler state */

		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case CHANGE_CONN_TYPE_SET_OP_MODE:
		/* retrieve the child status */
		Status = pstVapiReq->Status;
		if (Status != SUCCESS)
			goto finish;

		/* this allocate a new request (no User Data required) */
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(U8));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		*((U8 *) pstChildVapiReq->pvUserData) = pstFullChangeConnInfo->stChangeConnInfo.eChangeOpMode;

		/* initialise the Child request */
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */
				VFSM_SetNetworkParams,		/* Child Request Handler */
				INIT_CHANGED_CONN_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = CHANGE_CONN_TYPE_FINISHED;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case CHANGE_CONN_TYPE_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		goto finish;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_ChangeConnection: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;

	}

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_ChangeConnection: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, pstChnl->ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_ChangeConnection: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, pstChnl->ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_ChangeConnection: Exiting status(%d) conn(%u)\n", Status, pstChnl->ConnId);

	return Status;
}

