/*! \file conference_itf.c 
 @defgroup conference_level  Conference related API functions
 *  @{
 */

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "cfg.h"
#include "msp.h"


/***************************************************************************
 * VAPI_CreateConference: The function does the following things -
 ***************************************************************************/
/*!
* 	\n \b Description: \n
*	Creates a New conference with a user specified conference ID.\n
*	The VAPI library will map future references of conference ID to the correct device ID \n 
*	to pass to the GTL layer. \n 
*	Commands to enable enhanced conferencing are sent if bIsEnhancedConf is set to True.\n
*
*	\n \b Note:
*	<b>The VAPI_CreateConference() is divided in 2 main steps (if enhanced conferencing is enabled):</b>\n
*	- <b>The device conference creation (CONF_CREATE_CONFERENCE Comcerto command).</b>\n 
*	- <b>The device conference initialization (CONF_SPECIFY_DOMINANT_TALKERS Comcerto commands).</b>\n
*	<b>If an error occurs during the step 1 the conference doesn't exist at all 
*	and the returned error is VAPI_ERR_CREATECONF_FAIL.\n
*	If an error occurs during the step 2 the conference exists but not initialized as expected.\n
*	In this case the user has the opportunity to send some configuration commands or Destroy the conference
*	VAPI_DestroyConference().</b> \n
*	\n 
*
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Device identifier that uniquely identifies a device.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConfId</td>
*		<td style="vertical-align: top;">Conference ID is specified by the user application
*		that uniquely identifies the conference. \n
*		Conference ID may overlap with the values used for connection ID \n
*		(participant IDs may not overlap with connection IDs)</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstConfParams</td>
*		<td style="vertical-align: top;">To Enable/Disable Transcoding and DTMF Tone block feature.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">bIsEnhancedConf</td>
*		<td style="vertical-align: top;">Whether to use enhanced conferencing for this conference.\n
*		<b>Note: This is an optional feature of the Comcerto firmware </b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstConfDTSParams</td>
*		<td style="vertical-align: top;">Enhanced conferencing parameters.
*			Valid only if bIsEnhancedConf is True.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONFID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include conference.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li CONF_SPECIFY_DOMINANT_TALKERS
*	\li CONF_CREATE_CONFERENCE
*/
VSTATUS VAPI_CreateConference(IN DEVID DevId,
			      IN CONFID ConfId,
			      IN SConfParams * pstConfParams,
			      IN Boolean bIsEnhancedConf,
			      IN SConfDTSParams * pstConfDTSParams, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SConference *pstConf;
	SConfDTSParams *pstDTSParams;
	VSTATUS Status;
	Boolean bIsSync;
	SConference **ppstUserData;

	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_CreateConference", &pstDevice);
	if (Status != SUCCESS)
		goto err_out;

	UT_Log(APPITF, INFO, "Entered VAPI_CreateConference: conf(%u)\n", ConfId);

	if (pstConfParams == NULL)
	{
		UT_ErrorLog(APPITF, "Conference parameters not provided\n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	if (NULL != DMGR_GetConfById(ConfId))
	{
		UT_ErrorLog(APPITF, "Conference already exists.\n");
		Status = VAPI_ERR_INVALID_CONFID;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SConference *));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	/* Create the conference structure  
	This structure is freed by DestroyConference or by the VFSM_CreateConference
	in case the device create conference fails */
	pstConf = UT_Calloc(1, sizeof(SConference));
	if (pstConf == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_request;
	}

	ppstUserData = pstVapiReq->pvUserData;
	*ppstUserData = pstConf;

	if (bIsEnhancedConf && pstConfDTSParams != NULL)
	{

		/* log the parameters passed if required (define in appitf.h) */
		LOG_CREATE_CONFERENCE_PARAMS;

		pstDTSParams = (SConfDTSParams *) UT_AllocMem(sizeof(SConfDTSParams));
		if (pstDTSParams == NULL)
		{
			UT_ErrorLog(APPITF, "No memory available\n");
			Status = VAPI_ERR_NOMEM;
			goto err_conf;
		}

		UT_MemCopy(pstDTSParams, pstConfDTSParams, sizeof(SConfDTSParams));
	}
	else
	{
		pstDTSParams = NULL;
	}

	pstConf->pstConfParams = UT_AllocMem(sizeof(SConfParams));
	if (pstConf->pstConfParams == NULL)
	{
		UT_ErrorLog(APPITF, "No memory available\n");
		Status = VAPI_ERR_NOMEM;
		goto err_ext_conf;
	}

	pstConf->apstParticipant = UT_Calloc(GTL_GetDefaultMaxPartPerConf(pstDevice->ucDevType), sizeof(SChnl *));
	if (pstConf->apstParticipant == NULL)
	{
		UT_ErrorLog(APPITF, "No memory available\n");
		Status = VAPI_ERR_NOMEM;
		goto err_ext_conf2;
	}

	pstConf->ConfId = ConfId;
	pstConf->bIsEnhancedConf = bIsEnhancedConf;
	pstConf->pstConfDTSParams = pstDTSParams;
	pstConf->pstDev = pstDevice;
	pstConf->bIsTranscoding = pstConfParams->bTranscoding;

	UT_MemCopy(pstConf->pstConfParams, pstConfParams, sizeof(SConfParams));

	/* Fill the request with the regular parameters */
	VCORE_SetConferenceRequest(pstVapiReq, pstRequest,
				VFSM_CreateConference,	/* pfnReqFSMHdlr */
				ConfId);

	bIsSync = pstVapiReq->bIsSync;

	/* Send VAPI request to VCORE */
	Status = VCORE_ProcessRequest(&pstDevice->stSupvChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_CreateConference: Exiting status(%d), conf(%u)\n", Status, ConfId);
/* no error exit */
	return Status;

err_ext_conf2:
	UT_FreeMem(pstConf->pstConfParams);

err_ext_conf:
	/* this pointer is used only if complex conferencing is enabled */
	if (bIsEnhancedConf && pstConfDTSParams != NULL)
		UT_FreeMem(pstDTSParams);

err_conf:
	UT_FreeMem(pstConf);

err_request:
	VCORE_FreeRequest(pstVapiReq);

err_out:
	return Status;
}

/***************************************************************************
 * APPITF_CreateParticipant: The function does the following things -
 ***************************************************************************/

VSTATUS APPITF_CreateParticipant(IN CONFID ConfId,
				 IN PARTID PartId,
				 IN U16 usTdmTimeSlot,
				 IN U8 ucAU_Law,
				 IN EPartType ePartType, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback)
{

	SVapiReq *pstVapiReq;
	SConference *pstConf;
	SVoIPChnlParams *pstChnDef;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;
	SChnl **ppstUserData;

	Status = VCORE_ConferenceGenericCheck(ConfId, "APPITF_CreateParticipant", &pstConf);

	if (Status != SUCCESS)
		goto err_out;

	if (DMGR_GetConnection(PartId) != NULL)
	{
		UT_ErrorLog(APPITF, "Connection %u already exists\n", PartId);
		Status = VAPI_ERR_INVALID_CONNID;
		goto err_out;
	}

	if (ucAU_Law > 1)
	{
		UT_ErrorLog(APPITF, "APPITF_CreateParticipant: Invalid ucAU_Law. \n");
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	if ((ePartType == eTRANSP) && (pstConf->bIsTranscoding == False))
	{
		UT_ErrorLog(APPITF, "APPITF_CreateParticipant: Transcoding not enabled for conference(%u) \n", ConfId);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SChnl *));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	/* Create the connection structure to handle the channel
	This structure is freed by DestroyParticipant or by the VFSM_CreateParticipant
	in case the device create conference fails */
	pstChnl = UT_Calloc(1, sizeof(SChnl));
	if (pstChnl == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_req;
	}

	pstChnl->ConnId = PartId;
	pstChnl->pstDev = pstConf->pstDev;
	pstChnl->pstConf = pstConf;
	pstChnl->pfnEventCallback = pfnEventCallback;
	pstChnl->usTimeSlot = usTdmTimeSlot;
	pstChnl->ePartType = ePartType;
	pstChnl->usConnType = eVOIP;	/* This field is only applicable for LSP participant */
		/* the connection is not active */
	pstChnl->bIsActive = False;

	UT_Log(APPITF, INFO, "APPITF_CreateParticipant: Participant(%d), conf(%u), type(%u)\n", pstChnl->ConnId, pstChnl->pstConf, pstChnl->ePartType);

	/* pass the connection structure pointer to the request*/ 
	ppstUserData = pstVapiReq->pvUserData;
	*ppstUserData = pstChnl;

	pstChnl->pstChnlParams = UT_AllocMem(sizeof(SVoIPChnlParams));
	if (pstChnl->pstChnlParams == NULL)
	{
		UT_ErrorLog(APPITF, "No memory available\n");
		Status = VAPI_ERR_NOMEM;
		goto err_chnl;
	}

	UT_MemSet(pstChnl->pstChnlParams, 0, sizeof(SVoIPChnlParams));

	/* retrieve the default configuration for this type of channel (VoIP or FoIP)
	   if the channel type is not VoIP or FoIP there are no defualt parameters */
	pstChnDef = DMGR_GetChnlDefaults(eVOIP);
	if (pstChnDef == NULL)
	{
		UT_ErrorLog(APPITF, "No defaults are available\n");
		Status = VAPI_ERR_NOMEM;
		goto err_params;
	}

	UT_MemCopy(pstChnl->pstChnlParams, pstChnDef, sizeof(SVoIPChnlParams));

	if (ePartType == eLSP)
	{
		pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.compander = ucAU_Law;
		pstChnl->pstChnlParams->stVopena.mode = VOIP_VOPENA_MODE_ENABLE_LSP;
	}

	/* Initialize few variables, timer,and mutexes */
	Status = VCORE_OpenConnection(pstChnl);
	if (Status != SUCCESS)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_params;
	}

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_CreateParticipant,	/* pfnReqFSMHdlr */
				PartId);

	bIsSync = pstVapiReq->bIsSync;

	/* process the request */
	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}


	UT_Log(APPITF, INFO, "APPITF_CreateParticipant: Exiting status(%d), conf(%u), conn(%u)\n", 
			Status, ConfId, PartId);

/* no error exit */
	return Status;

     err_params:
	UT_FreeMem(pstChnl->pstChnlParams);
     err_chnl:
	UT_FreeMem(pstChnl);
     err_req:
	VCORE_FreeRequest(pstVapiReq);
     err_out:
	return Status;
}

/***************************************************************************
 * VAPI_AllocateParticipant: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API creates a participant in narrow or wide band mode on a conference. 
*	Conference ID specifies which conference this participant is a party of. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConfId</td>
*		<td style="vertical-align: top;">The conference on which participant is to be created.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">Participant identifier. \n 
*		The participant ID maybe used in the same way as connection ID for all API calls that are
*		 relevant to an IP-side connection.\n
*		i.e. VAPI_SetConnIpParams(), the user application may use this API function and pass the
*		participant ID in place of connection ID to configure the destination IP/UDP port for this 
*		participant.\n 
*		The library NACKs all API calls that does not apply to an IP side participant (like 
*		VAPI_StartCallerId() and VAPI_StartTone() with TDM side generation).\n</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ePartType</td>
*		<td style="vertical-align: top;">Specifies the type of participant (eLSP eRSP, eTRANS) </td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">NumParam</td>
*		<td style="vertical-align: top;">Number of parameters</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pusParams</td>
*		<td style="vertical-align: top;">Parameters for the new connection (i.e timeslots).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnEventCallback</td>
*		<td style="vertical-align: top;">Connection Level event callback. \n
*		When events will be received for this participant this event callback will be called</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include allocate_participant.c
*
*	\n \b  Commands:
*	List of Comcerto commands sent:
*	\li CONF_CREATE_PARTICIPANT
*/

VSTATUS VAPI_AllocateParticipant (IN CONFID ConfId,
					IN PARTID PartId,
					IN EPartType ePartType,
					IN U8 ucNumOfParams,
					IN U16 *pusParams,
					IN SRequest * pstRequest,
					IN PFNEventCallback pfnEventCallback)
{
	SVapiReq *pstVapiReq;
	SConference *pstConf;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;
	SChnl **ppstUserData;
	int i;

	Status = VCORE_ConferenceGenericCheck(ConfId, "VAPI_AllocateParticipant", &pstConf);

	if (Status != SUCCESS)
		goto err_out;

	if (DMGR_GetConnection(PartId) != NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_AllocateParticipant: Connection %u already exists\n", PartId);
		Status = VAPI_ERR_INVALID_CONNID;
		goto err_out;
	}

	if ((ePartType == eTRANSP) && (pstConf->bIsTranscoding == False))
	{
		UT_ErrorLog(APPITF, "VAPI_AllocateParticipant: Transcoding not enabled for conference(%u) \n", ConfId);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SChnl *));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	/* Create the connection structure to handle the channel
	This structure is freed by DestroyParticipant or by the VFSM_CreateParticipant
	in case the device create conference fails */
	pstChnl = UT_Calloc(1, sizeof(SChnl));
	if (pstChnl == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_req;
	}

	pstChnl->ConnId = PartId;
	pstChnl->pstDev = pstConf->pstDev;
	pstChnl->pstConf = pstConf;
	pstChnl->pfnEventCallback = pfnEventCallback;
	pstChnl->ePartType = ePartType;
	pstChnl->usConnType = eVOIP;	/* This field is only applicable for LSP participant */
	pstChnl->bIsActive = False; 	/* The connection is not active */

	if (ePartType == eLSP || ePartType == eLSPWB)
		pstChnl->usTimeSlot = pusParams[0];

	if (ePartType == eLSPWB)
	{
		/*limit the number of TS to 4 (in case of user mistake)*/
		if (ucNumOfParams > MAX_WB_TS)
			ucNumOfParams = MAX_WB_TS;

		for (i = 0; i < ucNumOfParams - 1; i++)
			pstChnl->usExtTimeSlot[i] = pusParams[i + 1];
	}

	UT_Log(APPITF, INFO, "VAPI_AllocateParticipant: Participant(%d), conf(%u), type(%u)\n", pstChnl->ConnId, ConfId, pstChnl->ePartType);

	/* pass the connection structure pointer to the request*/ 
	ppstUserData = pstVapiReq->pvUserData;
	*ppstUserData = pstChnl;

	pstChnl->pstChnlParams = UT_AllocMem(sizeof(SVoIPChnlParams));
	if (pstChnl->pstChnlParams == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_AllocateParticipant: No memory available\n");
		Status = VAPI_ERR_NOMEM;
		goto err_chnl;
	}

	UT_MemSet(pstChnl->pstChnlParams, 0, sizeof(SVoIPChnlParams));

	DMGR_SetVoIPDevDefaults(pstChnl->pstChnlParams);

	/* Initialize few variables, timer,and mutexes */
	Status = VCORE_OpenConnection(pstChnl);
	if (Status != SUCCESS)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_params;
	}

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_AllocateParticipant, PartId);

	bIsSync = pstVapiReq->bIsSync;

	/* process the request */
	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}


	UT_Log(APPITF, INFO, "VAPI_AllocateParticipant: Exiting status(%d), conf(%u), conn(%u)\n", 
			Status, ConfId, PartId);

/* no error exit */
	return Status;

     err_params:
	UT_FreeMem(pstChnl->pstChnlParams);
     err_chnl:
	UT_FreeMem(pstChnl);
     err_req:
	VCORE_FreeRequest(pstVapiReq);
     err_out:
	return Status;
}

/***************************************************************************
 * VAPI_DestroyParticipant: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This function destroys a participant, similar to how a connection is destroyed. \n
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">Participant to be destroyed.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	 \n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include destroy_participant.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li CONF_DESTROY_PARTICIPANT
*/
VSTATUS VAPI_DestroyParticipant(IN PARTID PartId, IN SRequest * pstRequest)
{
	VSTATUS Status;

	UT_Log(APPITF, INFO, "Entered VAPI_DestroyParticipant: conn(%u)\n", PartId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_DESTROY_PARTICIPANT_PARAMS;

	Status = VAPI_DestroyConnection(PartId, pstRequest);
	UT_Log(APPITF, INFO, "VAPI_DestroyParticipant: Exiting status(%d), conn(%u)\n", Status, PartId);
	return Status;
}

/***************************************************************************
 * VAPI_DestroyConference: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This function shuts down a conference and removes all participants, if still active.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConfId</td>
*		<td style="vertical-align: top;">Conference session identifier.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	 \n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONFID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include destroy_conference.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li VOIP_VOPENA,
*	\li CONF_DESTROY_PARTICIPANT,
*	\li CONF_DESTROY_CONFERENCE
*/

VSTATUS VAPI_DestroyConference(IN CONFID ConfId, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SConference *pstConf;
	SChnl *pstSupvChnl;
	CONFID *pConfId;
	VSTATUS Status;
	Boolean bIsSync;

	Status = VCORE_ConferenceGenericCheck(ConfId, "VAPI_DestroyConference", &pstConf);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_DestroyConference: conf(%u)\n", ConfId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_DESTROY_CONFERENCE_PARAMS;

	pstSupvChnl = DMGR_GetChannel(pstConf->pstDev, SUPV_CHANNEL);

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(CONFID));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pConfId = pstVapiReq->pvUserData;

	*pConfId = ConfId;
	/* Fill the request with the regular parameters */
	VCORE_SetConferenceRequest(pstVapiReq, pstRequest,
				VFSM_DestroyConference,	/* pfnReqFSMHdlr */
				pstConf->ConfId);

	bIsSync = pstVapiReq->bIsSync;

	/* Send VAPI request to VCORE */
	Status = VCORE_ProcessRequest(pstSupvChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_DestroyConference: Exiting status(%d), confid(%u)\n", Status, ConfId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_EnableParticipant: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This function enables the participant to join the conference, similar to VAPI_EnableConnection API. \n
*	For TDM-side participants, this puts the connection into signaling state,
*	so tones maybe generated and detected on the TDM side. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">Participant identifier.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	 \n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include enable_participant.c
*
*	\n \b Commands:
*	Comcerto commands sent:
*	\li VOIP_VOPENA,
*	\li VOIP_VCEOPT
*/
VSTATUS VAPI_EnableParticipant(IN PARTID PartId, IN SRequest * pstRequest)
{
	VSTATUS Status;

	UT_Log(APPITF, INFO, "Entered VAPI_EnableParticipant: conn(%u)\n", PartId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_ENABLE_PARTICIPANT_PARAMS;

	Status = VAPI_SetConnectionState(PartId, eActive, pstRequest);
	UT_Log(APPITF, INFO, "VAPI_EnableParticipant: Exiting status(%d), conn(%u)\n", Status, PartId);
	return Status;
}

/***************************************************************************
 * VAPI_DisableParticipant: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Disables packet generation for IP side participant, and reverts the connection status back to signaling state. \n
*	In signaling state, the user application may detect tones and issue tone generation APIs.
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">Participant identifier.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	 \n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include disable_participant.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li VOIP_VOPENA
*	\li VOIP_VCEOPT
*/
VSTATUS VAPI_DisableParticipant(IN PARTID PartId, IN SRequest * pstRequest)
{
	VSTATUS Status;

	UT_Log(APPITF, INFO, "Entered VAPI_DisableParticipant: conn(%u)\n", PartId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_DISABLE_PARTICIPANT_PARAMS;

	Status = VAPI_SetConnectionState(PartId, eTdmActive, pstRequest);
	UT_Log(APPITF, INFO, "VAPI_DisableParticipant: Exiting status(%d), conn(%u)\n", Status, PartId);
	return Status;
}

/***************************************************************************
 * VAPI_PutParticipantOnHold: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API puts the participant ON hold or OFF hold. \n
*	A held participant may hear announcements instead of the conference. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">Participant identifier.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eHoldState</td>
*		<td style="vertical-align: top;">Hold state can be
*			eHOLDOFF, eSIMPLEHOLDON, eANNHOLD</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiSrcPartId</td>
*		<td style="vertical-align: top;">Source Participant Id (for announcement hold only)</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include hold_mute_participant.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li  CONF_PUT_PARTICIPANT_ON_HOLD
*/
VSTATUS VAPI_PutParticipantOnHold(IN U32 PartId, IN EHoldState eHoldState, IN U32 uiSrcPartId, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SChnl *pstChnl;
	SChnl *pstSrcChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	SHoldStateData *pstUserData;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(PartId, "VAPI_PutParticipantOnHold", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_PutParticipantOnHold: conn(%u)\n", PartId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_PUT_PARTICIPANT_ONHOLD_PARAMS;

	if (pstChnl->ePartType == eTRANSP)
	{
		UT_ErrorLog(APPITF, "VAPI_PutParticipantOnHold: Transcoding participant (%u) can't be holded\n", PartId);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if ((eHoldState != eUNHOLD) && (eHoldState != eSIMPLEHOLD) && (eHoldState != eANNHOLD))
	{
		UT_ErrorLog(APPITF, "VAPI_PutParticipantOnHold:Invalid Hold State conn(%u)\n", PartId);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if (eHoldState == eANNHOLD)
	{
		if ((pstSrcChnl = DMGR_GetConnection(uiSrcPartId)) == NULL)
		{
			UT_ErrorLog(APPITF, "Invalid source participant id, conn(%u)\n", PartId);
			Status = VAPI_ERR_INVALID_CONNID;
			goto out;
		}

		if (((pstSrcChnl->ePartType != eLSP)
			&& (pstSrcChnl->ePartType != eRSP)
			&& (pstSrcChnl->ePartType != eTRANSP)
			&& (pstSrcChnl->ePartType != eLSPWB)
			&& (pstSrcChnl->ePartType != eRSPWB))
			|| (pstChnl->pstConf == NULL) || (PartId == uiSrcPartId))
		{
			UT_ErrorLog(APPITF, "VAPI_PutParticipantOnHold:"
				    "Invalid source participant id, conn(%u)\n", PartId);
			if (PartId == uiSrcPartId)
			{
				UT_ErrorLog(APPITF, "VAPI_PutParticipantOnHold:Participant"
					    "Id to be hold is same as Src Part id, conn(%u)\n", PartId);
			}
			Status = VAPI_ERR_INVALID_CONNID;
			goto out;
		}
	}
	if ((pstChnl->ePartType != eLSP) && (pstChnl->ePartType != eRSP))
	{
		UT_ErrorLog(APPITF, "Invalid participant id: Not a Participant\n");
		Status = VAPI_ERR_INVALID_CONNID;
		goto out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SHoldStateData));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	pstUserData->eHoldState = eHoldState;
	if (eHoldState == eANNHOLD)
	{
		pstUserData->usSrcPartId = pstSrcChnl->usMSPChnlId;
	}
	else
	{
		pstUserData->usSrcPartId = 0x0000;
	}
	pstUserData->usMspConfId = pstChnl->pstConf->usMSPConfId;
	pstUserData->usMspPartId = pstChnl->usMSPChnlId;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_PutParticipantOnHold,	/* pfnReqFSMHdlr */
				PartId);

	bIsSync = pstVapiReq->bIsSync;

	/* Send VAPI request to VCORE */
	Status = VCORE_ProcessRequest(&(pstChnl->pstDev->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_PutParticipantOnHold: Exiting status(%d), conn(%u)\n", Status, PartId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_MuteParticipant: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Puts the participant on mute or off mute.\n 
*	A muted participant does not speak into the conference, but may still monitor the conference.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">Participant identifier.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">bMute</td>
*		<td style="vertical-align: top;">If True participant muted, if False particiant unmuted</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiSrcPartId</td>
*		<td style="vertical-align: top;">Source Participant Id (for announcement hold only)</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include hold_mute_participant.c
*
*	 \n \b Commands:
*	List of Comcerto commands sent:
*	\li CONF_MUTE_PARTICIPANT
 */
VSTATUS VAPI_MuteParticipant(IN U32 PartId, IN Boolean bMute, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;
	SParticipantData *pstUserData;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(PartId, "VAPI_MuteParticipant", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_MuteParticipant: conn(%u)\n", PartId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_MUTE_PARTICIPANT_ONHOLD_PARAMS;

	if (pstChnl->ePartType == eTRANSP)
	{
		UT_ErrorLog(APPITF, "VAPI_MuteParticipant: Transcoding participant (%u) can't be muted \n", PartId);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if (bMute != 0 && bMute != 1)
	{
		UT_ErrorLog(APPITF, "VAPI_MuteParticipant: Invalid bMute parameter (%d)\n", bMute);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if ((pstChnl->ePartType != eLSP) 
		&& (pstChnl->ePartType != eRSP) 
		&& (pstChnl->ePartType != eTRANSP)
		&& (pstChnl->ePartType != eLSPWB)
		&& (pstChnl->ePartType != eRSPWB))
	{
		UT_ErrorLog(APPITF, "Invalid participant id: Not a Participant\n");
		Status =  VAPI_ERR_INVALID_CONNID;
		goto out;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SParticipantData));
	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	pstUserData->usIsMute = (U16) bMute;
	pstUserData->usMspConfId = pstChnl->pstConf->usMSPConfId;
	pstUserData->usMspPartId = pstChnl->usMSPChnlId;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest,
				VFSM_MutePartId,	/* pfnReqFSMHdlr */
				PartId);

	bIsSync = pstVapiReq->bIsSync;

	/* Send VAPI request to VCORE */
	Status = VCORE_ProcessRequest(&(pstChnl->pstDev->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_MuteParticipant: Exiting status(%d), conn(%u)\n", Status, PartId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_CreateTdmSideParticipant: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Creates a TDM side conference participant with the ID specified by the user application. \n
*	The conference ID specifies which conference this participant is a party of. \n
*
*	\n \b Note 1:
*	<b>The VAPI_CreateTdmSideParticipant() is divided in 2 main steps:</b>\n
*	- <b>The device channel creation (CONF_CREATE_PART Comcerto command).</b>\n 
*	- <b>The device channel initialization (VCEOPT, VOPENA, AGCSET Comcerto commands).</b>\n
*	<b>If an error occurs during the step 1 the connection doesn't exist at all 
*	and the returned error is VAPI_ERR_CREATECONN_FAIL.\n
*	If an error occurs during the step 2 the connection exists but not initialized as expected.\n
*	In this case the user has the opportunity to send some configuration commands
*	(i.e VAPI_EchoCancellerReset() or VAPI_DestroyParticipant())
*	to remove the connection.</b> \n
*
*	\n \b Note 2:
*	<b>In case that the VAPI_InitDevice() function has not been previously issued, \n
*	VAPI_CreateTdmSideParticipant() returns VAPI_ERR_DEVICE_NOT_INITIALIZED</b> \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConfId</td>
*		<td style="vertical-align: top;">The conference on which participant is to be created.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">Participant identifier. \n 
*		The participant ID maybe used in the same way as connection ID for all API functions calls 
*		that are relevant to a TDM connection. \n
*		The library NACKs all API calls that does not apply to a TDM side participant (like 
*		VAPI_SetConnIpParams()</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usTdmTimeSlot</td>
*		<td style="vertical-align: top;">TDM timeslot for the new participant
*		(Like for VAPI_CreateConnection())</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ucAU_Law</td>
*		<td style="vertical-align: top;">Encoding to be used (A_LAW or U_LAW) 
*		(Like for VAPI_CreateConnection())</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnEventCallback</td>
*		<td style="vertical-align: top;">Connection Level event callback. \n
*		When events will be received for this participant this event callback will be called</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include create_tdm_participant.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li CONF_CREATE_PARTICIPANT
*	\li VoIP_VOPENA
*	\li VoIP_VCEOPT
*/
VSTATUS VAPI_CreateTdmSideParticipant(IN CONFID ConfId,
				      IN PARTID PartId,
				      IN U16 usTdmTimeSlot,
				      IN U8 ucAU_Law, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback)
{
	VSTATUS Status;

	UT_Log(APPITF, INFO, "Entered VAPI_CreateTdmSideParticipant: conf(%u), conn(%u)\n", ConfId, PartId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_CREATE_TDM_SIDE_PARTICIPANT_PARAMS;

	Status = APPITF_CreateParticipant(ConfId, PartId, usTdmTimeSlot, ucAU_Law, eLSP, pstRequest, pfnEventCallback);

	UT_Log(APPITF, INFO, "VAPI_CreateTdmSideParticipant: Exiting status(%d), conf(%u), conn(%u)\n", Status,
	       ConfId, PartId);
	return Status;

}

/***************************************************************************
 * VAPI_CreateIpSideParticipant: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Creates an IP-side conference participant with the ID specified by the user application. \n
*	The conference ID specifies which conference this participant is a party of. \n
*
*	\n \b Note 1:
*	<b>The VAPI_CreateIpSideParticipant() is divided in 2 main steps:</b>\n
*	- <b>The device channel creation (CONF_CREATE_PART Comcerto command).</b>\n 
*	- <b>The device channel initialization (VCEOPT, VOPENA, AGCSET Comcerto commands).</b>\n
*	<b>If an error occurs during the step 1 the connection doesn't exist at all 
*	and the returned error is VAPI_ERR_CREATECONN_FAIL.\n
*	If an error occurs during the step 2 the connection exists but not initialized as expected.\n
*	In this case the user has the opportunity to send some configuration commands
*	(i.e VAPI_SetConnIpParams() or VAPI_DestroyParticipant())
*	to remove the connection.</b> \n
*
*	\n \b Note 2:
*	<b>In case that the VAPI_InitDevice() function has not been previously issued, \n
*	VAPI_CreateIpSideParticipant() returns VAPI_ERR_DEVICE_NOT_INITIALIZED</b> \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConfId</td>
*		<td style="vertical-align: top;">The conference on which participant is to be created.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">Participant identifier. \n 
*		The participant ID maybe used in the same way as connection ID for all API calls that are
*		 relevant to an IP-side connection.\n
*		i.e. VAPI_SetConnIpParams(), the user application may use this API function and pass the
*		participant ID in place of connection ID to configure the destination IP/UDP port for this 
*		participant.\n 
*		The library NACKs all API calls that does not apply to an IP side participant (like 
*		VAPI_StartCallerId() and VAPI_StartTone() with TDM side generation).\n</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnEventCallback</td>
*		<td style="vertical-align: top;">Connection Level event callback. \n
*		When events will be received for this participant this event callback will be called</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include create_ip_participant.c
*
*	\n \b  Commands:
*	List of Comcerto commands sent:
*	\li CONF_CREATE_PARTICIPANT
*	\li VoIP_VOPENA
*	\li VoIP_VCEOPT
*/
VSTATUS VAPI_CreateIpSideParticipant(IN CONFID ConfId,
				     IN PARTID PartId, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback)
{
	VSTATUS Status;

	UT_Log(APPITF, INFO, "Entered VAPI_CreateIpSideParticipant: conf(%u) conn(%u)\n", ConfId, PartId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_CREATE_IP_SIDE_PARTICIPANT_PARAMS;

	Status = APPITF_CreateParticipant(ConfId, PartId, 0x00, 0x00, eRSP, pstRequest, pfnEventCallback);

	UT_Log(APPITF, INFO, "VAPI_CreateIpSideParticipant: Exiting status(%d) conf(%u) conn(%u)\n", Status,
	       ConfId, PartId);
	return Status;
}

/***************************************************************************
 * VAPI_CreateTranscodingParticipant: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	Creates an transcoding type conference participant with the ID specified by the user application. \n
*	The conference ID specifies which conference this participant is a party of. \n
*	The conference must have been previously created with the transcoding option enabled (bTranscoding param of SConfParams) \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConfId</td>
*		<td style="vertical-align: top;">The conference on which participant is to be created.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">Participant identifier. \n 
*		The participant ID maybe used in the same way as connection ID for all API calls that are relevant to an transcodng connection.\n
*		i.e. VAPI_SetConnIpParams(), the user application may use this API function and pass the participant ID in place of connection ID \n
*		to configure the destination IP/UDP port for this participant.\n
*		The library NACKs all API calls that does not apply to an transcoding participant \n
*		(like VAPI_StartCallerId() and VAPI_StartTone() with TDM side generation).\n</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnEventCallback</td>
*		<td style="vertical-align: top;">Connection Level event callback. \n
*		When events will be received for this participant this event callback will be called</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include create_transcoding_participant.c
*
*	\n \b  Commands:
*	List of Comcerto commands sent:
*	\li CONF_CREATE_PARTICIPANT
*/
VSTATUS VAPI_CreateTranscodingParticipant(IN CONFID ConfId,
				     IN PARTID PartId, IN SRequest * pstRequest, IN PFNEventCallback pfnEventCallback)
{
	VSTATUS Status;

	UT_Log(APPITF, INFO, "Entered VAPI_CreateTranscodingParticipant: conf(%u) conn(%u)\n", ConfId, PartId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_CREATE_TRANSCODING_PARTICIPANT_PARAMS;

	Status = APPITF_CreateParticipant(ConfId, PartId, 0x00, 0x00, eTRANSP, pstRequest, pfnEventCallback);

	UT_Log(APPITF, INFO, "VAPI_CreateTranscodingParticipant: Exiting status(%d) conf(%u) conn(%u)\n", Status,
	       ConfId, PartId);
	return Status;
}


/***************************************************************************
 * VAPI_SetParticipantDgain: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	This API sets the attenuation for a particular participant of a particular conference.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">The participant on which gain is to be changed.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usGain</td>
*		<td style="vertical-align: top;">The Gain value, the same value in MSP CRM command, 16bit.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*
*	</table>
*
*	\n \b Returns:
*  	SUCCESS \n
*	\li VAPI_ERR_LIB_NOT_INITIALIZED\n
*	\li VAPI_ERR_INVALID_CONNID\n
*	\li VAPI_ERR_DEV_IS_NOT_UP\n
*	\li VAPI_ERR_INVALID_PARAM\n
*	\li VAPI_ERR_NOMEM\n]
*
*	\n \b Usage:
*	\include set_participant_dgain.c
*
*	\n \b  Commands:
*	List of Comcerto commands sent:
*	\li CONF_SET_PARTICIPANT_DGAIN
*/
VSTATUS VAPI_SetParticipantDgain(IN PARTID PartId, IN U16 usGain, IN SRequest *pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl* pstChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;	
	SDgainStateData *pstUserData;

	/* proceed to some basic verification to check if VAPI, the Device, the
	connection id are in the rigth state and valid */

	Status = VCORE_ConnectionGenericCheck(PartId, "VAPI_SetParticipantDgain", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	 UT_Log(APPITF, INFO, "Entered VAPI_SetParticipantDgain: conn(%u)\n", PartId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_PARTICIPANT_DGAIN_PARAMS;

	if((pstChnl->ePartType != eLSP) 
		&& (pstChnl->ePartType != eRSP)
		&& (pstChnl->ePartType != eTRANSP)
		&& (pstChnl->ePartType != eLSPWB)
		&& (pstChnl->ePartType != eRSPWB))
	{
		UT_ErrorLog(APPITF, "Invalid participant id: Not a Participant\n");
		return VAPI_ERR_INVALID_CONNID;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SDgainStateData));

	if(pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	pstUserData->usGain = usGain;
	pstUserData->usMspConfId = pstChnl->pstConf->usMSPConfId;
	pstUserData->usMspPartId = pstChnl->usMSPChnlId;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, 
				VFSM_SetPartDgain, /* pfnReqFSMHdlr */
				PartId);

	bIsSync = pstVapiReq->bIsSync;

	/* Send VAPI request to VCORE */
	Status = VCORE_ProcessRequest(&(pstChnl->pstDev->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SetParticipantDgain: Exiting status(%d), conn(%u)\n", Status, PartId);

out:
	return Status;
}

/***************************************************************************
 * VAPI_SetParticipantOpt: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API enables/disables RFC2833 processing for a RSP participant.\n
*	This API can be send anytime while the conference in session.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">PartId</td>
*		<td style="vertical-align: top;">The participant on which gain is to be changed.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usRfc2833OnOff</td>
*		<td style="vertical-align: top;">RFC2833 Processing Feature (0 = Disable (default), 1 = Enable).</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*
*	</table>
*
*	\n \b Returns:
*  	SUCCESS \n
*	\li VAPI_ERR_LIB_NOT_INITIALIZED\n
*	\li VAPI_ERR_INVALID_CONNID\n
*	\li VAPI_ERR_DEV_IS_NOT_UP\n
*	\li VAPI_ERR_INVALID_PARAM\n
*	\li VAPI_ERR_NOMEM\n]
*
*	\n \b Usage:
*	\include set_participant_opt.c
*
*	\n \b  Commands:
*	List of Comcerto commands sent:
*	\li CONF_SET_PARTICIPANT_OPT
*/
VSTATUS VAPI_SetParticipantOpt(IN PARTID PartId, IN U16 usRfc2833OnOff, IN SRequest *pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl* pstChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;	
	SRFC2833StateData *pstUserData;

	/* proceed to some basic verification to check if VAPI, the Device, the
	connection id are in the rigth state and valid */

	Status = VCORE_ConnectionGenericCheck(PartId, "VAPI_SetParticipantOpt", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	 UT_Log(APPITF, INFO, "Entered VAPI_SetParticipantOpt: conn(%u)\n", PartId);

	if (pstChnl->ePartType == eTRANSP)
	{
		UT_ErrorLog(APPITF, "VAPI_SetParticipantOpt: Can't set options on transcoding participant (%u) \n", PartId);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_PARTICIPANT_OPT_PARAMS;

	if((pstChnl->ePartType != eLSP) 
		&& (pstChnl->ePartType != eRSP)
		&& (pstChnl->ePartType != eTRANSP)
		&& (pstChnl->ePartType != eLSPWB)
		&& (pstChnl->ePartType != eRSPWB))
	{
		UT_ErrorLog(APPITF, "Invalid participant id: Not a Participant\n");
		return VAPI_ERR_INVALID_CONNID;
	}

	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(SRFC2833StateData));

	if(pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	pstUserData->usRfc2833OnOff = usRfc2833OnOff;
	pstUserData->usMspConfId = pstChnl->pstConf->usMSPConfId;
	pstUserData->usMspPartId = pstChnl->usMSPChnlId;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, 
				VFSM_SetPartOpt, /* pfnReqFSMHdlr */
				PartId);

	bIsSync = pstVapiReq->bIsSync;

	/* Send VAPI request to VCORE */
	Status = VCORE_ProcessRequest(&(pstChnl->pstDev->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_SetParticipantOpt: Exiting status(%d), conn(%u)\n", Status, PartId);

out:
	return Status;
}

/***************************************************************************
 * VAPI_SetConferenceDgain: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API sets the attenuation for all participants of all conferences. In effect, it sets the default gain for any participant\n
* 	that is created. Every participant, when created, will have the gain value specified by this API.\n
*	The Host can override the default gain setting on a per-participant basis using the VAPI_SetParticipantDgain() call.\n
*	However, when the participant is destroyed (e.g. because the conference is over), the per-participant gain value is lost.\n
*	When a new participant is created, it will have the default gain value set by this API.\n
*
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Device identifier that uniquely identifies a device.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usGain</td>
*		<td style="vertical-align: top;">The attenuation value</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\b Returns: \n
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include set_conference_dgain.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li CONF_SET_DGAIN
*/
VSTATUS VAPI_SetConferenceDgain(IN DEVID DevId, IN U16 usGain, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;

	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_SetConferenceDgain", &pstDevice);
	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetConferenceDgain: dev(%u)\n", DevId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_CONFERENCE_DGAIN_PARAMS;

	pstChnl = &(pstDevice->stSupvChnl);
	if (pstChnl == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_SetConferenceDgain: pointer NULL\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}


	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(usGain));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	*((U16 *) pstVapiReq->pvUserData) = usGain;

	/* Fill the request with the regular parameters */
	VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_SetConfDgain,	/* pfnReqFSMHdlr */
			      DevId);

	bIsSync = pstVapiReq->bIsSync;

	/*Process Request */
	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}
	UT_Log(APPITF, INFO, "VAPI_SetConferenceDgainVAPI_SetConferenceDgain: Exiting status(%d), dev(%u)\n", Status, DevId);

      out:
	return Status;
}

/***************************************************************************
 * VAPI_SetConferenceLatency: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	This API sets the round-trip (speaker's mouth to speaker's ear) latency for TDM-only conferences.\n
*	It must be called a single time before any conference is created, and the latency specified applies to all conferences\n
*	created subsequently. The usLatency value specifies whether enhanced conferencing will be used.\n
*	Enhanced conferencing support is needed for AGC and/or Dominant Talker algorithm.\n
*	\n
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">DevId</td>
*		<td style="vertical-align: top;">Device identifier that uniquely identifies a device.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">usLatency</td>
*		<td style="vertical-align: top;">The round-trip latency value</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\b Returns: \n
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_DEVID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include set_conference_latency.c
*
*	\n \b Commands:
*	List of Comcerto commands sent:
*	\li CONF_SET_LATENCY
*/
VSTATUS VAPI_SetConferenceLatency(IN DEVID DevId, IN U16 usLatency, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq;
	SDevice *pstDevice;
	SChnl *pstChnl;
	VSTATUS Status;
	Boolean bIsSync;

	Status = VCORE_DeviceGenericCheck(DevId, "VAPI_SetConferenceLatency", &pstDevice);
	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_SetConferenceLatency: dev(%u)\n", DevId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_SET_CONFERENCE_LATENCY_PARAMS;

	pstChnl = &(pstDevice->stSupvChnl);
	if (pstChnl == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_SetConferenceLatency: pointer NULL\n");
		Status = VAPI_ERR_NULL_POINTER_PASSED;
		goto out;
	}


	/*allocate memory for the request and its UserData. If error get out */
	pstVapiReq = VCORE_AllocateRequest(sizeof(usLatency));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

 	*((U16 *) pstVapiReq->pvUserData) = usLatency;

	/* Fill the request with the regular parameters */
	VCORE_SetDeviceRequest(pstVapiReq, pstRequest, VFSM_SetConfLatency,	/* pfnReqFSMHdlr */
			      DevId);

	bIsSync = pstVapiReq->bIsSync;

	/*Process Request */
	Status = VCORE_ProcessRequest(&(pstDevice->stSupvChnl), pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}
	UT_Log(APPITF, INFO, "VAPI_SetConferenceLatency: Exiting status(%d), dev(%u)\n", Status, DevId);

      out:
	return Status;
}

/** @} */ /*conference level*/
