/*! \file conference_fsm.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include"ut.h"
#include"vapi.h"
#include"dmgr.h"
#include"vcore.h"
#include"msp.h"
#include "vcore_voip.h"
#include "appitf.h"

/****************************************************************************
 * VFSM_CreateConference : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent\n
 *          -#  CONF_SET_LATENCY\n
 *          -#  CONF_SET_DGAIN\n
 *          -#  CONF_CREATE_CONFERENCE\n
 *      -#  When new conference is created using CONF_CREATE_CONFERENCE\n
 *          -#  Initialize it using VCORE_InitConference\n
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings conference are done by APPITF\n.
 *
 *
 *  \return   None
 *
 *  \param pstChnl  Firstly this will be a supervisory channel on which 
 *                  CONF_CREATE_CONFERENCE request will be sent. 
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */
VSTATUS VFSM_CreateConference(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	SConference *pstConf;
	VSTATUS Status;
	U16 *pausFifo;
	DEVID DevId;
	DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_CreateConference: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	/* retrieve the connection pointer from the request */
	pstConf = *((SConference **)pstVapiReq->pvUserData);

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case CREATE_CONFERENCE_INIT:

		pstVapiReq->usReqState = CREATE_CONFERENCE_CHECK;
		Status = VDEV_CreateConf(pstChnl, pstConf->pstConfParams);
		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_CREATECONF_FAIL;
			goto finish;
		}

		break;

	case CREATE_CONFERENCE_CHECK:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CONF_CREATE_CONFERENCE);

		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_CREATECONF_FAIL;
			goto finish;
		}

		pausFifo = (U16 *) pstMsg->fifo;

		/* The device conference now exists.
		add the conference structure to the array for this device */
		pstConf->usMSPConfId = UT_LE2CPU16(pausFifo[5]);
		/*FIXME if we can't add the conference ID to the array the error is not correctly handled*/
		Status = DMGR_AddConference(pstChnl->pstDev, pstConf);
		if (Status != SUCCESS)
			goto finish;

		/* add the conference structure to the AVL tree */
		Status = VCORE_AddConference(pstConf);
		if (Status != SUCCESS)
		{
			goto finish;
		}

		UT_Log(VCORE, DEBUG, "Created MSP Conference(%u) on Dev(%u) ConfId(%u)\n", pstConf->usMSPConfId, DevId, pstConf->ConfId);

		/* If no extended conferencing parameters finish now */ 
		if (pstConf->pstConfDTSParams == NULL)
			goto finish;

		/* send Dominant talkers option if we want to enable it */
		if (!pstConf->pstConfDTSParams->bDTSOnOff)
			goto finish;

		pstVapiReq->usReqState = CREATE_CONFERENCE_FINISHED;
		Status = VDEV_ConfSetDTS(pstChnl, pstConf->usMSPConfId, pstConf->pstConfDTSParams);
		if (Status != SUCCESS)
			goto finish;

		break;

	case CREATE_CONFERENCE_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP,
						   FC_CONF_SPECIFY_DOMINANT_TALKERS);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_CreateConference: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_CreateConference: Completing req state(%u) status(%d) dev(%u)\n", pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	if (Status == VAPI_ERR_CREATECONF_FAIL)
	{
		/* we fall is this situation when the device conference has not been created.
		So we need to free the Conference stucture */ 
		if (pstConf->pstConfDTSParams != NULL)
			UT_FreeMem(pstConf->pstConfDTSParams);
		UT_FreeMem(pstConf);
	}
	/* In case of SUCCESS or error else than VAPI_ERR_CONFERENCE_FAIL the conference exists */
	UT_Log(VCORE, INFO, "VFSM_CreateConference: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_CreateParticipant : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#. Following MSP messages will be sent \n
 *          -#  FC_CONF_CREATE_PARTICIPANT \n
 *          -#  VCEOPT \n
 *          -#  VOPENA \n
 *          -#  AGCSET \n
 *      -#  When new channel is created using FC_CONF_CREATE_PARTICIPANT \n
 *          -#  Initialize it using VCORE_InitChannel \n
 *          -#  Add the request to the new channel and remove it from 
 *              Supervisory channel\n
 *          -#  Launch the pending request on Supvisory channel \n
 *      -#  All of the rest of MSP request are sent on the new channel \n
 *      -#  When request is completed give a calback or signal request 
 *          completion semaphore \n
 *
 *  - Assumptions:\n
 *      -#  Default Parameter settings are done by APPITF\n.
 *
 *  \return   None
 *
 *  \param pstChnl  Firstly this will be a supervisory channel on which 
 *                  SUPV_CREATE_CHANNEL request will be sent. Subsequent 
 *                  calls to this processing function will have newly
 *                  created channel as this parameter.
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *                  by APPITF this is passed as NULL.
 */
VSTATUS VFSM_CreateParticipant(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SVoIPChnlParams *pstChnlParams;
	SChnl *pstSupvChnl;
	SVapiReq *pstChildVapiReq;
	SConferenceParams *pstConferenceParams;
	DEVID DevId;
	CONNID ConnId;

	DevId = pstChnl->pstDev->DevId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_CreateParticipant: Entering Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);
	pstConferenceParams = DMGR_GetConfDefaults();
	pstChnlParams = pstChnl->pstChnlParams;

	/*Overwrite  DTS params with conference ones, this participant is to be related to*/
	if (pstChnl->pstConf->pstConfDTSParams != NULL)
		UT_MemCopy(&pstConferenceParams->stDTSParams, pstChnl->pstConf->pstConfDTSParams, sizeof (SConfDTSParams));

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case CREATE_CONF_PARTICIPANT_INIT:

		/* This allocate a new request */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SChnl *));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_CREATECONN_FAIL;
			goto finish;
		}

		/* User Data for child request is comming from the parent request */
		UT_MemCopy(pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, sizeof(SChnl *));

		pstVapiReq->usReqState = CREATE_CONF_PARTICIPANT_VCEOPT;
		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_CreateChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */
				VFSM_CreateDeviceParticipant,	/* Child Request Handler */
				CREATE_PARTICIPANT_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case CREATE_CONF_PARTICIPANT_VCEOPT:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_CREATECONN_FAIL;
			goto finish;
		}

		/* add the participant to the array for this conference */
		/*FIXME if we can't add the participant ID to the array the error is not correctly handled*/
		Status = DMGR_AddParticipant(pstChnl);
		if (Status != SUCCESS)
			goto finish;

		/* The device participant now exists
		add the connection structure to the AVL tree */
		Status = VCORE_AddConnection(pstChnl);
		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		UT_Log(VCORE, DEBUG, "Created MSP Participant(%u) on Dev(%u) Participant Id(%u)\n", pstChnl->usMSPChnlId, DevId, ConnId);

		/* skip AGC setting if not enabled */
		if (!pstConferenceParams->stAGCParams.param_4.bits.pkt2pcm_enb && !pstConferenceParams->stAGCParams.param_4.bits.pcm2pkt_enb)
			pstVapiReq->usReqState = CREATE_CONF_PARTICIPANT_VOPENA;
		else
			pstVapiReq->usReqState = CREATE_CONF_PARTICIPANT_SET_AGC;

		Status = VDEV_SendVceopt(pstChnl, &(pstChnlParams->stVoiceOpt));
		if (Status != SUCCESS)
			goto finish;

		break;

	/*FIXME will need to call VFSM_IniChannel in multi cmd mode when created */
	case CREATE_CONF_PARTICIPANT_SET_AGC:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VCEOPT);
		if (Status != SUCCESS)
			goto finish;

		/*Send VCEOPT with packet generation disabled */
		pstVapiReq->usReqState = CREATE_CONF_PARTICIPANT_VOPENA;
		/*Send AGC */
		Status = VDEV_PartAGCSET(pstChnl, &(pstConferenceParams->stAGCParams));
		if (Status != SUCCESS)
			goto finish;

		break;

	case CREATE_CONF_PARTICIPANT_VOPENA:
		/* check the rigth FC according if AGC setting skipped or not */
		if (!pstConferenceParams->stAGCParams.param_4.bits.pkt2pcm_enb && !pstConferenceParams->stAGCParams.param_4.bits.pcm2pkt_enb)
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VCEOPT);
		else
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_AGCSET);

		if (Status != SUCCESS)
			goto finish;

		pstChnlParams = pstChnl->pstChnlParams;
		/*Send VOPENA */
		pstVapiReq->usReqState = CREATE_CONF_PARTICIPANT_FINISHED;
		Status = VDEV_SendVoipVopena(pstChnl, &(pstChnlParams->stVopena));
		if (Status != SUCCESS)
			goto finish;

		break;

	case CREATE_CONF_PARTICIPANT_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_VOPENA);

		if ((Status == SUCCESS) && (pstChnl->ePartType == eLSP))
			pstChnl->bIsActive = True;

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_CreateParticipant: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	};

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_CreateParticipant: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	/* In these 2 cases we need to free the connection structure, the participant is not created */
	if ((Status == VAPI_ERR_CREATECONN_FAIL) || (Status == VAPI_ERR_MULTI_CMD_NOT_INITIALIZED))
	{
		VCORE_CloseConnection(pstChnl);
		UT_FreeMem(pstChnl);
	}

	UT_Log(VCORE, INFO, "VFSM_CreateParticipant: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_AllocateParticipant
 ***************************************************************************/
VSTATUS VFSM_AllocateParticipant(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SVoIPChnlParams *pstChnlParams;
	SChnl *pstSupvChnl;
	SVapiReq *pstChildVapiReq;
	DEVID DevId;
	CONNID ConnId;

	DevId = pstChnl->pstDev->DevId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_AllocateParticipant: Entering Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstSupvChnl = DMGR_GetChannel(pstChnl->pstDev, SUPV_CHANNEL);
	pstChnlParams = pstChnl->pstChnlParams;

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case ALLOCATE_CONF_PARTICIPANT_INIT:
		/* This allocate a new request */
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SChnl *));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_CREATECONN_FAIL;
			goto finish;
		}

		/* User Data for child request is comming from the parent request */
		UT_MemCopy(pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, sizeof(SChnl *));

		pstVapiReq->usReqState = ALLOCATE_CONF_PARTICIPANT_FINISHED;
		/* initialise the Child request 
		The UserData is the channelID used by the VFSM_CreateChannel handler */
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */
				VFSM_CreateDeviceParticipant,	/* Child Request Handler */
				CREATE_PARTICIPANT_INIT);	/* Child Request handler state */

		/* process request in the supervisor channel */
		VCORE_ProcessRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case ALLOCATE_CONF_PARTICIPANT_FINISHED:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
		{
			Status = VAPI_ERR_CREATECONN_FAIL;
			goto finish;
		}

		/* add the participant to the array for this conference */
		/*FIXME if we can't add the participant ID to the array the error is not correctly handled*/
		Status = DMGR_AddParticipant(pstChnl);
		if (Status != SUCCESS)
			goto finish;

		UT_Log(VCORE, DEBUG, "Created MSP Participant(%u) on Dev(%u) Participant Id(%u)\n", pstChnl->usMSPChnlId, DevId, ConnId);

		/* The device participant now exists
		add the connection structure to the AVL tree if it's not add yet*/
		if(VCORE_CheckConnection(pstChnl) != SUCCESS)
			Status = VCORE_AddConnection(pstChnl);

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_AllocateParticipant: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;

	};

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_AllocateParticipant: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	/* In these 2 cases we need to free the connection structure, the participant is not created */
	if ((Status == VAPI_ERR_CREATECONN_FAIL) || (Status == VAPI_ERR_MULTI_CMD_NOT_INITIALIZED))
	{
		VCORE_CloseConnection(pstChnl);
		UT_FreeMem(pstChnl);
	}

	UT_Log(VCORE, INFO, "VFSM_AllocateParticipant: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_DestroyParticipant : The function does the following things -
 ***************************************************************************/
/*! 
 *  \b Implementation: \n
 *      -#  Send destroy participant to the device
 *
 *  \b Inputs-Outputs: \n
 *	\li pstVapiRep->pvUserData contains the Conference ID/Participant ID to destroy
 *	\li pstChnl The channel on which the request is going.
 *	\li pstVapiReq The processed request.
 *	\li pstMsg  Message obtained from the device.
 *
 *  \b Returns: \n
 *	\li SUCCESS
 *	\li VAPI_ERR_INVALID_PARAM
 *
 */
VSTATUS VFSM_DestroyParticipant(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status=SUCCESS;
	SFsmParticipantInfo *ParticipantInfo;
	SChnl *pstChnl;
	DEVID DevId;
	DevId = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_DestroyParticipant: Entering dev(%u) state(%d)\n", DevId, pstVapiReq->usReqState);

	ParticipantInfo = (SFsmParticipantInfo *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case DESTROY_PARTICIPANT_INIT:

		pstVapiReq->usReqState = DESTROY_PARTICIPANT_FINISHED;
		Status = VDEV_DestroyPart(pstSupvChnl, ParticipantInfo->DevPartId, ParticipantInfo->DevConfId);

		if (Status != SUCCESS)
			goto finish;

		break;
 
	case DESTROY_PARTICIPANT_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstSupvChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CONF_DESTROY_PARTICIPANT);

		/* If we are trying to destroy a non existing device channel
		while the connection exists. So let continue with the regular process to remove the 
		VAPI conenction structure*/ 
		if (Status == -ERR_CHANCTRL_NO_SUCH_CHAN)
		{
			UT_ErrorLog(VCORE, "VFSM_DestroyParticipant: Device channel doesn't (%u) exist continue\n", ParticipantInfo->DevPartId);
			Status = SUCCESS;
		}

		pstChnl = DMGR_GetChannel(pstSupvChnl->pstDev, ParticipantInfo->DevPartId);

		if (pstChnl == NULL)
		{
			UT_ErrorLog(VCORE, "VFSM_DestroyParticipant %u doesn't exists\n", ParticipantInfo->DevPartId);
			Status = VAPI_ERR_INVALID_CONNID;
			goto finish;
		}

		if (Status == SUCCESS)
		{
			VCORE_CleanParticipant(pstChnl);
			DMGR_RemoveChannel(pstSupvChnl->pstDev, ParticipantInfo->DevPartId);
		}

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_DestroyParticipant: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:
	UT_Log(VCORE, DEBUG,"VFSM_DestroyParticipant: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_DestroyParticipant: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;
}

/****************************************************************************
 * VFSM_DestroyConference : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Destroy all live participant.
 *      -#  Semd MSP request for CONF_DESTROY_CONFERENCE
 *      -#  When response is received, inform application
 *      -#  Cleanup the channel from device manger
 *
 *  - Assumptions
 *      -#  There will not be any other request queued up on the channel
 *          being removed.
 *  
 *  \return 
 *  VAPI_ERR_INVALID_PARAM Invalid parameter supplied
 *  
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_DestroyConference(IN SChnl * pstSupvChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SVapiReq *pstChildVapiReq;
	SConference *pstConf;
	DEVID DevId;
	DevId = pstSupvChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_DestroyConference: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	pstConf = DMGR_GetConfById(*((CONFID *)pstVapiReq->pvUserData));
	if (pstConf == NULL)
	{
		UT_ErrorLog(VCORE, "VFSM_DestroyConference: Conference %u doesn't exists\n", 
			*((CONFID *)pstVapiReq->pvUserData));
		Status = VAPI_ERR_INVALID_CONNID;
		goto finish;
	}

	switch (pstVapiReq->usReqState)
	{

	case DESTROY_CONFERENCE_INIT:

		/* this allocate a new request (no User Data required) */ 
		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(CONFID));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		UT_MemCopy(pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, sizeof(CONFID));

		/* the VFSM_PurgeConference function will destroy any pending
		 participant of the current conference*/ 
		/* initialise the Child request 
		No particular data are required for the child fsm_handler */ 
		VCORE_SetChildRequest(pstChildVapiReq,	/* Child Request */
				pstVapiReq,		/* Parent Request */ 
				VFSM_PurgeConference,	/* Child Request Handler */
				PURGE_CONFERENCE_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = DESTROY_CONFERENCE_DESTROY_CONF;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstSupvChnl, pstChildVapiReq);

		break;

	case DESTROY_CONFERENCE_DESTROY_CONF:

		/* retrieve the child status */
		Status = pstVapiReq->Status;
		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = DESTROY_CONFERENCE_FINISHED;
		Status = VDEV_DestroyConf(pstSupvChnl, pstConf->usMSPConfId);
		if (Status != SUCCESS)
			goto finish;

		break;


	case DESTROY_CONFERENCE_FINISHED:

		Status = VCORE_CheckStatus(Event, pstMsg, pstSupvChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CONF_DESTROY_CONFERENCE);

		if( Status == SUCCESS)
		{
			/*remove the conference from the AVL tree */
			VCORE_RemoveConference(pstConf);
			/*remove the conference from device manager */
			/* Free the conference structure */
			DMGR_RemoveConf(pstSupvChnl->pstDev, pstConf->usMSPConfId);
		}

		goto finish;

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_DestroyConference: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_DestroyConference: Completing req state(%u) status(%d) dev(%u)\n",
			pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstSupvChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_DestroyConference: Exiting status(%d) dev(%u)\n", Status, DevId);

	return Status;

}

/****************************************************************************
 * VFSM_PutParticipantOnHold : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Call put participant on hold and check its status\n
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_PutParticipantOnHold(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SHoldStateData *pstHoldStateData;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_PutParticipantOnHold: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstHoldStateData = (SHoldStateData *) pstVapiReq->pvUserData;

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case HOLD_PART_INIT:
		pstVapiReq->usReqState = HOLD_PART_FINISHED;
		Status = VDEV_HoldParticipant(pstChnl,
					       pstHoldStateData->eHoldState,
					       pstHoldStateData->usSrcPartId,
					       pstHoldStateData->usMspConfId, pstHoldStateData->usMspPartId);
		if (Status != SUCCESS)
			goto finish;

		break;

	case HOLD_PART_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, SUPV_CHANNEL,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CONF_PUT_PARTICIPANT_ON_HOLD);

		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_PutParticipantOnHold: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		

	};

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_PutParticipantOnHold: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_PutParticipantOnHold: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_MutePartId : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -# Call mute participant and check its status\n
 *  \return 
 *  SUCCESS 
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_MutePartId(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	SParticipantData *pstUserData;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_MutePartId: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SParticipantData *) pstVapiReq->pvUserData;
	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case MUTE_PART_INIT:
		pstVapiReq->usReqState = MUTE_PART_FINISHED;
		Status = VDEV_MuteParticipant(pstChnl, pstUserData->usIsMute,
					       pstUserData->usMspConfId, pstUserData->usMspPartId);
		if (Status != SUCCESS)
			goto finish;
		break;

	case MUTE_PART_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, SUPV_CHANNEL,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CONF_MUTE_PARTICIPANT);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_MutePartId: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	};

	return SUCCESS;

finish:

	UT_Log(VCORE, DEBUG, "VFSM_MutePartId: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_MutePartId: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SetPartDgain : The function does the following things -
 ***************************************************************************/
/*!
 *  - Implementation
 *      -# Call set  participant dgain and check its status\n
 *  \return
 *  SUCCESS
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_SetPartDgain(IN SChnl *pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t *pstMsg)
{
	VSTATUS Status;
	SDgainStateData *pstUserData = NULL;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE,DEBUG,"VFSM_SetPartDgain: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SDgainStateData *) pstVapiReq->pvUserData;

	/*Take action according to current state of request*/
	switch(pstVapiReq->usReqState)
	{
	case SET_PART_DGAIN_INIT:
		pstVapiReq->usReqState = SET_PART_DGAIN_FINISHED;

		Status = VDEV_SetPartDgain(pstChnl, pstUserData->usGain, 
					pstUserData->usMspConfId, pstUserData->usMspPartId);
		if (Status != SUCCESS)
			goto finish;
		break;

	case SET_PART_DGAIN_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg,SUPV_CHANNEL,
					CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP,FC_CONF_SET_PARTICIPANT_DGAIN);
		goto finish;
		break;
	};

	return SUCCESS;
finish:

	UT_Log(VCORE,DEBUG, "VFSM_SetPartDgain: Completing req state(%u) status(%u) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE,INFO, "VFSM_SetPartDgain: Exiting with status(%u) conn(%u)\n",Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SetPartOpt: The function does the following things -
 ***************************************************************************/
/*!
 *  - Implementation
 *      -# Call set  participant opt and check its status\n
 *  \return
 *  SUCCESS
 *  VAPI_ERR_INVALID_PARAM
 *  VAPI_ERR_UNDEFINED
 *
 *  \param pstChnl The channel on which the request is going.
 *  \param pstMsg  Message obtained from MSP.
 */
VSTATUS VFSM_SetPartOpt(IN SChnl *pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t *pstMsg)
{
	VSTATUS Status;
	SRFC2833StateData *pstUserData = NULL;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE,DEBUG,"VFSM_SetPartOpt: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstUserData = (SRFC2833StateData *) pstVapiReq->pvUserData;

	/*Take action according to current state of request*/
	switch(pstVapiReq->usReqState)
	{
	case SET_PART_OPT_INIT:
		pstVapiReq->usReqState = SET_PART_OPT_FINISHED;

		Status = VDEV_SetPartOpt(pstChnl, pstUserData->usRfc2833OnOff, 
					pstUserData->usMspConfId, pstUserData->usMspPartId);
		if (Status != SUCCESS)
			goto finish;
		break;

	case SET_PART_OPT_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg,SUPV_CHANNEL,
					CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP,FC_CONF_SET_PARTICIPANT_OPT);
		goto finish;
		break;
	};

	return SUCCESS;
finish:

	UT_Log(VCORE,DEBUG, "VFSM_SetPartOpt: Completing req state(%u) status(%u) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE,INFO, "VFSM_SetPartOpt: Exiting with status(%u) conn(%u)\n",Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_SetConfDgain : The function does the following things -
 ***************************************************************************/
VSTATUS VFSM_SetConfDgain(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	U16 usLatency;
	DEVID DevId;
	DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetConfDgain: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	usLatency = *((U16 *) pstVapiReq->pvUserData);

	switch (pstVapiReq->usReqState)
	{
	case SET_CONF_DGAIN_INIT:
		pstVapiReq->usReqState = SET_CONF_DGAIN_FINISHED;

		Status = VDEV_SetConfDgain(pstChnl, usLatency);
		if (Status != SUCCESS)
			goto finish;
		break;

	case SET_CONF_DGAIN_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CONF_SET_DGAIN);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetConfDgain: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;
finish:

	UT_Log(VCORE, DEBUG, "VFSM_SetConfDgain: Completing req state(%u) status(%d) dev(%u)\n",
		       pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetConfDgain: Exiting status(%d) dev(%u)\n", Status, DevId);
	return Status;
}

/****************************************************************************
 * VFSM_SetConfLatency : The function does the following things -
 ***************************************************************************/
VSTATUS VFSM_SetConfLatency(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status;
	U16 usLatency;
	DEVID DevId;
	DevId = pstChnl->pstDev->DevId;

	UT_Log(VCORE, DEBUG, "VFSM_SetConfLatency: Entering dev(%u) state(%u)\n", DevId, pstVapiReq->usReqState);

	usLatency = *((U16 *) pstVapiReq->pvUserData);

	switch (pstVapiReq->usReqState)
	{
	case SET_CONF_LATENCY_INIT:
		pstVapiReq->usReqState = SET_CONF_LATENCY_FINISHED;

		Status = VDEV_SetConfLatency(pstChnl, usLatency);
		if (Status != SUCCESS)
			goto finish;
		break;

	case SET_CONF_LATENCY_FINISHED:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_RESP, FC_CONF_SET_LATENCY);
		goto finish;
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_SetConfLatency: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}

	return SUCCESS;
finish:

	UT_Log(VCORE, DEBUG, "VFSM_SetConfLatency: Completing req state(%u) status(%d) dev(%u)\n",
		       pstVapiReq->usReqState, Status, DevId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_SetConfLatency: Exiting status(%d) dev(%u)\n", Status, DevId);
	return Status;
}

