/*! \file announcement_itf.c
 @defgroup announcement Announcement related API functions
 *  @{
 */

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "cfg.h"
#include "msp.h"

/***************************************************************************
 * VAPI_StartRecord: The function does the following things -
 ***************************************************************************/
/*! 
*	 \b Description: \n
*	This API is used handle host speech frames coming on the channel \n
*	for recording the input frames in host speech frame format as given in CRM Section 
*	"Formatting of Recorded Speech Data".\n 
*	If application passes a non-NULL pfnFrameHndlr then VAPI will call the pfnFrameHndlr \n
*	as soon as it retrieves a complete frame from SYNCDAT/SYNEOF indications.\n 
*	If application provides a pointer to a buffer and pfnFrameHndlr is NULL then VAPI will write the frames \n
*	it will receive into the buffer till the buffer is full or timeout occurs.\n
*	VAPI_StopRecording() API can be called any time to stop recording.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which to start the recording.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRecData</td>
*		<td style="vertical-align: top;">Pointer to rec data.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include startrecord.c
*
*	\n \b Commands:
*	- SET_FLOWCON
*	- SET_CHANNEL_MODE
*	- SYNCDAT
*	- SYNCEOF
*	- AS_STOP
*	- DRAIN_PLAYREC_BUFFER
*/
VSTATUS VAPI_StartRecord(IN CONNID ConnId, IN SRecData * pstRecData, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	VSTATUS Status = SUCCESS;
	Boolean bIsSync;
	SHostSpeechData *pstUserData;
	SVAPIDeviceConfig *pstDevConfig;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_StartRecord", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_StartRecord: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_START_RECORD_PARAMS;

	if (ePCI != pstRecData->eMediaHost && eCSMENCAPS != pstRecData->eMediaHost)
	{
		UT_ErrorLog(APPITF, "VAPI_StartRecord: Invalid MediaHost %d\n", pstRecData->eMediaHost);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if (ePCM != pstRecData->eSourceType && eRTP != pstRecData->eSourceType && eBOTH != pstRecData->eSourceType)
	{
		UT_ErrorLog(APPITF, "VAPI_StartRecord: Invalid Source type %d\n", pstRecData->eSourceType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	pstDevConfig = DMGR_GetVAPIDeviceConfig(pstChnl->pstDev->ucDevType);
	if (pstDevConfig == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_StartRecord: Device type for this conn(%u) is invalid \n", ConnId);
		Status = VAPI_ERR_INVALID_DEV_TYPE;
		goto out;
	}

	if ((pstRecData->eCodingScheme != eCS_G711_ULAW) &&
		(pstRecData->eCodingScheme != eCS_G711_ALAW) &&
		(pstRecData->eCodingScheme != eCS_G723_1) &&
		(pstRecData->eCodingScheme != eCS_G729_A) &&
		(pstRecData->eCodingScheme != eCS_G726_32) &&
		(pstRecData->eCodingScheme != eCS_G726_16_audio) &&
		(pstRecData->eCodingScheme != eCS_G726_24_audio) &&
		(pstRecData->eCodingScheme != eCS_G726_40_audio) &&
		(pstRecData->eCodingScheme != eCS_RawPcm))
	{
		UT_ErrorLog(APPITF, "VAPI_StartRecord: Invalid coding scheme %d\n", pstRecData->eCodingScheme);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	if (pstChnl->usConnType != eVOIP)
	{
		UT_ErrorLog(APPITF, "VAPI_StartRecord: Invalid channel type %d\n", pstChnl->usConnType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto out;
	}

	pstVapiReq = VCORE_AllocateRequest(sizeof(SHostSpeechData));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto out;
	}

	pstUserData = pstVapiReq->pvUserData;

	pstUserData->ucMediaHost = pstRecData->eMediaHost;

	if (ePCM == pstRecData->eSourceType)
	{
		if (pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.compander == 0)
			pstUserData->ucSrcCoding = pstDevConfig->astCodecInfo[eG711_ALAW_PCM].usPTIndex;
		else
			pstUserData->ucSrcCoding = pstDevConfig->astCodecInfo[eG711_ULAW_PCM].usPTIndex;
	}
	else
		pstUserData->ucSrcCoding =
		    pstDevConfig->astCodecInfo[pstChnl->pstChnlParams->eCodecVal].usPTIndex;

	pstUserData->ucDestinationType = eHSF;
	pstUserData->ucSourceType = pstRecData->eSourceType;
	pstUserData->pstSTimer = UT_TimerCreate();
	pstUserData->uiTimeOut = pstRecData->uiTimeOut;
	pstUserData->ucDestCoding = pstRecData->eCodingScheme;
	pstUserData->pucBuffer = pstRecData->pucBuffer;
	pstUserData->uiBufferLength = pstRecData->uiBufferLength;
	UT_StrCpy((char *) pstUserData->strAnnFileName, pstRecData->strAnnFileName);
	pstUserData->pfnFrameHndlr = pstRecData->pfnFrameHndlr;
	pstUserData->uiHostSpeechDataFormat = pstRecData->uiHostSpeechDataFormat;	
	
	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_StartRecord,	/* pfnReqFSMHdlr */
				  ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_StartRecord: Exiting status(%d), conn(%u)\n", Status, ConnId);

/* no error exit */
out:
	return Status;
}

/***************************************************************************
 * VAPI_StartRecording: The function does the following things -
 ***************************************************************************/
/*! 
*	 \b Description: \n
*	This one is now obsolete and replaced by VAPI_StartRecord().
*	It is kept in this version for backward compatibility but may be removed in the next releases.
*
*	This API is used handle host speech frames coming on the channel \n
*	for recording the input frames in host speech frame format as given in CRM Section 
*	"Formatting of Recorded Speech Data".\n 
*	If application passes a non-NULL pfnFrameHndlr then VAPI will call the pfnFrameHndlr \n
*	as soon as it retrieves a complete frame from SYNCDAT/SYNEOF indications.\n 
*	If application provides a pointer to a buffer and pfnFrameHndlr is NULL then VAPI will write the frames \n
*	it will receive into the buffer till the buffer is full or timeout occurs.\n
*	VAPI_StopRecording() API can be called any time to stop recording.\n
*
*	\n 
*	<table style="text-align: left;" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which to start the recording.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eMediaHost</td>
*		<td style="vertical-align: top;">Supported values:
*				- 0 for PCI
*				- 1 for CSM_ENCAPS 
*				Not being used in current implementation.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eCodingScheme</td>
*		<td style="vertical-align: top;">Destination coding scheme for Host Speech.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eSourceType</td>
*		<td style="vertical-align: top;">Source can be one of the following:
*					- 8-bit PCM
*					- RTP
*					- Both.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiTimeOut</td>
*		<td style="vertical-align: top;">Specifies the time (in Sec) for which recording needs to be done. \n
*					If its value is 0 ignore this parameter.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pucBuffer</td>
*		<td style="vertical-align: top;">Pointer to the Buffer in which recorded data needs to be stored.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiBufferLength</td>
*		<td style="vertical-align: top;">Length of the Buffer passed to the API.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnFrameHndlr</td>
*		<td style="vertical-align: top;">If not NULL, pfnFrameHndlr is called for 
*				each complete frame received from SYNCDAT/SYNEOF.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include startrecording.c
*
*	\n \b Commands:
*	- SET_FLOWCON
*	- SET_CHANNEL_MODE
*	- SYNCDAT
*	- SYNCEOF
*	- AS_STOP
*	- DRAIN_PLAYREC_BUFFER
*/
VSTATUS VAPI_StartRecording(IN CONNID ConnId,
			IN EMediaHostType eMediaHost,
			IN ECodingScheme eCodingScheme,
			IN EAnnounceType eSourceType,
			IN U32 uiTimeOut,
			IN U8 * pucBuffer,
			IN U32 uiBufferLength, IN PFNIoCompCallback pfnFrameHndlr, IN SRequest * pstRequest)
{
	SRecData stRecData;

	stRecData.eMediaHost = eMediaHost;
	stRecData.eCodingScheme = eCodingScheme;
	stRecData.eSourceType = eSourceType;
	stRecData.uiTimeOut = uiTimeOut;
	stRecData.pucBuffer = pucBuffer;
	stRecData.uiBufferLength = uiBufferLength;
	stRecData.pfnFrameHndlr = pfnFrameHndlr;
	stRecData.uiHostSpeechDataFormat = eHedFormat;

	return VAPI_StartRecord(ConnId, &stRecData, pstRequest);
}

/***************************************************************************
 * VAPI_StartRecordFile: The function does the following things -
 ***************************************************************************/
/*! 
*	 \b Description: \n
*	This API is used handle host speech frames coming on the channel \n
*	for recording the input frames in host speech frame format as given in CRM Section 
*	"Formatting of Recorded Speech Data".\n 
*	If application passes a non-NULL pfnFrameHndlr then VAPI will call the pfnFrameHndlr \n
*	as soon as it retrieves a complete frame from SYNCDAT/SYNEOF indications.\n 
*	If pfnFrameHndlr is NULL then VAPI will write the frames\n
*	it will receive into the file till timeout occurs.\n
*	VAPI_StopRecording() / VAPI_StopRecord() APIs can be called any time to stop recording.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which to start the recording.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRecData</td>
*		<td style="vertical-align: top;">Pointer to rec data.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include startrecordfile.c
*
*	\n \b Commands:
*	- SET_FLOWCON
*	- SET_CHANNEL_MODE
*	- SYNCDAT
*	- SYNCEOF
*	- AS_STOP
*	- DRAIN_PLAYREC_BUFFER
*/
VSTATUS VAPI_StartRecordFile(IN CONNID ConnId, IN SRecData * pstRecData, IN SRequest * pstRequest)
{
	pstRecData->pucBuffer = NULL;
	pstRecData->uiBufferLength = ~0;

	return VAPI_StartRecord(ConnId, pstRecData, pstRequest);

}

/***************************************************************************
 * VAPI_StopRecord: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API will stop any ongoing recording on this channel. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the recording has to be stopped.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eStopAnnType</td>
*		<td style="vertical-align: top;">Stop type of announcement API.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include stoprecord.c
*/
VSTATUS VAPI_StopRecord(IN CONNID ConnId, IN EStopAnnType eStopAnnType, IN SRequest * pstRequest)
{
/*    SVapiReq *pstVapiReq = NULL;*/
	SChnl *pstChnl = NULL;
	SLstNode *pstNode;
	VSTATUS Status = SUCCESS;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_StopRecord", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_StopRecord: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_STOP_RECORD_PARAMS;

	if (pstChnl->stRecPlayInfo.uiStopRecCount < pstChnl->stRecPlayInfo.uiStartRecCount)
	{
		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.eStopAnnType = eStopAnnType;
		pstChnl->stRecPlayInfo.uiStopRecCount++;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);

		/*Get Front node */
		UT_MutexLock(&pstChnl->stReqList.stSem);
		
		pstNode = __UT_GetFrontNode(&pstChnl->stReqList);

		if (pstNode == NULL)
		{
			UT_ErrorLog(VCORE, "VAPI_StopRecord: Request list empty\n");
			UT_MutexUnLock(&pstChnl->stReqList.stSem);
			goto out;
		}
		UT_MutexUnLock(&pstChnl->stReqList.stSem);

		VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_PLAY_REC_STOP, NULL);
		
		Status = SUCCESS;
		
	}
	else
	{
		UT_ErrorLog(APPITF, "VAPI_StopRecord: No start recording called w.r.t. to stop recording\n");
		Status = VAPI_ERR_STOP_REC_EXIST;
	}

	UT_Log(APPITF, INFO, "VAPI_StopRecord: Exiting status(%d), conn(%u)\n", Status, ConnId);

out:
	/* if ASYNC mode call the callback function*/
	if ((pstRequest != NULL) && (pstRequest->pfnIoCompCallback != NULL))
	{
		pstRequest->pfnIoCompCallback(ConnId, CMD_LEVEL_CONN, Status, NULL, 0, pstRequest->uiReqId);
		Status = VAPI_ERR_PENDING;
	}

	return Status;
}

/***************************************************************************
 * VAPI_StopRecording: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This one is now obsolete and replaced by VAPI_StopRecord().
*	It is kept in this version for backward compatibility but may be removed in the next releases.
*
*	This API will stop any ongoing recording on this channel. \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the recording has to be stopped.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include stoprecording.c
*/
VSTATUS VAPI_StopRecording(IN CONNID ConnId, IN SRequest * pstRequest)
{
	return VAPI_StopRecord(ConnId, eAS_STOP, pstRequest);
}

/***************************************************************************
 * VAPI_StartPlayback: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API will start the playback from the buffer/file on the channel till the complete buffer/file 
*	is exhausted or timeout occurs. \n
*	The buffer/file should have data in Host Speech Frame format as given in CRM Section 
*	"Formatting of Recorded Speech Data".\n
*	VAPI_StopPlaying() / VAPI_StopPlayback()  API can be called at any time to stop the ongoing playout.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which to start the playing.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstPlayData</td>
*		<td style="vertical-align: top;">Pointer to play data .</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include startplayback.c
*
*	\n \b Commands:
*	- SET_CHANNEL_MODE
*	- SYNCDAT
*	- SYNCEOF
* 	- VOIP_AS_SET_RATE 
*	- AS_STOP
*	- DRAIN_PLAYREC_BUFFER
*/
VSTATUS VAPI_StartPlayback(IN CONNID ConnId, IN SPlayData * pstPlayData, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	VSTATUS Status;
	Boolean bIsSync;
	SHostSpeechData *pstUserData;
	U8 *pucTempBuffer;
	SVAPIDeviceConfig *pstDevConfig;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_StartPlayback", &pstChnl);

	if (Status != SUCCESS)
		goto err_out;

	UT_Log(APPITF, INFO, "Entered VAPI_StartPlayback: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_START_PLAYBACK_PARAMS;

	if (ePCI != pstPlayData->eMediaHost && eCSMENCAPS != pstPlayData->eMediaHost)
	{
		UT_ErrorLog(APPITF, "VAPI_StartPlayback: Invalid MediaHost %d\n", pstPlayData->eMediaHost);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	if (ePCM != pstPlayData->eDestinationType && eRTP != pstPlayData->eDestinationType && eBOTH != pstPlayData->eDestinationType)
	{
		UT_ErrorLog(APPITF, "VAPI_StartPlayback: Invalid Source type %d\n", pstPlayData->eDestinationType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	pstDevConfig = DMGR_GetVAPIDeviceConfig(pstChnl->pstDev->ucDevType);
	if (pstDevConfig == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_StartPlayback: Device type for this conn(%u) is invalid \n", ConnId);
		Status = VAPI_ERR_INVALID_DEV_TYPE;
		goto err_out;
	}

	if ((pstPlayData->eCodingScheme != eCS_G711_ULAW) &&
		(pstPlayData->eCodingScheme != eCS_G711_ALAW) &&
		(pstPlayData->eCodingScheme != eCS_G723_1) &&
		(pstPlayData->eCodingScheme != eCS_G729_A) &&
		(pstPlayData->eCodingScheme != eCS_G726_32) &&
		(pstPlayData->eCodingScheme != eCS_G726_16_audio) &&
		(pstPlayData->eCodingScheme != eCS_G726_24_audio) &&
		(pstPlayData->eCodingScheme != eCS_G726_40_audio) &&
		(pstPlayData->eCodingScheme != eCS_RawPcm))
	{
		UT_ErrorLog(APPITF, "VAPI_StartPlayback: Invalid coding scheme %d\n", pstPlayData->eCodingScheme);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	if (pstChnl->usConnType != eVOIP)
	{
		UT_ErrorLog(APPITF, "VAPI_StartPlayback: Invalid channel type %d\n", pstChnl->usConnType);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	pstVapiReq = VCORE_AllocateRequest(sizeof(SHostSpeechData));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	pstUserData = pstVapiReq->pvUserData;

	if (pstPlayData->pucBuffer)
	{
		pucTempBuffer = UT_AllocMem(sizeof(U8) * pstPlayData->uiBufferLength);

		if (pucTempBuffer == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto err_request;
		}

		UT_MemCopy(pucTempBuffer, pstPlayData->pucBuffer, pstPlayData->uiBufferLength);
	}
	else
		pucTempBuffer = NULL;

	pstUserData->ucMediaHost = pstPlayData->eMediaHost;
	pstUserData->ucSrcCoding = pstPlayData->eCodingScheme;
	pstUserData->ucDestinationType = pstPlayData->eDestinationType;
	pstUserData->ucSourceType = eHSF;
	pstUserData->pstSTimer = UT_TimerCreate();
	pstUserData->uiTimeOut = pstPlayData->uiTimeOut;
	pstUserData->ePlaybackMode = pstPlayData->ePlaybackMode;
	pstUserData->pucBuffer = pucTempBuffer;
	pstUserData->uiBufferLength = pstPlayData->uiBufferLength;
	UT_StrCpy((char *) pstUserData->strAnnFileName, pstPlayData->strAnnFileName);
	pstUserData->uiHostSpeechDataFormat = pstPlayData->uiHostSpeechDataFormat;
	
	if (ePCM == pstPlayData->eDestinationType)
	{
		if (pstChnl->pstChnlParams->stVoiceOpt.param_4.bits.compander == 0)
			pstUserData->ucDestCoding = pstDevConfig->astCodecInfo[eG711_ALAW_PCM].usPTIndex;
		else
			pstUserData->ucDestCoding = pstDevConfig->astCodecInfo[eG711_ULAW_PCM].usPTIndex;
	}
	else
		pstUserData->ucDestCoding =
		    pstDevConfig->astCodecInfo[pstChnl->pstChnlParams->eCodecVal].usPTIndex;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_StartPlayback,	/* pfnReqFSMHdlr */
				  ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_StartPlayback: Exiting status(%d), conn(%u)\n", Status, ConnId);

/* no error exit */
	return Status;

      err_request:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}

/***************************************************************************
 * VAPI_StartPlaying: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This one is now obsolete and replaced by VAPI_StartPlayback().
*	It is kept in this version for backward compatibility but may be removed in the next releases.
*
*	This API will start the playback from the buffer on the channel till the complete buffer 
*	is exhausted or timeout occurs. \n
*	The buffer should have data in Host Speech Frame format as given in CRM Section 
*	"Formatting of Recorded Speech Data".\n
*	VAPI_StopPlaying() / VAPI_StopPlayback() API can be called at any time to stop the ongoing playout.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which to start the playing.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eMediaHost</td>
*		<td style="vertical-align: top;">Supported values:
*				- 0 for PCI
*				- 1 for CSM_ENCAPS 
*				Not being used in current implementation.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eCodingScheme</td>
*		<td style="vertical-align: top;">Source coding schemes for Host Speech Frame 
*				for playing e.g. G.711u-law PCM audio, G.723 audio etc.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eDestinationType</td>
*		<td style="vertical-align: top;">Destination can be one of the following:
*					- 8-bit PCM
*					- RTP
*					- Both.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiTimeOut</td>
*		<td style="vertical-align: top;">Specifies the time (in Sec) for which playback needs to be done. \n
*					If its value is 0 ignore this parameter.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pucBuffer</td>
*		<td style="vertical-align: top;">Pointer to the Buffer from which play out is done.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiBufferLength</td>
*		<td style="vertical-align: top;">Length of the Buffer passed to the API.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include startplaying.c
*
*	\n \b Commands:
*	- SET_CHANNEL_MODE
*	- SYNCDAT
*	- SYNCEOF
* 	- VOIP_AS_SET_RATE 
*	- AS_STOP
*	- DRAIN_PLAYREC_BUFFER
*/
VSTATUS VAPI_StartPlaying(IN CONNID ConnId,
			  IN EMediaHostType eMediaHost,
			  IN ECodingScheme eCodingScheme,
			  IN EAnnounceType eDestinationType,
			  IN U32 uiTimeOut, IN U8 * pucBuffer, IN U32 uiBufferLength, IN SRequest * pstRequest)
{
	SPlayData stPlayData;

	stPlayData.eMediaHost = eMediaHost;
	stPlayData.eCodingScheme = eCodingScheme;
	stPlayData.eDestinationType = eDestinationType;
	stPlayData.ePlaybackMode = ePlaybackSpeedChangeable;
	stPlayData.uiTimeOut = uiTimeOut;
	stPlayData.pucBuffer = pucBuffer;
	stPlayData.uiBufferLength = uiBufferLength;
	stPlayData.uiHostSpeechDataFormat = eHedFormat;
	
	return VAPI_StartPlayback(ConnId, &stPlayData, pstRequest);
	
}

/***************************************************************************
 * VAPI_StartPlaybackFile: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API will start the playback from the file on the channel till the complete file 
*	is exhausted or timeout occurs. \n
*	The file should have data in Host Speech Frame format as given in CRM Section 
*	"Formatting of Recorded Speech Data".\n
*	VAPI_StopPlaying() / VAPI_StopPlayback() API can be called at any time to stop the ongoing playout.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which to start the playing.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstPlayData</td>
*		<td style="vertical-align: top;">Pointer to play data .</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Usage:
*	\include startplaybackfile.c
*
*	\n \b Commands:
*	- SET_FLOWCON
*	- SET_CHANNEL_MODE
*	- SYNCDAT
*	- SYNCEOF
* 	- VOIP_AS_SET_RATE 
*	- AS_STOP
*	- DRAIN_PLAYREC_BUFFER
*/
VSTATUS VAPI_StartPlaybackFile(IN CONNID ConnId, IN SPlayData * pstPlayData, IN SRequest * pstRequest)
{

	pstPlayData->pucBuffer = NULL;
	pstPlayData->uiBufferLength = 0;

	return VAPI_StartPlayback(ConnId, pstPlayData, pstRequest);
}

/***************************************************************************
 * VAPI_StopPlayback: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API will start any ongoing playback on this channel.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the playback has to be stopped.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">eStopAnnType</td>
*		<td style="vertical-align: top;">Stop type of announcement API.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include stopplayback.c
*/
VSTATUS VAPI_StopPlayback(IN CONNID ConnId, IN EStopAnnType eStopAnnType, IN SRequest * pstRequest)
{
	/*SVapiReq *pstVapiReq = NULL; */
	SChnl *pstChnl = NULL;
	SLstNode *pstNode;
	VSTATUS Status = SUCCESS;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_StopPlayback", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_StopPlayback: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_STOP_PLAYBACK_PARAMS;

	if (pstChnl->stRecPlayInfo.uiStopPlayCount < pstChnl->stRecPlayInfo.uiStartPlayCount)
	{
		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.eStopAnnType = eStopAnnType;
		pstChnl->stRecPlayInfo.uiStopPlayCount++;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);

		/*Get Front node */
		UT_MutexLock(&pstChnl->stReqList.stSem);
		
		pstNode = __UT_GetFrontNode(&pstChnl->stReqList);

		if (pstNode == NULL)
		{
			UT_ErrorLog(VCORE, "VAPI_StopPlayback: Request list empty\n");
			UT_MutexUnLock(&pstChnl->stReqList.stSem);
			goto out;
		}
		UT_MutexUnLock(&pstChnl->stReqList.stSem);

		VCORE_ProcessFSMhandler(pstChnl, FSM_EVENT_PLAY_REC_STOP, NULL);
		
		Status = SUCCESS;
	}
	else
	{
		UT_ErrorLog(APPITF, "VAPI_StopPlayback: No Start_Playing called w.r.t. to stop playing\n");
		Status = VAPI_ERR_STOP_PLAY_EXIST;
	}

	UT_Log(APPITF, INFO, "VAPI_StopPlayback: Exiting status(%d), conn(%u)\n", Status, ConnId);

out:
	/* if ASYNC mode call the callback function*/
	if ((pstRequest != NULL) && (pstRequest->pfnIoCompCallback != NULL))
	{
		pstRequest->pfnIoCompCallback(ConnId, CMD_LEVEL_CONN, Status, NULL, 0, pstRequest->uiReqId);
		Status = VAPI_ERR_PENDING;
	}

	return Status;

}

/***************************************************************************
 * VAPI_StopPlaying: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This one is now obsolete and replaced by VAPI_StopPlayback().
*	It is kept in this version for backward compatibility but may be removed in the next releases.
*
*	This API will start any ongoing playback on this channel.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the playback has to be stopped.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include stopplaying.c
*/
VSTATUS VAPI_StopPlaying(IN CONNID ConnId, IN SRequest * pstRequest)
{
	return VAPI_StopPlayback(ConnId, eAS_STOP, pstRequest);
}

/***************************************************************************
 * VAPI_PlaybackSetRate: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	VAPI_PlaybackSetRate() makes the playback speed faster or slower.\n
*	For example, a message that takes 20 seconds, can be played in 10 seconds if the API is called with ePlaybackRate_2_1.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">The identifier of connection that is to be destroyedr</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ePlaybackRate</td>
*		<td style="vertical-align: top;">Enumeration of supported fast-slow rates</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*
*	\n \b Usage:
*	\include playback_set_rate.c
*
*/
VSTATUS VAPI_PlaybackSetRate(IN CONNID ConnId, EPlaybackRate ePlaybackRate, IN SRequest * pstRequest)
{
	/*SVapiReq *pstVapiReq = NULL; */
	SChnl *pstChnl = NULL;
	VSTATUS Status = SUCCESS;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, " VAPI_PlaybackSetRate", &pstChnl);

	if (Status != SUCCESS)
		goto out;

	UT_Log(APPITF, INFO, "Entered VAPI_PlaybackSetRate: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_PLAYBACK_SET_RATE_PARAMS;

	if (pstChnl->stRecPlayInfo.uiStartPlayCount > 0)
	{
		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.uiChangeRate = True;
		pstChnl->stRecPlayInfo.ePlaybackRate = ePlaybackRate;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		Status = SUCCESS;
	}
	else
	{
		UT_ErrorLog(APPITF, " VAPI_PlaybackSetRate: Start_Playing in not called\n");
		Status = VAPI_ERR_STOP_PLAY_EXIST;
	}

	UT_Log(APPITF, INFO, " VAPI_PlaybackSetRate: Exiting status(%d), conn(%u)\n", Status, ConnId);

out:
	/* if ASYNC mode call the callback function*/
	if ((pstRequest != NULL) && (pstRequest->pfnIoCompCallback != NULL))
	{
		pstRequest->pfnIoCompCallback(ConnId, CMD_LEVEL_CONN, Status, NULL, 0, pstRequest->uiReqId);
		Status = VAPI_ERR_PENDING;
	}

	return Status;
}

/***************************************************************************
 * VAPI_ ConvertHostSpeechData: The function does the following things -
 ***************************************************************************/
/*! 
*	\n \b Description: \n
*	This API will convert the Host speech frames in the buffer to ucNewCoding scheme.\n
*
*	<b>Note: not tested yet.</b>\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">ConnId</td>
*		<td style="vertical-align: top;">Connection on which the speech frames have to be converted.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstConvertData</td>
*		<td style="vertical-align: top;">Pointer to struct with host speech data to be processed.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstRequest</td>
*		<td style="vertical-align: top;">If NULL then the call is in blocking mode (synchronous).</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*	\li SUCCESS or VAPI_ERR_PENDING
*	\li VAPI_ERR_LIB_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_NOMEM
*	\li VAPI_ERR_INVALID_PARAM
*
*	\n \b Commands:
*	- SET_CHANNEL_MODE
*	- SYNCDAT
*	- SYNCEOF
*	- SET_FLOWCON
*
*	\n \b Usage:
*	\include convert_host_speech.c
*/
VSTATUS VAPI_ConvertHostSpeechData(IN CONNID ConnId, IN SConvertData *pstConvertData, IN SRequest * pstRequest)
{
	SVapiReq *pstVapiReq = NULL;
	SChnl *pstChnl = NULL;
	U8 *pucTempBuffer = NULL;
	SHostSpeechData *pstUserData;
	VSTATUS Status;
	Boolean bIsSync;
	SVAPIDeviceConfig *pstDevConfig;

	/* proceed to some basic verification to check if VAPI, the Device, the connection id
	   are in the rigth state and valid */
	Status = VCORE_ConnectionGenericCheck(ConnId, "VAPI_ConvertHostSpeechData", &pstChnl);

	if (Status != SUCCESS)
		goto err_out;

	UT_Log(APPITF, INFO, "Entered VAPI_ConvertHostSpeechData: conn(%u)\n", ConnId);

	/* log the parameters passed if required (define in appitf.h) */
	LOG_CONVERT_HOST_SPEECH_DATA_PARAMS;

	pstDevConfig = DMGR_GetVAPIDeviceConfig(pstChnl->pstDev->ucDevType);
	if (pstDevConfig == NULL)
	{
		UT_ErrorLog(APPITF, "VAPI_ConvertHostSpeechData: Device type for this conn(%u) is invalid \n", ConnId);
		Status = VAPI_ERR_INVALID_DEV_TYPE;
		goto err_out;
	}

	if (pstConvertData->ucCurrentCoding >= NUM_CODECS)
	{
		UT_ErrorLog(APPITF, "VAPI_ConvertHostSpeechData: Invalid coding scheme %d\n", pstConvertData->ucCurrentCoding);
		Status = VAPI_ERR_INVALID_PARAM;
		goto err_out;
	}

	pstVapiReq = VCORE_AllocateRequest(sizeof(SHostSpeechData));

	if (pstVapiReq == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_out;
	}

	pstUserData = pstVapiReq->pvUserData;

	pucTempBuffer = UT_AllocMem(sizeof(U8) * pstConvertData->uiSrcBufferLength);

	if (pucTempBuffer == NULL)
	{
		Status = VAPI_ERR_NOMEM;
		goto err_request;
	}

	UT_MemCopy(pucTempBuffer, pstConvertData->pucSrcBuffer, pstConvertData->uiSrcBufferLength);

	pstUserData->ucMediaHost = eCSMENCAPS;
	pstUserData->ucSrcCoding = pstDevConfig->astCodecInfo[(U32) pstConvertData->ucCurrentCoding].usPTIndex;
	/* freed in VFSM_ConvertHostSpeechData */
	pstUserData->pucBuffer = pucTempBuffer;
	pstUserData->uiBufferLength = pstConvertData->uiSrcBufferLength;
	pstUserData->pucOutBuffer = pstConvertData->pucOutBuffer;
	pstUserData->uiOutBufferLength = pstConvertData->uiOutBufferLength;
	pstUserData->ucSourceType = eHSF;
	pstUserData->ucDestinationType = eHSF;
	pstUserData->ucDestCoding = pstDevConfig->astCodecInfo[(U32) pstConvertData->ucNewCoding].usPTIndex;

	/* Fill the request with the regular parameters */
	VCORE_SetConnectionRequest(pstVapiReq, pstRequest, VFSM_ConvertHostSpeechData,	/* pfnReqFSMHdlr */
				  ConnId);	/*uiID */

	bIsSync = pstVapiReq->bIsSync;

	Status = VCORE_ProcessRequest(pstChnl, pstVapiReq);

	/* In async mode everything is freed in DoReqCompletion */
	if (bIsSync)
	{
		if (pstVapiReq->pvCallbackData != NULL)
			UT_FreeMem(pstVapiReq->pvCallbackData);

		VCORE_FreeRequest(pstVapiReq);
	}

	UT_Log(APPITF, INFO, "VAPI_ConvertHostSpeechData: Exiting status(%d), conn(%u)\n", Status, ConnId);

/* no error exit */
	return Status;

      err_request:
	VCORE_FreeRequest(pstVapiReq);
      err_out:
	return Status;
}

/** @} */ /*announcement*/
