/*! \file announcement_fsm.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"
#include "vapi.h"
#include "dmgr.h"
#include "vcore.h"
#include "msp.h"
#include "vcore_voip.h"
#include "appitf.h"
#include "comcerto-flow-control-types.h"

static void* VCORE_GetFrame(SHostSpeechData *pstHostSpeechData, int offset, int size)
{
	VSTATUS Status = SUCCESS;
	void *buf = NULL;

	UT_Log(VCORE, DEBUG, "Entered VCORE_GetFrame offset = (%d), size = (%d)\n", offset, size);

	if (pstHostSpeechData == NULL)
	{
		Status = VAPI_ERR_INVALID_PARAM;
		return NULL;
	}	

	if (!pstHostSpeechData->pucBuffer) 
	{
		buf = UT_AllocMem(size);
		if (buf == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			return NULL;
		}

		if (pstHostSpeechData->iFileDisc == -1)
			pstHostSpeechData->iFileDisc = UT_Open(pstHostSpeechData->strAnnFileName, O_RDONLY, 0444);

		if (pstHostSpeechData->iFileDisc < 0)
		{
			UT_ErrorLog(VCORE, "VCORE_GetFrame: announcement file could not be opened %s\n", pstHostSpeechData->strAnnFileName);
			Status = VAPI_ERR_INVALID_PARAM;
			goto err;
		}

		if ((pstHostSpeechData->uiBufferLength = UT_Lseek(pstHostSpeechData->iFileDisc, 0L, SEEK_END)) < 0)
		{
			UT_ErrorLog(VCORE, "VCORE_GetFrame: file length invalid\n");
			Status = VAPI_ERR_INVALID_PARAM;
			goto err;
		}

		if (pstHostSpeechData->uiBufferLength < offset + size)
		{
			UT_ErrorLog(VCORE, "VCORE_GetFrame: announcement file too small");
			Status = VAPI_ERR_INVALID_IMAGE;
			goto err;
		}

		if ((UT_Lseek(pstHostSpeechData->iFileDisc, offset, SEEK_SET)) != offset)
		{
			UT_ErrorLog(VCORE,"VCORE_GetFrame: pointer could not be reset\n");
			Status = VAPI_ERR_INVALID_PARAM;
			goto err;
		}		

		if (UT_Read(pstHostSpeechData->iFileDisc, (char *) buf, size) != size)
		{
			UT_ErrorLog(VCORE, "VCORE_GetFrame: error in reading announcement file\n");
			Status = VAPI_ERR_INVALID_PARAM;
			goto err;
		}

		UT_Log(VCORE, DEBUG,  "VCORE_GetFrame: Exiting status(%d)\n", Status);
		return buf;

err:
		UT_FreeMem(buf);
		UT_ErrorLog(VCORE,"VCORE_GetFrame: Exiting status(%d)\n", Status);
		return NULL;

	}
	else 
	{
		UT_Log(VCORE, DEBUG,  "VCORE_GetFrame: Exiting status(%d)\n", Status);
     		return pstHostSpeechData->pucBuffer + offset;	
	}	
}

static int VCORE_PutFrame(SHostSpeechData *pstHostSpeechData, void *buf, int offset, int size)
{
	VSTATUS Status = SUCCESS;

	UT_Log(VCORE, DEBUG, "Entered VCORE_PutFrame offset = (%d), size = (%d)\n", offset, size);

	if (pstHostSpeechData == NULL)
	{
		Status = VAPI_ERR_INVALID_PARAM;
		return Status;
	}	

	if (!pstHostSpeechData->pucBuffer)
	{
		if (pstHostSpeechData->iFileDisc == -1)
			pstHostSpeechData->iFileDisc = UT_Open(pstHostSpeechData->strAnnFileName, O_WRONLY | O_CREAT, 0466);

		if (pstHostSpeechData->iFileDisc < 0)
		{
			UT_ErrorLog(VCORE, "VCORE_PutFrame: announcement file could not be opened %s\n", pstHostSpeechData->strAnnFileName);
			Status = VAPI_ERR_INVALID_PARAM;
			goto out;
		}

		if (UT_Lseek(pstHostSpeechData->iFileDisc, offset, SEEK_SET) != offset)
		{
			UT_ErrorLog(VCORE,"VCORE_PutFrame: pointer could not be reset\n");
			Status = VAPI_ERR_INVALID_PARAM;
			goto out;
		}		

		if (UT_Write(pstHostSpeechData->iFileDisc, (char *) buf, size) != size)
		{
			UT_ErrorLog(VCORE, "VCORE_PutFrame: error in writing announcement file\n");
			Status = VAPI_ERR_INVALID_PARAM;
			goto out;
		}

out:
		UT_Log(VCORE, DEBUG,  "VCORE_PutFrame: Exiting status(%d)\n", Status);
		return Status;
	}
	else 
	{
		UT_Log(VCORE, DEBUG,  "VCORE_PutFrame: Exiting status(%d)\n", Status);
     		UT_MemCopy ((void *) (pstHostSpeechData->pucBuffer + offset), buf, size);
		return Status;
	}	
}

static void VCORE_CleanFrame(SHostSpeechData *pstHostSpeechData, void *buf)
{
	UT_Log(VCORE, DEBUG, "Entered VCORE_CleanFrame\n");
	if (!pstHostSpeechData->pucBuffer)
	{
		if (NULL != buf)
			UT_FreeMem(buf);
	}

	UT_Log(VCORE, DEBUG, "VCORE_CleanFrame: Exit\n");
}

VSTATUS VFSM_RecDataTransfer(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SFrameInfo *pstFrameData;
	SHostSpeechData *pstHostSpeechData;
	U16 usActFrameLen = 0;
	U8 *pstAllocatedPool;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_RecDataTransfer: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstFrameData = (SFrameInfo *) pstVapiReq->pvExecData;
	pstHostSpeechData = (SHostSpeechData *) pstVapiReq->pvUserData;

	if (pstChnl->stRecPlayInfo.uiStopRecCount > 0)
	{
		if (pstChnl->stRecPlayInfo.eRecPlayState == eRec)
		{
			/* stop timer, proceed to comlete recording*/
			UT_TimerStop(pstHostSpeechData->pstSTimer);
		}
	}

	/* timeout could occur or stop record could be called, proceed to comlete recording*/
	if (Event == FSM_EVENT_PLAY_REC_STOP || Event == FSM_EVENT_REC_PLAY_TERMINATOR)
			pstHostSpeechData->uiProceedToStop = True;		

	switch (pstVapiReq->usReqState)
	{
	case REC_DATA_TRANSFER_INIT:
		if (pstHostSpeechData->uiTimeOut != 0)
		{	
			/* start timer with defined playback time*/
			UT_TimerStart(pstHostSpeechData->pstSTimer, pstHostSpeechData->uiTimeOut, 
					VCORE_RecPlayTerminator, (void *)pstChnl);
		}

		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.uiStartRecCount++;
		pstChnl->stRecPlayInfo.eStopAnnType = eDRAIN;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);

		pstHostSpeechData->iFileDisc = -1;

		pstVapiReq->pvExecData = UT_AllocMem(sizeof(SFrameInfo));

		if (pstVapiReq->pvExecData == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		pstFrameData = (SFrameInfo *) pstVapiReq->pvExecData;

		pstFrameData->uiFrameLen = 0;
		pstFrameData->ucIsHdrPresent = True;
		pstFrameData->pucFrameBuffer = NULL;
		pstFrameData->uiBufferSizeLeft = pstHostSpeechData->uiBufferLength;

		if ((pstAllocatedPool = (U8 *) UT_AllocMem(MAX_FRAME_SIZE)) == NULL)
		{
			UT_ErrorLog(VCORE, "VFSM_RecDataTransfer: No Memory available\n");
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		UT_MemSet(pstAllocatedPool, 0, MAX_FRAME_SIZE);
		pstFrameData->pucFrameBuffer = pstAllocatedPool;
		pstVapiReq->usReqState = REC_DATA_TRANSFER_PROCEED;

		break;

	case REC_DATA_TRANSFER_PROCEED:		
		/* wait for SYNCDAT/SYNCEOF indications */
		if (pstMsg != NULL && pstMsg->fifo != NULL && FC_SYNCDAT == UT_LE2CPU16(*((U16 *) (pstMsg->fifo) + 2)))
		{
			pstHostSpeechData->uiReceivedIndType = FC_SYNCDAT;
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_INDICATION, FC_SYNCDAT);		
		}
		else if (pstMsg != NULL && pstMsg->fifo != NULL && FC_SYNCEOF == UT_LE2CPU16(*((U16 *) (pstMsg->fifo) + 2)))
		{
			pstHostSpeechData->uiReceivedIndType = FC_SYNCEOF;
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_INDICATION, FC_SYNCEOF);
		}
		else
		{
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					CMD_CLASS_CONF_CHANNEL, CMD_TYPE_INDICATION, FC_VOIP_DRAIN_PLAYREC_BUFFER_IND);

			/* got DRAIN indication, complete request*/
			if ((Status == SUCCESS) && (Event != FSM_EVENT_PLAY_REC_STOP ) && (Event != FSM_EVENT_REC_PLAY_TERMINATOR))
				goto finish;
		}

		/* timeout could occur or stop record could be called*/
		if (Event == FSM_EVENT_PLAY_REC_STOP || Event == FSM_EVENT_REC_PLAY_TERMINATOR)
			break;

		/* timeout could occur or stop record could be called*/
		if (pstHostSpeechData->uiProceedToStop == True)
		{
			UT_Log(VCORE, INFO, "VFSM_RecDataTransfer: got FSM_EVENT_PLAY_REC_STOP/FSM_EVENT_REC_PLAY_TERMINATOR\n");

			pstHostSpeechData->eStopAnnType = pstChnl->stRecPlayInfo.eStopAnnType;

			if (eAS_STOP == pstHostSpeechData->eStopAnnType)
			{
				pstVapiReq->usReqState = REC_DATA_TRANSFER_FINISH;
				Status = VDEV_SetAs_Stop(pstChnl);

				/* we was already in proceeding of stop procedure by DRAIN type, 
				change this type to AS_STOP*/
				if (pstHostSpeechData->uiDrainStopType == True)
					pstHostSpeechData->uiDrainToAsStop = True;
			}
			else
			{
				/* We're waiting DRAIN indication thus there is no sense 
				of sending DRAIN again, but only AS_STOP.*/
				if (pstHostSpeechData->uiDrainStopType != True)
				{
					pstVapiReq->usReqState = REC_DATA_TRANSFER_COMPLETE_REMAIN;
					/* after received responce, don't forget send ACK indication*/
					Status = VDEV_SetDrainPlayRecBuffer(pstChnl);
				}
			}

			if (Status != SUCCESS)
				goto finish;

			break;
			
		}
		else
		{
			if (Status != SUCCESS)
				goto finish;

			pstAllocatedPool = pstFrameData->pucFrameBuffer;
			Status = VCORE_HandleFrame(pstMsg, pstFrameData);

			if (Status == MALFORMED_PACKET)
				break;

			if (SYNCEOF_PACKET != Status && SYNCDAT_PACKET != Status)
			{
				Status = VAPI_ERR_INVALID_FRAME;
				goto finish;
			}

			if (Status == SYNCEOF_PACKET)
			{
				Status = SUCCESS;
				
				if (NULL != pstHostSpeechData->pfnFrameHndlr)
				{
					pstHostSpeechData->pfnFrameHndlr(pstVapiReq->uiID,
								pstVapiReq->ucCmdLevel,
								Status,
								(void *)(pstFrameData->pucFrameBuffer),
								pstFrameData->uiFrameLen, 
								pstVapiReq->uiReqId);
				}
				else
				{
					if (pstFrameData->uiBufferSizeLeft == 0)
					{
						UT_TimerStop(pstHostSpeechData->pstSTimer);
						pstFrameData->pucFrameBuffer = pstAllocatedPool;

						/* we must set only AS_STOP type, since there is no left size to write*/
						pstHostSpeechData->eStopAnnType = eAS_STOP;

						pstVapiReq->usReqState = REC_DATA_TRANSFER_FINISH;
						Status = VDEV_SetAs_Stop(pstChnl);

						/* we was already in proceeding of stop procedure by DRAIN type, 
						change this type to AS_STOP*/
						if (pstHostSpeechData->uiDrainStopType == True)
							pstHostSpeechData->uiDrainToAsStop = True;							
							
						if (Status != SUCCESS)
							goto finish;

						break;
					}
					else
					{
						usActFrameLen = pstFrameData->uiFrameLen;
						usActFrameLen = ALIGN_MESSAGE_LENGTH(usActFrameLen);
					
						if (pstFrameData->uiBufferSizeLeft > usActFrameLen)
						{
							*((U16 *) pstFrameData->pucFrameBuffer + OFFSET_FOR_BUF_LEN) = 
								UT_CPU2LE16(usActFrameLen - FRAME_HEADER_LEN);

							if (eNoHedFormat == pstHostSpeechData->uiHostSpeechDataFormat)
							{
								UT_MemCopy(pstFrameData->pucFrameBuffer, 
										pstFrameData->pucFrameBuffer + FRAME_HEADER_LEN, 
										pstFrameData->uiFrameLen - FRAME_HEADER_LEN);

								pstFrameData->uiFrameLen -= FRAME_HEADER_LEN;
								usActFrameLen -= FRAME_HEADER_LEN;
							}

							Status = VCORE_PutFrame(pstHostSpeechData, 
									(void *) pstFrameData->pucFrameBuffer, 
									pstHostSpeechData->uiBufferLength - pstFrameData->uiBufferSizeLeft,
									pstFrameData->uiFrameLen);

							if (Status != SUCCESS)
								goto finish;
							
							/*This function should write the complete frame into the buffer */
							pstFrameData->uiBufferSizeLeft -= usActFrameLen;
						}
						else
							pstFrameData->uiBufferSizeLeft = 0;

					}
				}

			pstFrameData->uiFrameLen = 0;
			pstFrameData->ucIsHdrPresent = True;

			UT_MemSet(pstAllocatedPool, 0, MAX_FRAME_SIZE);
			pstFrameData->pucFrameBuffer = pstAllocatedPool;
			}
		}	

		/* ACK indication*/
		if (pstMsg != NULL && pstMsg->fifo != NULL && FC_SYNCDAT == UT_LE2CPU16(*((U16 *) (pstMsg->fifo) + 2)))
			Status = VDEV_SendDataInd(pstChnl, FC_SYNCDAT);
		else
			Status = VDEV_SendDataInd(pstChnl, FC_SYNCEOF);

		if (Status != SUCCESS)
			goto finish;		

		break;

	case REC_DATA_TRANSFER_COMPLETE_REMAIN:
		/* timeout could occur or stop record could be called*/
		if (Event == FSM_EVENT_PLAY_REC_STOP || Event == FSM_EVENT_REC_PLAY_TERMINATOR)
			break;

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_DRAIN_PLAYREC_BUFFER);

		/*continue to rec if DRAIN stop type*/
		pstVapiReq->usReqState = REC_DATA_TRANSFER_PROCEED;

		/* We got response on DRAIN, we have to ACK previous SYNCDAT/SYNCEOF indication*/
		if (pstHostSpeechData->uiReceivedIndType == FC_SYNCDAT)
			Status = VDEV_SendDataInd(pstChnl, FC_SYNCDAT);
		else
			Status = VDEV_SendDataInd(pstChnl, FC_SYNCEOF);			

		if (Status != SUCCESS)
			goto finish;

		/* record accumulated speech data on device, till DRAIN indication receive*/
		pstHostSpeechData->uiProceedToStop = False;

		/* set that we proceed stop by DRAIN type*/
		pstHostSpeechData->uiDrainStopType = True;

		break;

	case REC_DATA_TRANSFER_FINISH:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_AS_STOP);

		/* got responce for AS_STOP, it's ramained to wait for DRAIN inidication*/
		if (pstHostSpeechData->uiDrainToAsStop == True)
		{
			pstHostSpeechData->uiDrainToAsStop = False;
			pstVapiReq->usReqState = REC_DATA_TRANSFER_PROCEED;
			break;
		}

		goto finish;		

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_RecDataTransfer: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	
	}	

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_RecDataTransfer: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_RecDataTransfer: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	UT_FreeMem(pstFrameData->pucFrameBuffer);

	if (pstHostSpeechData->iFileDisc >= 0)
		UT_CloseFd(pstHostSpeechData->iFileDisc);

	if (Status == SUCCESS)
	{
		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		if (pstChnl->stRecPlayInfo.uiStopRecCount)
			pstChnl->stRecPlayInfo.uiStopRecCount--;

		if (pstChnl->stRecPlayInfo.uiStartRecCount)
			pstChnl->stRecPlayInfo.uiStartRecCount--;

		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
	}	
	
	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_RecDataTransfer: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

VSTATUS VFSM_PlayDataTransfer(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SVapiReq * pstChildVapiReq;
	SFrameInfo *pstFrameData;
	U8 *pucFrameBufferWithHeader;
	U8 uiNumberOfFrames = 0;
	SHostSpeechData *pstHostSpeechData;
	U16 uiMaxFrameLen;
	U32 uiFrameLenConst = 0;
	U32 uiFrameTimeStamp = 0;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_PlayDataTransfer: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstFrameData = (SFrameInfo *) pstVapiReq->pvExecData;
	pstHostSpeechData = (SHostSpeechData *) pstVapiReq->pvUserData;


	if (pstChnl->stRecPlayInfo.uiStopPlayCount > 0)
	{
		if (pstChnl->stRecPlayInfo.eRecPlayState == ePlay 
			|| pstChnl->stRecPlayInfo.eRecPlayState == ePlayExt
			|| pstChnl->stRecPlayInfo.eRecPlayState == ePlaySpeedChangeable)
		{
			/* stop timer, proceed to comlete playing*/
			UT_TimerStop(pstHostSpeechData->pstSTimer);
		}
	}

	/* timeout could occur or stop playback could be called, proceed to comlete playing*/
	if (Event == FSM_EVENT_PLAY_REC_STOP || Event == FSM_EVENT_REC_PLAY_TERMINATOR)
			pstHostSpeechData->uiProceedToStop = True;	

	/* played data doesn't contain headers for each frame */
	if (pstHostSpeechData->uiHostSpeechDataFormat == eNoHedFormat)
	{
		switch (pstHostSpeechData->ucDestCoding)
		{
		case eCS_G711_ULAW:
		case eCS_G711_ALAW:
			uiFrameLenConst = 160;
			uiFrameTimeStamp = 160;
			break;

		case eCS_G723_1:
			uiFrameLenConst = 20;
			uiFrameTimeStamp = 240;
			break;
			
		case eCS_G729_A:
			uiFrameLenConst = 10;
			uiFrameTimeStamp = 80;
			break;

		case eCS_G726_32:
			uiFrameLenConst = 40;
			uiFrameTimeStamp = 80;
			break;

		case eCS_G726_16_audio:
			uiFrameLenConst = 20;
			uiFrameTimeStamp = 80;
			break;

		case eCS_G726_24_audio:
			uiFrameLenConst = 30;
			uiFrameTimeStamp = 80;
			break;

		case eCS_G726_40_audio:
			uiFrameLenConst = 50;
			uiFrameTimeStamp = 80;
			break;

		case eCS_RawPcm:
			uiFrameLenConst = 320;
			uiFrameTimeStamp = 160;
			break;			

		}

	}

	/* set the max supported frame lenght*/
	if (pstHostSpeechData->ePlaybackMode == ePlaybackExt)
		uiMaxFrameLen = MAX_FRAME_LEN_EXT_SYNCEOF;
	else
		uiMaxFrameLen = MAX_FRAME_LEN_SYNCEOF;

	switch (pstVapiReq->usReqState)
	{
	case PLAY_DATA_TRANSFER_INIT:
		if (pstHostSpeechData->uiTimeOut != 0)
		{
			/* start timer with defined playback time*/
			UT_TimerStart(pstHostSpeechData->pstSTimer, pstHostSpeechData->uiTimeOut, 
					VCORE_RecPlayTerminator, (void *) pstChnl);		
		}

		pstHostSpeechData->iFileDisc = -1;

		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.uiStartPlayCount++;
		/* internal default stop type in case of timeout or file end*/
		pstChnl->stRecPlayInfo.eStopAnnType = eDRAIN;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);

		pstVapiReq->pvExecData = UT_Calloc(1, sizeof(SFrameInfo));

		if (pstVapiReq->pvExecData == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		pstFrameData = (SFrameInfo *) pstVapiReq->pvExecData;

		/* just to initialize uiBufferLength in case of file*/
		if(!pstHostSpeechData->pucBuffer)
		{
			pstFrameData->pucFrameBuffer = VCORE_GetFrame(pstHostSpeechData, 0, FRAME_HEADER_LEN);
			VCORE_CleanFrame(pstHostSpeechData, pstFrameData->pucFrameBuffer);
		}

		pstFrameData->uiBufferSizeLeft = pstHostSpeechData->uiBufferLength;
		pstFrameData->ucIsHdrPresent = True;

		/* form frame length*/
		if (eHedFormat == pstHostSpeechData->uiHostSpeechDataFormat)
		{
			pstFrameData->pucFrameBuffer = VCORE_GetFrame(pstHostSpeechData, 
							pstHostSpeechData->uiBufferLength - pstFrameData->uiBufferSizeLeft, 
							FRAME_HEADER_LEN);

			if (pstFrameData->pucFrameBuffer == NULL)
				goto finish;

			pstFrameData->uiFrameLen = UT_LE2CPU16(*((U16 *) pstFrameData->pucFrameBuffer + OFFSET_FOR_BUF_LEN));

			uiNumberOfFrames = 1;
		}
		else
		{
			uiNumberOfFrames = 0;
			pstFrameData->uiFrameLen = 0;

			while (pstFrameData->uiBufferSizeLeft > (pstFrameData->uiFrameLen + uiFrameLenConst) 
					&& (pstFrameData->uiFrameLen + uiFrameLenConst + FRAME_HEADER_LEN) < uiMaxFrameLen)
			{
				pstFrameData->uiFrameLen = pstFrameData->uiFrameLen + uiFrameLenConst;
				uiNumberOfFrames++;
			}
		}

		/* check frame lenght to be sent with left buffer size*/
		if (pstFrameData->uiBufferSizeLeft < pstFrameData->uiFrameLen || uiNumberOfFrames == 0)
		{
			VCORE_CleanFrame(pstHostSpeechData, pstFrameData->pucFrameBuffer);
			pstFrameData->uiBufferSizeLeft = 0;
			/* since thre is nothing to send at all, stop timer, comlete transmitting */
			UT_TimerStop(pstHostSpeechData->pstSTimer);

			goto finish;
		}

		pstFrameData->uiFrameLen += FRAME_HEADER_LEN;

		/* allocate frame buffer to be sent*/
		pucFrameBufferWithHeader = UT_Calloc(1, pstFrameData->uiFrameLen);

		if (pucFrameBufferWithHeader == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}		

		/* fill frame buffer to be sent*/
		if (eHedFormat == pstHostSpeechData->uiHostSpeechDataFormat)
		{
			VCORE_CleanFrame(pstHostSpeechData, pstFrameData->pucFrameBuffer);

			pstFrameData->pucFrameBuffer = VCORE_GetFrame(pstHostSpeechData, 
							pstHostSpeechData->uiBufferLength - pstFrameData->uiBufferSizeLeft, 
							pstFrameData->uiFrameLen);

			if (pstFrameData->pucFrameBuffer == NULL)
				goto finish;

			UT_MemCopy(pucFrameBufferWithHeader, pstFrameData->pucFrameBuffer, pstFrameData->uiFrameLen);
		}
		else
		{
			pstFrameData->pucFrameBuffer = VCORE_GetFrame(pstHostSpeechData, 
							pstHostSpeechData->uiBufferLength - pstFrameData->uiBufferSizeLeft, 
							uiFrameLenConst * uiNumberOfFrames);

			if (pstFrameData->pucFrameBuffer == NULL)
				goto finish;
			
			*((U32 *) pucFrameBufferWithHeader + OFFSET_FOR_TIMESTAMP) = UT_CPU2LE32(pstFrameData->uiFrameTimeStamp);
			*((U16 *) pucFrameBufferWithHeader + OFFSET_FOR_FRAME_TYPE) = UT_CPU2LE16(1);
			*((U16 *) pucFrameBufferWithHeader + OFFSET_FOR_BUF_LEN) = UT_CPU2LE16(uiFrameLenConst * uiNumberOfFrames);

			UT_MemCopy(pucFrameBufferWithHeader + FRAME_HEADER_LEN, pstFrameData->pucFrameBuffer, uiNumberOfFrames * uiFrameLenConst);

			pstFrameData->uiFrameTimeStamp += uiNumberOfFrames * uiFrameTimeStamp;
		}

		VCORE_CleanFrame(pstHostSpeechData, pstFrameData->pucFrameBuffer);
		pstFrameData->pucFrameBufferStart = pucFrameBufferWithHeader;
		pstFrameData->pucFrameBuffer = pstFrameData->pucFrameBufferStart;

		if (eHedFormat == pstHostSpeechData->uiHostSpeechDataFormat)
			pstFrameData->uiBufferSizeLeft -= pstFrameData->uiFrameLen;
		else
			pstFrameData->uiBufferSizeLeft -= (pstFrameData->uiFrameLen - FRAME_HEADER_LEN);

		pstVapiReq->usReqState = PLAY_DATA_TRANSFER_PROCEED;

		/* send packet with using SYNCDAT/SYNCEOF command depending on frame length */
		if (pstFrameData->uiFrameLen > uiMaxFrameLen)
		{
			/* use SYNCDAT command to transmitt packets with length more then  umax frame length*/
			Status = VDEV_SendData(pstChnl, pstFrameData, FC_SYNCDAT);
			pstFrameData->pucFrameBuffer = pstFrameData->pucFrameBuffer + uiMaxFrameLen;

			pstFrameData->uiFrameLen -= uiMaxFrameLen;
			pstFrameData->ucIsHdrPresent = False;
		}
		else
		{
			/* use SYNCEOF command to transmitt packets with length less then max frame length*/
			if (pstHostSpeechData->ePlaybackMode == ePlaybackExt)
				Status = VDEV_SendDataExtended(pstChnl, pstFrameData);
			else
				Status = VDEV_SendData(pstChnl, pstFrameData, FC_SYNCEOF);

			if (NULL != pstFrameData->pucFrameBufferStart)
				UT_FreeMem(pstFrameData->pucFrameBufferStart);
			
			pstFrameData->uiFrameLen = 0;
			pstFrameData->pucFrameBuffer = NULL;
			pstFrameData->ucIsHdrPresent = True;
		}

		if (Status != SUCCESS)
			goto finish;		
		
		break;

	case PLAY_DATA_TRANSFER_PROCEED:
		/*check the response status of SYNCDAT/SYNCEOF command */
		if (pstFrameData->uiFrameLen > 0)			
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_SYNCDAT);
		else
			Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
						   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_SYNCEOF);

		/* timeout could occur or stop playback could be called*/
		if (Event == FSM_EVENT_PLAY_REC_STOP || Event == FSM_EVENT_REC_PLAY_TERMINATOR)
			break;

		if (Status != SUCCESS)
			goto finish;

		/* proceed to stop*/
		if (pstHostSpeechData->uiProceedToStop == True)
		{
			pstHostSpeechData->eStopAnnType = pstChnl->stRecPlayInfo.eStopAnnType;

			/* send MSP command with appropriate stop type*/
			if (pstHostSpeechData->eStopAnnType == eAS_STOP)
			{
				pstVapiReq->usReqState = PLAY_DATA_TRANSFER_FINISH;
				Status = VDEV_SetAs_Stop(pstChnl);
			}	
			else
			{
				pstVapiReq->usReqState = PLAY_DATA_TRANSFER_COMPLETE_REMAIN;
				Status = VDEV_SetDrainPlayRecBuffer(pstChnl);
			}

			if (Status != SUCCESS)
				goto finish;			
			
			break;
		}

		/* check wheather VAPI_PlaybackSetRate had been called*/
		if (pstFrameData->uiFrameLen == 0 && pstChnl->stRecPlayInfo.uiChangeRate == True)
		{			
			pstChildVapiReq  = VCORE_AllocateRequest(sizeof(EPlaybackRate));
			if (!pstChildVapiReq)
			{
				Status = VAPI_ERR_NOMEM;
				goto finish;
			}

			/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
			pstChildVapiReq->pvUserData = (void *) &pstChnl->stRecPlayInfo.ePlaybackRate;			
					
			/* initialise the Child request */
			VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
					pstVapiReq,			/* Parent Request */
					VFSM_PlaybackSetRate,		/* Child Request Handler */
					PLAY_SET_RATE_INIT);		/* Child Request handler state */

			pstVapiReq->usReqState = PLAY_DATA_TRANSFER_NEXT_PACKET;
			/*add it at the front of the request list of the channel
			and call the ProcessRequestList function to process this child request first.*/
			VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

			UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
			pstChnl->stRecPlayInfo.uiChangeRate = False;
			UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);

			break;
		}

	case PLAY_DATA_TRANSFER_NEXT_PACKET:
		pstVapiReq->usReqState = PLAY_DATA_TRANSFER_PROCEED;

		/* proceed to the next frame*/
		if (pstFrameData->uiFrameLen == 0)
		{
			pstFrameData->ucIsHdrPresent = True;

			/* form frame length*/
			if (eHedFormat == pstHostSpeechData->uiHostSpeechDataFormat)
			{
				pstFrameData->pucFrameBuffer = VCORE_GetFrame(pstHostSpeechData, 
						pstHostSpeechData->uiBufferLength - pstFrameData->uiBufferSizeLeft, 
						FRAME_HEADER_LEN);

				if (pstFrameData->pucFrameBuffer == NULL)
					goto finish;	
	
				pstFrameData->uiFrameLen = UT_LE2CPU16(*((U16 *) pstFrameData->pucFrameBuffer + OFFSET_FOR_BUF_LEN));

				uiNumberOfFrames = 1;
			}
			else
			{
				uiNumberOfFrames = 0;
				pstFrameData->uiFrameLen = 0;

				while (pstFrameData->uiBufferSizeLeft >= (pstFrameData->uiFrameLen + uiFrameLenConst)
						&& (pstFrameData->uiFrameLen + uiFrameLenConst + FRAME_HEADER_LEN) <= uiMaxFrameLen)
				{
					pstFrameData->uiFrameLen += uiFrameLenConst;
					uiNumberOfFrames++;
				}
			}

			/* check frame lenght to be sent with left buffer size*/
			if ((pstFrameData->uiBufferSizeLeft < pstFrameData->uiFrameLen) 
				||(pstFrameData->uiFrameLen == 0) 
				|| uiNumberOfFrames == 0)
			{
				VCORE_CleanFrame(pstHostSpeechData, pstFrameData->pucFrameBuffer);
				pstFrameData->uiBufferSizeLeft = 0;
				/* since thre is nothing to send at all, stop timer, comlete transmitting */
				UT_TimerStop(pstHostSpeechData->pstSTimer);

				pstHostSpeechData->eStopAnnType = pstChnl->stRecPlayInfo.eStopAnnType;

				/* send MSP command with appropriate stop type*/
				if (eAS_STOP == pstHostSpeechData->eStopAnnType)
				{
					pstVapiReq->usReqState = PLAY_DATA_TRANSFER_FINISH;
					Status = VDEV_SetAs_Stop(pstChnl);
				}
				else
				{
					pstVapiReq->usReqState = PLAY_DATA_TRANSFER_COMPLETE_REMAIN;
					Status = VDEV_SetDrainPlayRecBuffer(pstChnl);
				}

				if (Status != SUCCESS)
					goto finish;

				break;
			}

			pstFrameData->uiFrameLen += FRAME_HEADER_LEN;

			/* allocate frame buffer to be sent*/
			pucFrameBufferWithHeader = UT_Calloc(1, pstFrameData->uiFrameLen);

			if (pucFrameBufferWithHeader == NULL)
			{
				Status = VAPI_ERR_NOMEM;
				goto finish;
			}	

			/* fill frame buffer to be sent*/
			if (eHedFormat == pstHostSpeechData->uiHostSpeechDataFormat)
			{
				VCORE_CleanFrame(pstHostSpeechData, pstFrameData->pucFrameBuffer);

				pstFrameData->pucFrameBuffer = VCORE_GetFrame(pstHostSpeechData, 
						pstHostSpeechData->uiBufferLength - pstFrameData->uiBufferSizeLeft, 
						pstFrameData->uiFrameLen);

				if (pstFrameData->pucFrameBuffer == NULL)
					goto finish;

				UT_MemCopy(pucFrameBufferWithHeader, pstFrameData->pucFrameBuffer, pstFrameData->uiFrameLen);
			}
			else
			{
				pstFrameData->pucFrameBuffer = VCORE_GetFrame(pstHostSpeechData, 
						pstHostSpeechData->uiBufferLength - pstFrameData->uiBufferSizeLeft, 
						uiFrameLenConst * uiNumberOfFrames);

				if (pstFrameData->pucFrameBuffer == NULL)
					goto finish;
									
				*((U32 *) pucFrameBufferWithHeader + OFFSET_FOR_TIMESTAMP) = UT_CPU2LE32(pstFrameData->uiFrameTimeStamp);
				*((U16 *) pucFrameBufferWithHeader + OFFSET_FOR_FRAME_TYPE) = UT_CPU2LE16(1);
				*((U16 *) pucFrameBufferWithHeader + OFFSET_FOR_BUF_LEN) = UT_CPU2LE16(uiFrameLenConst * uiNumberOfFrames);

				UT_MemCopy(pucFrameBufferWithHeader + FRAME_HEADER_LEN, pstFrameData->pucFrameBuffer, uiNumberOfFrames * uiFrameLenConst);

				pstFrameData->uiFrameTimeStamp += uiNumberOfFrames * uiFrameTimeStamp;
			}

			VCORE_CleanFrame(pstHostSpeechData, pstFrameData->pucFrameBuffer);
			pstFrameData->pucFrameBufferStart = pucFrameBufferWithHeader;
			pstFrameData->pucFrameBuffer = pstFrameData->pucFrameBufferStart;

			if (eHedFormat == pstHostSpeechData->uiHostSpeechDataFormat)
						pstFrameData->uiBufferSizeLeft -= pstFrameData->uiFrameLen;
			else
						pstFrameData->uiBufferSizeLeft -= (pstFrameData->uiFrameLen - FRAME_HEADER_LEN);
		}

		UT_Log(VCORE, INFO, "VFSM_PlayDataTransfer:Len in Frame = %d Hdr = %d SizeLeft = %d\n",
				       pstFrameData->uiFrameLen,
				       pstFrameData->ucIsHdrPresent,
				       pstFrameData->uiBufferSizeLeft);

		/* send packet with using SYNCDAT/SYNCEOF command depending on frame length */
		if (pstFrameData->uiFrameLen > uiMaxFrameLen)
		{
			UT_Log(VCORE, INFO, "VFSM_PlayDataTransfer: Sending SYNCDAT Command\n");
			Status = VDEV_SendData(pstChnl, pstFrameData, FC_SYNCDAT);

			if (Status != SUCCESS)
				goto finish;				

			/* whether header is/isn't*/
			if (pstFrameData->ucIsHdrPresent)
			{
				pstFrameData->uiFrameLen -= uiMaxFrameLen;
				pstFrameData->pucFrameBuffer = pstFrameData->pucFrameBuffer + uiMaxFrameLen;
			}
			else
			{
				pstFrameData->uiFrameLen -= uiMaxFrameLen + PACKET_HEADER_LEN;
				pstFrameData->pucFrameBuffer =
					pstFrameData->pucFrameBuffer + uiMaxFrameLen + PACKET_HEADER_LEN;
			}

			pstFrameData->ucIsHdrPresent = False;
		}
		else
		{
			UT_Log(VCORE, INFO, "VFSM_PlayDataTransfer: Sending SYNCEOF Command\n");

			if (ePlaybackExt == pstHostSpeechData->ePlaybackMode)
				Status = VDEV_SendDataExtended(pstChnl, pstFrameData);
			else
				Status = VDEV_SendData(pstChnl, pstFrameData, FC_SYNCEOF);

			if (Status != SUCCESS)
				goto finish;				

			if (NULL != pstFrameData->pucFrameBufferStart)
				UT_FreeMem(pstFrameData->pucFrameBufferStart);

			pstFrameData->uiFrameLen = 0;
			pstFrameData->pucFrameBuffer = NULL;
			pstFrameData->ucIsHdrPresent = True;
		}

		break;

	case PLAY_DATA_TRANSFER_COMPLETE_REMAIN:
		/* timeout could occur or stop playback could be called*/
		if (Event == FSM_EVENT_PLAY_REC_STOP || Event == FSM_EVENT_REC_PLAY_TERMINATOR)
			break;

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_DRAIN_PLAYREC_BUFFER);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = PLAY_DATA_TRANSFER_WAIT_IND;

		break;

	case PLAY_DATA_TRANSFER_WAIT_IND:
		/* timeout could occur or stop playback could be called*/
		if (Event == FSM_EVENT_PLAY_REC_STOP || Event == FSM_EVENT_REC_PLAY_TERMINATOR)
		{
			UT_Log(VCORE, INFO, "VFSM_PlayDataTransfer: got FSM_EVENT_PLAY_REC_STOP/FSM_EVENT_REC_PLAY_TERMINATOR\n");

			pstHostSpeechData->eStopAnnType = pstChnl->stRecPlayInfo.eStopAnnType;

			/* We're waiting DRAIN indication thus there is no sense 
			of sending DRAIN again, but only AS_STOP.*/
			if (eAS_STOP != pstHostSpeechData->eStopAnnType)
				break;

			pstVapiReq->usReqState = PLAY_DATA_TRANSFER_FINISH;
			Status = VDEV_SetAs_Stop(pstChnl);

			if (Status != SUCCESS)
				goto finish;			

			/* Stop type was DRAIN but now it's changed to AS_STOP, 
			we must get AS_STOP responce and still get indication.*/
			pstHostSpeechData->uiDrainToAsStop = True;

			break;
		}

		goto finish;
		break;

	case PLAY_DATA_TRANSFER_FINISH:
		/* timeout could occur or stop playback could be called*/
		if (Event == FSM_EVENT_PLAY_REC_STOP || Event == FSM_EVENT_REC_PLAY_TERMINATOR)
			break;

		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_AS_STOP);

		/* got responce for AS_STOP, it's ramained to wait for DRAIN inidication*/
		if (pstHostSpeechData->uiDrainToAsStop == True)
		{
			pstVapiReq->usReqState = PLAY_DATA_TRANSFER_WAIT_IND;
			pstHostSpeechData->uiDrainToAsStop = False;
			break;
		}

		goto finish;		
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_PlayDataTransfer: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}	

	return SUCCESS;

finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_PlayDataTransfer: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_PlayDataTransfer: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	if (pstHostSpeechData->iFileDisc >= 0)
		UT_CloseFd(pstHostSpeechData->iFileDisc);

	if (Status == SUCCESS)
	{
		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);

		if (pstChnl->stRecPlayInfo.uiStopPlayCount)
			pstChnl->stRecPlayInfo.uiStopPlayCount--;

		if (pstChnl->stRecPlayInfo.uiStartPlayCount)
			pstChnl->stRecPlayInfo.uiStartPlayCount--;

		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
	}

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_PlayDataTransfer: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_StartRecord : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#  Following MSP messages will be sent\n
 *          -#  SET_FLOW_CONTROL\n
 *          -#  SET_CHNL_MODE\n
 *      -#  Start the timer and register RecPlayTerminator as the 
 *          timeout handler.
 *      -#  Handle the indications containing the recorded data and store 
 *          the data in buffer.
 *          -#  VOIP_SYNCDAT\n
 *          -#  VOIP_SYNCEOF\n
 *      -#  Send data indication response for each recording indication
 *          recieved from MSP.
 *      -#  Again send SET_CHNL_MODE to set the channel back in VoP mode.
 *
 *  Assumptions:\n
 *      -#  Flow control is enabled by default\n.   
 *
 *  \return   SUCCESS or FAILURE code.
 *
 *  \param pstChnl  This will be a channel on which MSP commands will be sent.
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *  
 */
VSTATUS VFSM_StartRecord(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SVapiReq * pstChildVapiReq;
	SHostSpeechData *pstHostSpeechData = NULL;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_StartRecord: Entering conn(%u) state(%u) \n", ConnId, pstVapiReq->usReqState);

	pstHostSpeechData = (SHostSpeechData *) pstVapiReq->pvUserData;

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case START_RECORDING_FLOW_CONTROL:
		pstVapiReq->usReqState = START_RECORDING_SWITH_REC_MODE;
		Status = VDEV_SetFlowControl(pstChnl);

		if (Status != SUCCESS)
			goto finish;		

		break;

	case START_RECORDING_SWITH_REC_MODE:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_SET_FLOWCON);

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = START_RECORDING_INIT;
		Status = VDEV_SetChnlMode(pstChnl, eRec, pstHostSpeechData);

		if (Status != SUCCESS)
			goto finish;		
		
		break;

	case START_RECORDING_INIT:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_SET_CHANNEL_MODE);

		if (Status != SUCCESS)
			goto finish;

		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.eRecPlayState = eRec;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);			

		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SHostSpeechData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}

		/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
		UT_MemCopy(pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, sizeof(SHostSpeechData));		

		/* initialise the Child request */
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */ 
				VFSM_RecDataTransfer,		/* Child Request Handler */
				REC_DATA_TRANSFER_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = START_RECORDING_SWITCH_VOP_MODE;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case START_RECORDING_SWITCH_VOP_MODE:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = START_RECORDING_FINISH;
		Status = VDEV_SetChnlMode(pstChnl, eVop, pstHostSpeechData);

		if (Status != SUCCESS)
			goto finish;

		break;

	case START_RECORDING_FINISH:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_SET_CHANNEL_MODE);
	
		if (Status != SUCCESS)
			goto finish;
		
		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.eRecPlayState = eVop;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);

		goto finish;
				
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_StartRecord: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	};

	return SUCCESS;
finish:

	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_StartRecord: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);
	
	UT_TimerDelete(pstHostSpeechData->pstSTimer);

	UT_Log(VCORE, DEBUG, "VFSM_StartRecord: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_StartRecord: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_StartPlayback: The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#  Following MSP messages will be sent\n
 *          -#  SET_FLOW_CONTROL\n
 *          -#  SET_CHNL_MODE\n
 *      -#  Start the timer and register RecPlayTerminator as the 
 *          timeout handler.
 *      -#  Read the Data from buffer and send the data to MSP for playout
 *          -#  VOIP_SYNCDAT\n
 *          -#  VOIP_SYNCEOF\n
 *      -#  Again send SET_CHNL_MODE to set the channel back in VoP mode.
 *
 *  Assumptions:\n
 *      -#  Flow control is enabled by default\n.   
 *
 *  \return   SUCCESS or FAILURE code.
 *
 *  \param pstChnl  This will be a channel on which MSP commands will be sent.
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
 *  
 */
VSTATUS VFSM_StartPlayback(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SVapiReq * pstChildVapiReq;
	SHostSpeechData *pstHostSpeechData = NULL;
	CONNID ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_StartPlayback: Entering conn(%u) state(%u) \n", ConnId, pstVapiReq->usReqState);

	pstHostSpeechData = (SHostSpeechData *) pstVapiReq->pvUserData;

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case START_PLAYING_SWITCH_PLAY_MODE:
		pstVapiReq->usReqState = START_PLAYING_INIT;
		switch (pstHostSpeechData->ePlaybackMode)
		{
		case ePlaybackStandard:
			Status = VDEV_SetChnlMode(pstChnl, ePlay, pstHostSpeechData);
			break;

		case ePlaybackExt:
			Status = VDEV_SetChnlMode(pstChnl, ePlayExt, pstHostSpeechData);
			break;
			
		case ePlaybackSpeedChangeable:
			Status = VDEV_SetChnlMode(pstChnl, ePlaySpeedChangeable, pstHostSpeechData);
			break;
		}

		if (Status != SUCCESS)
			goto finish;		

		break;

	case START_PLAYING_INIT:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_SET_CHANNEL_MODE);

		if (Status != SUCCESS)
			goto finish;

		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);

		switch (pstHostSpeechData->ePlaybackMode)
		{
		case ePlaybackStandard:
			pstChnl->stRecPlayInfo.eRecPlayState = ePlay;
			break;
			
		case ePlaybackExt:
			pstChnl->stRecPlayInfo.eRecPlayState = ePlayExt;
			break;
			
		case ePlaybackSpeedChangeable:
			pstChnl->stRecPlayInfo.eRecPlayState = ePlaySpeedChangeable;
			break;
		}

		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);

		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SHostSpeechData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}		

		/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
		UT_MemCopy(pstChildVapiReq->pvUserData, pstVapiReq->pvUserData, sizeof(SHostSpeechData));		
		
		/* initialise the Child request */
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */ 
				VFSM_PlayDataTransfer,		/* Child Request Handler */
				PLAY_DATA_TRANSFER_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = START_PLAYING_SWITCH_VOP_MODE;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case START_PLAYING_SWITCH_VOP_MODE:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;		

		pstVapiReq->usReqState = START_PLAYING_FINISH;
		Status = VDEV_SetChnlMode(pstChnl, eVop, pstHostSpeechData);

		if (Status != SUCCESS)
			goto finish;

		break;

	case START_PLAYING_FINISH:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_SET_CHANNEL_MODE);

		if (Status != SUCCESS)
			goto finish;	

		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.eRecPlayState = eVop;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		goto finish;
			
	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_StartPlayback: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	};

	return SUCCESS;

finish:	
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_StartPlayback: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
		pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);		

	UT_Log(VCORE, DEBUG, "VFSM_StartPlayback: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, pstChnl->ConnId);

	UT_TimerDelete(pstHostSpeechData->pstSTimer);

	if (pstHostSpeechData->pucBuffer != NULL)
		UT_FreeMem(pstHostSpeechData->pucBuffer);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);
	
	UT_Log(VCORE, INFO, "VFSM_StartPlayback: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

VSTATUS VFSM_PlaybackSetRate(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	EPlaybackRate *pstPlaybackRate;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_PlaybackSetRate: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstPlaybackRate = (EPlaybackRate *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case PLAY_SET_RATE_INIT:
		pstVapiReq->usReqState = PLAY_SET_RATE_FINISH;
		Status = VDEV_PlaybackSetRate(pstChnl, *pstPlaybackRate);

		if (Status != SUCCESS)
			goto finish;

		break;

	case PLAY_SET_RATE_FINISH:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
				CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_AS_SET_RATE);

		goto finish;		

		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_PlaybackSetRate: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}	

	return SUCCESS;

finish:
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_PlaybackSetRate: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, pstChnl->ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_PlaybackSetRate: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, pstChnl->ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_PlaybackSetRate: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

VSTATUS VFSM_ConvertSpeechTransfer(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SConvertFrameInfo *pstConvertFrameData;
	SHostSpeechData	*pstHostSpeechData;
	U16 usActFrameLen = 0;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_ConvertSpeechTransfer: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstConvertFrameData = (SConvertFrameInfo *) pstVapiReq->pvExecData;
	pstHostSpeechData = (SHostSpeechData *) pstVapiReq->pvUserData;

	switch (pstVapiReq->usReqState)
	{
	case CONVERT_SPEECH_TRANSFER_INIT:
		pstVapiReq->pvExecData = UT_Calloc(1, sizeof(SConvertFrameInfo));

		if (pstVapiReq->pvExecData == NULL)
		{
			Status = VAPI_ERR_NOMEM;
			break;
		}

		pstConvertFrameData = (SConvertFrameInfo *) pstVapiReq->pvExecData;
		pstConvertFrameData->stInFrame.uiBufferSizeLeft = pstHostSpeechData->uiBufferLength;
		pstConvertFrameData->stInFrame.pucFrameBuffer = pstHostSpeechData->pucBuffer;
		pstConvertFrameData->stInFrame.ucIsHdrPresent = True;

		pstConvertFrameData->stInFrame.uiFrameLen =
		    UT_LE2CPU16(*((U16 *) pstConvertFrameData->stInFrame.pucFrameBuffer + OFFSET_FOR_BUF_LEN)) + FRAME_HEADER_LEN;

		if (pstConvertFrameData->stInFrame.uiBufferSizeLeft < pstConvertFrameData->stInFrame.uiFrameLen)
		{
			pstConvertFrameData->stInFrame.uiBufferSizeLeft = 0;
			goto finish;
		}

		pstConvertFrameData->stInFrame.uiBufferSizeLeft -= pstConvertFrameData->stInFrame.uiFrameLen;
		pstVapiReq->usReqState = CONVERT_SPEECH_TRANSFER_NEXT;
		
		if (pstConvertFrameData->stInFrame.uiFrameLen > MAX_FRAME_LEN_SYNCEOF)
		{
			Status = VDEV_SendData(pstChnl, &pstConvertFrameData->stInFrame, FC_SYNCDAT);
			pstConvertFrameData->stInFrame.uiFrameLen -= MAX_FRAME_LEN_SYNCEOF;
		}
		else
		{
			Status = VDEV_SendData(pstChnl, &pstConvertFrameData->stInFrame, FC_SYNCEOF);
			pstConvertFrameData->stInFrame.uiFrameLen = 0;
			pstConvertFrameData->stInFrame.pucFrameBuffer = NULL;
		}

		if (Status != SUCCESS)
			goto finish;

		pstHostSpeechData->uiNumOfPacketsToBeConverted++;
		
		pstConvertFrameData->stOutFrame.uiFrameLen = 0;
		pstConvertFrameData->stOutFrame.ucIsHdrPresent = True;
		pstConvertFrameData->stOutFrame.pucFrameBuffer = NULL;
		pstConvertFrameData->stOutFrame.uiBufferSizeLeft = pstHostSpeechData->uiOutBufferLength;

		break;

	case CONVERT_SPEECH_TRANSFER_NEXT:
		/*If it's not an indication then it's response to SYNCDAT or SYNCEOF */
		if (CMD_TYPE_INDICATION == *((U8 *) pstMsg->fifo + 2))
		{
			if (FC_SYNCDAT == UT_LE2CPU16(*((U16 *) (pstMsg->fifo) + 2)))
			{
				Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
							   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_INDICATION,
							   FC_SYNCDAT);
				if (Status != SUCCESS)
					goto finish;

				/* ACK indication*/
				VDEV_SendDataInd(pstChnl, FC_SYNCDAT);
			}
			else
			{
				Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
							   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_INDICATION,
							   FC_SYNCEOF);
				if (Status != SUCCESS)
					goto finish;

				/* ACK indication*/
				VDEV_SendDataInd(pstChnl, FC_SYNCEOF);				
			}

			pstHostSpeechData->uiNumOfPacketsToBeConverted--;

			if (pstHostSpeechData->uiProceedToStop == True && pstHostSpeechData->uiNumOfPacketsToBeConverted == 0)
				goto finish;

			Status = VCORE_HandleFrame(pstMsg, &pstConvertFrameData->stOutFrame);

			if (Status == MALFORMED_PACKET)
				break;

			if (SYNCEOF_PACKET != Status && SYNCDAT_PACKET != Status)
			{
				Status = VAPI_ERR_INVALID_FRAME;
				goto finish;
			}
			
			if (Status == SYNCEOF_PACKET)
			{
				Status = SUCCESS;

				if (pstConvertFrameData->stOutFrame.uiBufferSizeLeft <= 0)
				{
					pstHostSpeechData->uiProceedToStop = True;
					if (pstHostSpeechData->uiNumOfPacketsToBeConverted == 0)
						/*all indications on sent packets are received */
						goto finish;
					else
						/*wait while we get all indications on sent packets*/
						break;
				}

				usActFrameLen = pstConvertFrameData->stOutFrame.uiFrameLen;
				usActFrameLen = ALIGN_MESSAGE_LENGTH(usActFrameLen);

				if (pstConvertFrameData->stOutFrame.uiBufferSizeLeft > usActFrameLen)
				{
					/*This function should write the complete frame into the buffer */
					UT_MemCopy((U8 *) (pstHostSpeechData->pucOutBuffer +
							   pstHostSpeechData->uiOutBufferLength -
							   pstConvertFrameData->stOutFrame.uiBufferSizeLeft),
						(void *)(pstConvertFrameData->stOutFrame.pucFrameBuffer),
						pstConvertFrameData->stOutFrame.uiFrameLen);

					pstConvertFrameData->stOutFrame.uiBufferSizeLeft -= usActFrameLen;
					pstConvertFrameData->stOutFrame.uiFrameLen = 0;
					pstConvertFrameData->stOutFrame.ucIsHdrPresent = True;
					pstConvertFrameData->stOutFrame.pucFrameBuffer = NULL;
				}
				else
				{	pstConvertFrameData->stOutFrame.uiBufferSizeLeft = 0;
					pstHostSpeechData->uiProceedToStop = True;
					
					if (pstHostSpeechData->uiNumOfPacketsToBeConverted == 0)
						/*all indications on sent packets are received */
						goto finish;
					else
						/*wait while we get all indications on sent packets*/
						break;
				}

			}
		}
		else
		{
			/*we must not send any packet, since we're proceeding to stop*/
			if (pstHostSpeechData->uiProceedToStop == True)
				break;
			
			/* proceed to the next frame*/
			if (pstConvertFrameData->stInFrame.uiFrameLen == 0)
			{
				pstConvertFrameData->stInFrame.pucFrameBuffer =
						(pstHostSpeechData->pucBuffer + (pstHostSpeechData->uiBufferLength -
						pstConvertFrameData->stInFrame.uiBufferSizeLeft));

				pstConvertFrameData->stInFrame.uiFrameLen =
						UT_CPU2LE16(*((U16 *) (pstConvertFrameData->stInFrame.pucFrameBuffer) 
						+ OFFSET_FOR_BUF_LEN)) 
				    		+ FRAME_HEADER_LEN;

				if ((pstConvertFrameData->stInFrame.uiBufferSizeLeft <
					pstConvertFrameData->stInFrame.uiFrameLen)
					|| (pstConvertFrameData->stInFrame.uiFrameLen - FRAME_HEADER_LEN == 0))
				{
					pstConvertFrameData->stInFrame.uiBufferSizeLeft = 0;
					pstHostSpeechData->uiProceedToStop = True;
					break;
				}

				pstConvertFrameData->stInFrame.ucIsHdrPresent = True;
				pstConvertFrameData->stInFrame.uiBufferSizeLeft -= pstConvertFrameData->stInFrame.uiFrameLen;

			}

			UT_Log(VCORE, INFO, "VFSM_ConvertSpeechTransfer:Len in Frame = %d Hdr = %d SizeLeft = %d\n",
				       pstConvertFrameData->stInFrame.uiFrameLen,
				       pstConvertFrameData->stInFrame.ucIsHdrPresent,
				       pstConvertFrameData->stInFrame.uiBufferSizeLeft);

			if (pstConvertFrameData->stInFrame.uiFrameLen > MAX_FRAME_LEN_SYNCEOF)
			{
				UT_Log(VCORE, INFO, "VFSM_ConvertSpeechTransfer: Sending SYNCDAT Command\n");
				Status = VDEV_SendData(pstChnl, &pstConvertFrameData->stInFrame, FC_SYNCDAT);

				if (Status != SUCCESS)
				{
					pstHostSpeechData->uiProceedToStop = True;
					break;
				}

				/* whether header is/isn't*/
				if (pstConvertFrameData->stInFrame.ucIsHdrPresent)
				{
					pstConvertFrameData->stInFrame.uiFrameLen -= MAX_FRAME_LEN_SYNCEOF;
					pstConvertFrameData->stInFrame.pucFrameBuffer = pstConvertFrameData->stInFrame.pucFrameBuffer 
											+ MAX_FRAME_LEN_SYNCEOF;
				}
				else
				{
					pstConvertFrameData->stInFrame.uiFrameLen -= MAX_FRAME_LEN_SYNCEOF + PACKET_HEADER_LEN;
					pstConvertFrameData->stInFrame.pucFrameBuffer = pstConvertFrameData->stInFrame.pucFrameBuffer 
											+ MAX_FRAME_LEN_SYNCEOF 
											+ PACKET_HEADER_LEN;
				}

				pstConvertFrameData->stInFrame.ucIsHdrPresent = False;
			}
			else
			{
				UT_Log(VCORE, INFO, "VFSM_ConvertSpeechTransfer: Sending SYNCEOF Command\n");

				Status = VDEV_SendData(pstChnl, &pstConvertFrameData->stInFrame, FC_SYNCEOF);

				if (Status != SUCCESS)
				{
					pstHostSpeechData->uiProceedToStop = True;
					break;
				}
				
				pstHostSpeechData->uiNumOfPacketsToBeConverted++;
				
				pstConvertFrameData->stInFrame.uiFrameLen = 0;
				pstConvertFrameData->stInFrame.pucFrameBuffer = NULL;
				pstConvertFrameData->stInFrame.ucIsHdrPresent = True;
			}
			
		}
		
		break;

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_ConvertSpeechTransfer: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}	

	return SUCCESS;

finish:
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_ConvertSpeechTransfer: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, pstChnl->ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_ConvertSpeechTransfer: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, pstChnl->ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_ConvertSpeechTransfer: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}

/****************************************************************************
 * VFSM_ConvertHostSpeechData : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      -#  Obtain the VAPI request node from the the channel structure.
 *      -#  Check the state of the request usReqState\n
 *      -#  Check the input GTL message (response) for SUCCESS.\n
 *      -#  Send the MSP message according to current state using 
 *          VCORE_SendMSPReq
 *      -#  Update the current state. \n
 *      -#  Following MSP messages will be sent\n
 *          -#  SET_FLOW_CONTROL\n
 *          -#  SET_CHNL_MODE\n
 *      -#  Start the timer and register RecPlayTerminator as the 
 *          timeout handler.
 *      -#  Read the Data from buffer and send the data to MSP.
 *          -#  VOIP_SYNCDAT\n
 *          -#  VOIP_SYNCEOF\n
 *      -#  Handle the corresponding indication from MSP and store it in
 *          the destination buffer.
 *      -#  Again send SET_CHNL_MODE to set the channel back in VoP mode.
 *
 *  Assumptions:\n
 *      -#  Flow control is enabled by default\n.
 *      -#  Code is untested
 *
 *  \return   SUCCESS or FAILURE code.
 *
 *  \param pstChnl  This will be a channel on which MSP commands will be sent.
 *  \param pstMsg   Message (response) obtained from GTL. When it is called
  * *  
 */
VSTATUS VFSM_ConvertHostSpeechData(IN SChnl * pstChnl, IN SVapiReq * pstVapiReq, IN U16 Event, IN gtl_msg_t * pstMsg)
{
	VSTATUS Status = SUCCESS;
	SVapiReq * pstChildVapiReq;
	SHostSpeechData *pstHostSpeechData = NULL;
	CONNID ConnId;
	ConnId = pstChnl->ConnId;

	UT_Log(VCORE, DEBUG, "VFSM_ConvertHostSpeechData: Entering conn(%u) state(%u)\n", ConnId, pstVapiReq->usReqState);

	pstHostSpeechData = (SHostSpeechData *) pstVapiReq->pvUserData;

	/*Take action according to current state of request */
	switch (pstVapiReq->usReqState)
	{
	case CONVERT_SPEECH_SET_FLOW_CTR:
		pstVapiReq->usReqState = CONVERT_SPEECH_SWITCH_PLAY_MODE;
		Status = VDEV_SetFlowControl(pstChnl);

		if (Status != SUCCESS)
			goto finish;
		
		break;

	case CONVERT_SPEECH_SWITCH_PLAY_MODE:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_SET_FLOWCON);
		if (Status != SUCCESS)
			goto finish;

		pstVapiReq->usReqState = CONVERT_SPEECH_INIT;
		Status = VDEV_SetChnlMode(pstChnl, ePlay, pstHostSpeechData);

		if (Status != SUCCESS)
			goto finish;
		
		break;
		
	case CONVERT_SPEECH_INIT:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_SET_CHANNEL_MODE);
		if (Status != SUCCESS)
			goto finish;
		
		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.eRecPlayState = ePlay;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);

		pstChildVapiReq  = VCORE_AllocateRequest(sizeof(SHostSpeechData));
		if (!pstChildVapiReq)
		{
			Status = VAPI_ERR_NOMEM;
			goto finish;
		}		

		/* Initialise the UserData (allocated by VCORE_AllocateRequest) */
		UT_MemCopy(pstChildVapiReq->pvUserData, pstHostSpeechData, sizeof(SHostSpeechData));		
		
		/* initialise the Child request */
		VCORE_SetChildRequest(pstChildVapiReq,		/* Child Request */
				pstVapiReq,			/* Parent Request */ 
				VFSM_ConvertSpeechTransfer,	/* Child Request Handler */
				CONVERT_SPEECH_TRANSFER_INIT);	/* Child Request handler state */

		pstVapiReq->usReqState = CONVERT_SPEECH_SWITCH_VOIP_MODE;
		/*add it at the front of the request list of the channel
		and call the ProcessRequestList function to process this child request first.*/ 
		VCORE_StartChannelChildRequest(pstChnl, pstChildVapiReq);

		break;

	case CONVERT_SPEECH_SWITCH_VOIP_MODE:
		/* retrieve the child status */
		Status = pstVapiReq->Status;

		if (Status != SUCCESS)
			goto finish;		

		pstVapiReq->usReqState = CONVERT_SPEECH_FINISH;
		Status = VDEV_SetChnlMode(pstChnl, eVop, pstHostSpeechData);

		if (Status != SUCCESS)
			goto finish;

		break;

	case CONVERT_SPEECH_FINISH:
		Status = VCORE_CheckStatus(Event, pstMsg, pstChnl->usMSPChnlId,
					   CMD_CLASS_CONF_CHANNEL, CMD_TYPE_CONF_RESP, FC_VOIP_SET_CHANNEL_MODE);

		if (Status != SUCCESS)
			goto finish;	

		UT_MutexLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		pstChnl->stRecPlayInfo.eRecPlayState = eVop;
		UT_MutexUnLock(&pstChnl->stRecPlayInfo.stSemRecPlay);
		goto finish;		

	default:
		Status = VAPI_ERR_UNDEFINED;
		UT_ErrorLog(VCORE, "VFSM_ConvertHostSpeechData: wrong state %u\n", pstVapiReq->usReqState);
		goto finish;		
	}
	
	return SUCCESS;

finish:
	if (Status != SUCCESS)
		UT_ErrorLog(VCORE, "VFSM_ConvertHostSpeechData: Error dev(%u) devchnl(%u) conn(%u) req(0x%x) reqid(%u) pstChnl(0x%x)\n",
			pstChnl->pstDev->DevId, pstChnl->usMSPChnlId, pstChnl->ConnId, pstVapiReq, pstVapiReq->usReqState, pstChnl);

	UT_Log(VCORE, DEBUG, "VFSM_ConvertHostSpeechData: Completing req state(%u) status(%d) conn(%u)\n",
			pstVapiReq->usReqState, Status, pstChnl->ConnId);

	VCORE_DoReqCompletion(pstChnl, pstVapiReq, Status);

	UT_Log(VCORE, INFO, "VFSM_ConvertHostSpeechData: Exiting status(%d) conn(%u)\n", Status, ConnId);

	return Status;
}


