/*! \file ut.h 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#ifndef _UT_H
#define _UT_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <stdarg.h>
#include <netinet/in.h>
#ifndef DONT_USE_PTHREADS
#include <pthread.h>
#endif
#include "avl.h"
#include "ut_port.h"

typedef struct timespec STimeSpec;

/**************************
 Constants
***************************/

/** Indicate success of an operation*/
#define SUCCESS 0

/** Indicate failure of an operation*/
#define FAILURE -1

#define INFINITE_TIMEOUT	0xFFFFFFFE
#define MAX_TIMEOUT		30000	/* 30 sec */

typedef enum {
	UT_INITIALIZING = 0,
	UT_INITIALIZED = 1,
	UT_CLOSING = 2,
	UT_CLOSED = 3
} UT_Status;

/*Defined for specifying input output parameters in function prototypes*/
#define IN
#define OUT

/**************************
 Memory handling functions 
***************************/
void *__UT_malloc(int iSize, char *fName, int lineNo);
void __UT_free(void *ptr, char *fName, int lineNo);

#ifndef MEMDBG

#define UT_AllocMem(size) __UT_malloc(size, __FILE__, __LINE__)
#define UT_FreeMem(ptr) __UT_free(&ptr, __FILE__, __LINE__)

#else

void *UT_AllocMemDbg(char *strFileName, INT iLineNo, INT iSize);
void UT_FreeMemDbg(char *strFileName, INT iLineNo, void *pvMem);
void UT_DumpMemInfo(void);

#define UT_AllocMem(iSize)		UT_AllocMemDbg(__FILE__, __LINE__, iSize)
#define UT_FreeMem(ptrPtr)		if (ptrPtr) UT_FreeMemDbg(__FILE__, __LINE__, &ptrPtr)

#endif


/************************
Endianness conversion functions
*************************/
#define UT_SWAP16(usWord)	(U16)((U16)((U16)(usWord) << 8) | (U16)((U16)(usWord) >> 8))

#define UT_SWAP32(ulWord)	((((ulWord) & 0x000000ff) << 24) |	\
				(((ulWord) & 0x0000ff00) <<  8) |	\
				(((ulWord) & 0x00ff0000) >>  8) |	\
				(((ulWord) & 0xff000000) >> 24))

#ifdef VAPI_BIGENDIAN
#define UT_CPU2LE16(usWord)	UT_SWAP16(usWord)
#define UT_CPU2BE16(usWord)	(usWord)
#define UT_CPU2LE32(ulWord)	UT_SWAP32(ulWord)
#define UT_CPU2BE32(ulWord)	(ulWord)
#define UT_LE2CPU16(usWord)	UT_SWAP16(usWord)
#define UT_BE2CPU16(usWord)	(usWord)
#define UT_LE2CPU32(ulWord)	UT_SWAP32(ulWord)
#define UT_BE2CPU32(ulWord)	(ulWord)
#else
#define UT_CPU2LE16(usWord)	(usWord)
#define UT_CPU2BE16(usWord)	UT_SWAP16(usWord)
#define UT_CPU2LE32(ulWord)	(ulWord)
#define UT_CPU2BE32(ulWord)	UT_SWAP32(ulWord)
#define UT_LE2CPU16(usWord)	(usWord)
#define UT_BE2CPU16(usWord)	UT_SWAP16(usWord)
#define UT_LE2CPU32(ulWord)	(ulWord)
#define UT_BE2CPU32(ulWord)	UT_SWAP32(ulWord)
#endif

#define UT_MAX(a, b)       ((a) < (b) ? (b) : (a))
#define UT_MIN(a, b)       ((a) < (b) ? (a) : (b))

/**Interface Functions*/
/** Initialize UT*/
INT UT_Init(void);

/** Cleanup UT*/
void UT_Close(void);

/** Initialize UT Port Module*/
INT UT_OSWInit(void);

/** Close UT Port Module*/
void UT_OSWClose(void);

/***********************
Thread related utility 
************************/
SThread *UT_ThreadCreate(char *pTName, PFNThreadEntry pfnFunc, void *pvArg);
INT UT_ThreadCancel(SThread * pstThread);
INT UT_ThreadJoin(SThread * pstThread);
INT UT_ThreadDetach(SThread * pstThread);
void UT_ThreadTestCancel(void);
INT UT_ThreadSetCancelParams(Boolean iEnableCancel, Boolean iEnableAsync);

/**********************
Semaphore Utilities
***********************/
/**This function initializes the semaphore*/
SSem *UT_SemInit(IN UINT uiValue);

/**This function waits for the specified amount of time with semTake*/
INT UT_SemWait(IN SSem * pstSem, IN UINT uiTimeout);

/**It releases the semaphore passed to it using semGive*/
INT UT_SemPost(IN SSem * pstSem);

/**It destroys the semaphore passed to it*/
INT UT_SemDestroy(IN SSem * pstSem);

/*********************
  Mutexes Utilities 
*********************/
INT UT_MutexInit(IN SMutex * pstMutex);
INT UT_MutexDestroy(IN SMutex * pstMutex);
INT UT_MutexLock(IN SMutex * pstMutex);
INT UT_MutexUnLock(IN SMutex * pstMutex);


#define UT_OffsetOf(type, member)	((unsigned long) &(((type *)0)->member))

#define UT_ContainerOf(p, type, member)  ((type *) ((char *)(p) - UT_OffsetOf(type, member)))

/***********************
Link List Utilities
************************/
/**Structures for a Linked list's node and linked list */
typedef struct _SLstNode {
	struct _SLstNode *pstNext;	/**<Pointer to the next node in the list*/
	struct _SLstNode *pstPrev;	/**<Pointer to the previous node in the list*/
} SLstNode;

typedef struct _SLinkedList {
	SLstNode stHead;
	SMutex stSem;		/**<Pointer to the semaphore linked with a linked list*/
} SLinkedList;

typedef int (*PFNListFunc) (SLinkedList *pstLinkedList, SLstNode *pstNode, void *pvData);

/**This function creates a linked list and return the pointer to the same*/
INT UT_ListInit(IN SLinkedList * pstLinkedList);

/**This function adds an element in front of the linked list*/
void __UT_AddToFront(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode);
void UT_AddToFront(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode);

/**This function adds an element in the end of the linked list*/
void __UT_AddToTail(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode);
void UT_AddToTail(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode);

/**This function adds an element before defined one of the linked list*/
void __UT_ListAddBefore(SLstNode * pstBaseNode, SLstNode * pstNode);
void UT_ListAddBefore(IN SLinkedList * pstLinkedList, IN SLstNode * pstBaseNode, IN SLstNode * pstNode);

/**This function adds an element after defined one of the linked list*/
void __UT_ListAddAfter(SLstNode * pstBaseNode, SLstNode * pstNode);
void UT_ListAddAfter(IN SLinkedList * pstLinkedList, IN SLstNode * pstBaseNode, IN SLstNode * pstNode);

/**This function list each element of the linked list*/
#define __UT_ListForEach(pos, head) \
	for (pos = (head)->pstNext; pos != (head); \
        	pos = pos->pstNext)

/**This function list each element of the linked list in reversed way*/
#define __UT_ListForEachReversed(pos, head) \
	for (pos = (head)->pstPrev; pos != (head); \
		pos = pos->pstPrev)

/**This function removes an element from the linked list*/
void __UT_RemoveNode(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode);
void UT_RemoveNode(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode);

/**This function gets the front node of the linked list*/
SLstNode *__UT_GetFrontNode(IN SLinkedList * pstLinkedList);
SLstNode *UT_GetFrontNode(IN SLinkedList * pstLinkedList);

/**This function checks if the Linked list is empty or not*/
Boolean __UT_IsListEmpty(IN SLinkedList * pstLinkedList);
Boolean UT_IsListEmpty(IN SLinkedList * pstLinkedList);

/**This function deletes the linked list*/
INT UT_ListDestroy(IN SLinkedList * pstLinkedList);

/***********************
AVL tree Utilities
************************/
/**Structure of the AVL tree with its semaphore*/
typedef struct _SAVLTree {
	SAVL_Tree stAvlTree;	    /**<Structure of AVL tree*/
	SMutex stSem;		    /**<Pointer to the semaphore attached to the AVL tree*/
} SAVLTree;

/**Structure of the node of the AVL tree*/
typedef struct _SAvlNode {
	SAVL stAvl;	/**<Structure of the AVL tree's node*/
	U32 uiKey;	/**<Key to identify a node in the tree*/
	void *pvData;	/**<Pointer to the data to be attached to the node*/
} SAvlNode;

/**This function allocates the memory for the AVL tree*/
SAVLTree *UT_CreateAVLTree(void);

/**This function adds the data passed to it in the AVL tree*/
INT UT_AddToAVLTree(IN SAVLTree * pstAVLTree, IN U32 uiKey, IN void *pvData);

/**This function removes an element from the AVL tree*/
void UT_RemFromAVLTree(IN SAVLTree * pstAVLTree, IN U32 uiKey);

/**This function searches the node in the tree acoording to the key passed*/
void *UT_SearchInAVLTree(IN SAVLTree * pstAVLTree, IN U32 uiKey);

/**This functions deletes the AVL tree by freeing the memory allotted to it*/
void UT_DeleteAVLTree(IN SAVLTree * pstAVLTree);

void *UT_Calloc(INT num, INT size);

/*********************************
Sorted Array Function Definitions
**********************************/
/**Structure of an element of the Array*/
typedef struct _SArrayNode {
	U32 uiKey;	/**<Unique key to identify an element of the array*/
} SArrayNode;

typedef struct _SSortArray {
	SArrayNode **pastStart;	/**<Pointer to the first element of the array*/
	U32 uiSize;		/**<Number of elements that can be added in the array*/
	U32 uiNumElement;	/**<Number of elements present in the array*/
	SMutex stSem;			/**<Pointer to the semaphore attached to the array*/
} SSortArray;

/**This function will allocate the memory corresponding to the uSize
   parameter and will return a pointer to the newly created array*/
SSortArray *UT_CreateSortedArray(IN UINT uiSize);

/**This function adds an element to the array and then sorts the array by bubble sort*/
INT UT_AddElement(IN SSortArray * pstArray, IN SArrayNode * pstArrNode);

/**This functions removes an element from the Array*/
INT UT_RemElement(IN SSortArray * pstArray, IN SArrayNode * pstArrNode);

/**This function searches a node using its key in the array*/
SArrayNode *UT_SearchElement(IN SSortArray * pstArray, IN U32 uiKey);

/**Deletes an element from the array pointer passed*/
void UT_DeleteSortedArray(IN SSortArray * pstArray);

/************************
Time Utilities
*************************/
INT UT_GetTimeOfDay(OUT STimeSpec *psTime);

typedef void (*PFNTimerFunc) (void *pvData);

/* timer's states */
typedef enum {
	eIDLE_T = 0,
	eSCHEDULE_T = 1,
	eRUNNING_T = 2
} ETimerState;

typedef struct _STimer {
	SLstNode stNode;
	SMutex stMutex;
	ETimerState eState;
	STimeSpec stTime;
	void (*pfnHdlrFunc) (void *);
	void *pvArg;
} STimer;

SThread * UT_TimerModuleInit(char *tname);
void UT_TimerShutDown(void);
STimer *UT_TimerCreate(void);
void UT_TimerInit(STimer *pstTimer);
void UT_TimerStartMS(STimer *pstTimer, UINT uimSec, PFNTimerFunc pfnTimerFunc, void *pvArg);
void UT_TimerStart(STimer *pstTimer, UINT uiSec, PFNTimerFunc pfnTimerFunc, void *pvArg);
void __UT_TimerStop(STimer *pstTimer);
void UT_TimerStop(STimer *pstTimer);
void __UT_TimerStopSync(STimer *pstTimer);
void UT_TimerStopSync(STimer *pstTimer);
void __UT_TimerDelete(STimer *pstTimer);
void UT_TimerDelete(STimer * pstTimer);

/** It updates host mac address against eth name*/
int UT_GetInterfaceMac(IN char *pcEthName, char *pcHostMacAddr);

#define UT_ARRAY_SIZE(x) (sizeof(x)/sizeof(x[0]))


/*****************************
 VAPI Logging Functions
 *****************************/
#ifndef INFO
#define INFO	1
#endif
#ifndef DEBUG
#define DEBUG   2
#endif
#ifndef PACKET
#define PACKET  3
#endif
#ifndef WARN
#define WARN	4
#endif

/**Special severity level*/
#ifndef ALWYS
#define ALWYS	5		/*Should not be used for setting debug level in array */
#endif

#ifndef NODBG
#define NODBG   6		/*Should not be used with UT_Log calls */
#endif

/**Debug levels for VAPI modules*/
#ifndef NO_DEBUG
#define NO_DEBUG		0
#endif

#ifndef API_LEVEL
#define API_LEVEL		1
#endif
#ifndef API_PACKET_LEVEL
#define API_PACKET_LEVEL	2
#endif
#ifndef VAPI_INFO
#define VAPI_INFO		3
#endif
#ifndef VAPI_GTL_INFO
#define VAPI_GTL_INFO		4
#endif
#ifndef VAPI_UT
#define VAPI_UT			5
#endif
#ifndef ALL_INFO
#define ALL_INFO		6
#endif

/**Module Names used in case of logging*/
#define MAX_MODULES	5
#define APPITF		0
#define VCORE		1
#define GTL		2
#define UT		3
#define TEST		4

void UT_SetDebugLevel(IN U8 ucVapiDebugLevel, IN U8 ucAppDebugLevel);

#ifdef VAPI_NODBG

#if !defined(VXWORKS) || defined(_WRS_GNU_VAR_MACROS)
#define UT_Log(agrs...)
#else
#define UT_Log(...)
#endif

#else

void UT_Log(IN U8 ucModule, IN U8 ucSeverity, IN char *pstrFormat, ...);

#endif				/* VAPI_NODBG */

void UT_ErrorLog(IN U8 ucModule, IN char *pstrFormat, ...);

void UT_VErrorLog(IN U8 ucModule, IN char *pstrFormat, va_list args);

#ifdef __cplusplus
};
#endif

#endif /* _UT_H */
