/*! \file ut.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "ut.h"

/*Semaphore to protect the Global aucSeverity_g*/
SMutex stDebugSem_g;
static UT_Status ut_state = UT_CLOSED;

/*APPITF,VCORE,GTL,UT,TEST*/
U8 aucSeverity_g[MAX_MODULES] = { INFO, PACKET, NODBG, NODBG, NODBG };

void *__UT_malloc(int iSize, char *fName, int lineNo)
{
	void *ptr = NULL;

	if (iSize <= 0)
	{
		UT_Log(UT, INFO, "Warning: called malloc() for 0 size from %s[%d]\n", fName, lineNo);
		return NULL;
	}
	ptr = UT_malloc(iSize);
	if (NULL == ptr)
	{
		UT_Log(UT, INFO, "Warning: malloc() returned NULL pointer. Called from %s[%d]\n", fName, lineNo);
	}
	return ptr;
}

void __UT_free(void *ptr, char *fName, int lineNo)
{
	void **p = (void **)ptr;
	if (!(*p))
	{
		UT_Log(UT, INFO, "Warning: called free() for NULL pointer from %s[%d]\n", fName, lineNo);
		return;
	}
	UT_free(*p);
	*p = NULL;
	return;
}

#ifdef MEMDBG

#ifdef MEMDBG_CORRUPT
#define MEM_PAD_PATTERN 0xE8
#define MEM_PAD_SIZE 24

U8 aucMemPattern_g[MEM_PAD_SIZE] = { 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8,
	0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8,
	0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8
};

#endif				/* MEMDBG_CORRUPT */

#ifdef MEMDBG_SYNC
SMutex semMalloc_g;
#endif				/* MEMDBG_SYNC */

#ifdef MEMDBG_LEAKS
#define MEMINFO_ARR_SIZE 1500
#define MAX_SRC_NAME 84

typedef struct _SMallocInfo {
	S32 iSize;		/*Size of memory block allocated */
	S32 iIdx;
	S32 iLineNo;
	char strFileName[MAX_SRC_NAME];
} SMallocInfo;

int iMaxIdx_g;
int iAllocCtr_g;
int iFreeCtr_g;

SMallocInfo *apstMemInfo_g[MEMINFO_ARR_SIZE];
#endif				/* MEMDBG_LEAKS */

#endif				/* MEMDBG */

/*Interface Function Definitions*/

/***************************************************************************
 * UT_Init: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function initializes the utility module.\n
 *
 *  Assumptions: 
 *      None\n
 *
 *  \return 
 *   SUCCESS    if UT is initialized properly.\n
 *   FAILURE    if UT is couldn't be initialized.\n
 */
INT UT_Init()
{
	if (ut_state != UT_CLOSED)
		return -1;
	ut_state = UT_INITIALIZING;
#ifdef MEMDBG

#ifdef MEMDBG_LEAKS
	iAllocCtr_g = 0;
	iFreeCtr_g = 0;
	iMaxIdx_g = 0;
	UT_MemSet(apstMemInfo_g, 0, sizeof(apstMemInfo_g));
#endif				/* MEMDBG_LEAKS */

#ifdef MEMDBG_SYNC
	if (UT_MutexInit(&semMalloc_g) < 0)
		goto err0;
#endif				/* MEMDBG_SYNC */

#endif				/* MEMDBG */

	if (UT_MutexInit(&stDebugSem_g) < 0)
		goto err1;

	ut_state = UT_INITIALIZED;
	if (UT_OSWInit() < 0)
		goto err2;

	return 0;

      err2:
	UT_MutexDestroy(&stDebugSem_g);

      err1:
#ifdef MEMDBG
#ifdef MEMDBG_SYNC
	UT_MutexDestroy(&semMalloc_g);
      err0:
#endif
#endif
	ut_state = UT_CLOSED;
	return -1;
}

void UT_Close()
{
	if (ut_state != UT_INITIALIZED)
		return;
	ut_state = UT_CLOSING;

	UT_OSWClose();

	UT_MutexDestroy(&stDebugSem_g);

#ifdef MEMDBG

#ifdef MEMDBG_SYNC
	UT_MutexDestroy(&semMalloc_g);
#endif

#ifdef MEMDBG_LEAKS
	UT_DumpMemInfo();
#endif				/* MEMDBG_LEAKS */

#endif				/* MEMDBG */
	ut_state = UT_CLOSED;
}

/***************************************************************************
 * AVL_Cmpint: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function compares the two node of the AVL tree. 
 *
 *  \return
 *   Difference  of the Keys of the two nodes passed\n
 *
 *  \param  pvA void pointer to the node of the AVL node.
 *  \param  pvB void pointer to the node of the AVL node.
 */
static INT AVL_Cmpint(IN void *pvA, IN void *pvB)
{
	return ((SAvlNode *) pvA)->uiKey - ((SAvlNode *) pvB)->uiKey;
}

/***************************************************************************
 * UT_CreateAVLTree: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function allocates the memory for the AVL tree.\n
 *  2.Initializes the semaphore attached to the tree.\n
 *
 *  \return
 *  SAVLTree *   Pointer to the newly created AVL tree.\n
 *  NULL         In case if memory couldn't be allocaed.\n
 */

SAVLTree *UT_CreateAVLTree()
{
	SAVLTree *pstAvlTree;

	pstAvlTree = (SAVLTree *) UT_AllocMem(sizeof(SAVLTree));
	if (pstAvlTree == NULL)
	{
		goto err0;
	}
	UT_MemSet(pstAvlTree, 0, sizeof(SAVLTree));

	pstAvlTree->stAvlTree.pfnCompare = AVL_Cmpint;

	if (UT_MutexInit(&pstAvlTree->stSem) < 0)
	{
		UT_ErrorLog(UT, "UT_CreateAVLTree : MutexInit failed so tree can not be created");
		goto err1;
	}

	return pstAvlTree;

      err1:
	UT_FreeMem(pstAvlTree);

      err0:
	return NULL;
}

/***************************************************************************
 * UT_AddToAVLTree: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function adds the data passed to it in the AVL tree.\n
 *  2.It checks the uniqueness of the key and only after that it adds the\n
 *    the data to the tree.\n
 *  3.It creates an AVLNode and adds ths data to it and then adds in the tree\n
 *
 *  \return
 *   SUCCESS    If the node is successfully added to the tree\n
 *   FAILURE    If the node couldn't be added to the tree\n
 *
 *  \param  pstAVLTree  Pointer to the AVL tree.
 *  \param  uiKey       The key associated to a node in the tree.
 *  \param  pvData      Pointer to the data to be added in the AVL tree.
 */

INT UT_AddToAVLTree(IN SAVLTree * pstAVLTree, IN U32 uiKey, IN void *pvData)
{
	SAvlNode *pstAvlNode;

	pstAvlNode = UT_AllocMem(sizeof(SAvlNode));
	if (pstAvlNode == NULL)
		goto err0;

	UT_MutexLock(&pstAVLTree->stSem);

	pstAvlNode->uiKey = uiKey;
	pstAvlNode->pvData = pvData;

	if (__AVL_Search(&pstAVLTree->stAvlTree, &pstAvlNode->stAvl) != NULL)
	{
		UT_MutexUnLock(&pstAVLTree->stSem);
		UT_Log(UT, WARN, "The key =% d is already present in the AVL Tree\n", uiKey);

		goto err1;
	}

	__AVL_Insert(&pstAVLTree->stAvlTree, (SAVL *) pstAvlNode);
	UT_MutexUnLock(&pstAVLTree->stSem);

	return 0;

      err1:
	UT_FreeMem(pstAvlNode);

      err0:
	return -1;
}

/***************************************************************************
 * UT_RemFromAVLTree: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function removes an element from the AVL tree.\n
 *  2. It searches if the key is present if yes it removes the node and if \n
 *     no it logs the message in log file. 
 *  
 *  \return
 *   None\n
 *
 *  \param  pstAVLTree  Pointer to the AVL tree.
 *  \param  uiKey       The key associated to a node in the tree.
 */

void UT_RemFromAVLTree(IN SAVLTree * pstAVLTree, IN U32 uiKey)
{
	SAvlNode *pstAvlNode, stAvlTemp;

	stAvlTemp.uiKey = uiKey;

	UT_MutexLock(&pstAVLTree->stSem);

	if ((pstAvlNode = (SAvlNode *) __AVL_Search(&pstAVLTree->stAvlTree, &stAvlTemp.stAvl)) == NULL)
	{
		UT_Log(UT, WARN, "The key =% d is not present in the AVL Tree.\n", uiKey);
		goto err_unlock;
	}

	__AVL_Remove(&pstAVLTree->stAvlTree, &pstAvlNode->stAvl);

	UT_MutexUnLock(&pstAVLTree->stSem);

	UT_FreeMem(pstAvlNode);

	return;

      err_unlock:
	UT_MutexUnLock(&pstAVLTree->stSem);
}

/***************************************************************************
 * UT_SearchInAVLTree: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function searches the node in the tree acoording to the key passed\n
 *  
 *  Assumptions:\n
 *  There is only one node in the tree corresponding to the key.\n
 *
 *  \return
 *   Pointer to the the node in case it is present\n
 *   NULL   In case element is not present in the AVL tree.\n
 *
 *  \param  pstAVLTree  Pointer to the AVL tree.
 *  \param  uiKey       The key associated to a node in the tree.
 */
void *UT_SearchInAVLTree(IN SAVLTree * pstAVLTree, IN U32 uiKey)
{
	SAvlNode stAvlNode;
	SAvlNode *pstTemp;

	UT_MutexLock(&pstAVLTree->stSem);

	stAvlNode.uiKey = uiKey;
	pstTemp = (SAvlNode *) __AVL_Search(&pstAVLTree->stAvlTree, &stAvlNode.stAvl);

	UT_MutexUnLock(&pstAVLTree->stSem);

	if (pstTemp == NULL)
	{
		return NULL;
	}

	return pstTemp->pvData;
}

/***************************************************************************
 * UT_DeleteAVLTree: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This functions deletes the AVL tree by freeing the memory allotted to\n
 *     it.\n
 *
 *  \return
 *   None\n
 *
 *  \param  pstAVLTree pointer to the AVL tree to be deleted.
 */
void UT_DeleteAVLTree(IN SAVLTree * pstAVLTree)
{
	SAVL *pstTempNode;

	UT_MutexLock(&pstAVLTree->stSem);

	while (pstAVLTree->stAvlTree.pstRoot)
	{
		pstTempNode = pstAVLTree->stAvlTree.pstRoot;
		__AVL_RemoveRoot(&pstAVLTree->stAvlTree);
		UT_FreeMem(pstTempNode);
	}

	UT_MutexUnLock(&pstAVLTree->stSem);

	UT_MutexDestroy(&pstAVLTree->stSem);
	UT_FreeMem(pstAVLTree);

	return;
}

/***************************************************************************
 * UT_ListInit: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function allocates the meory to the lined list's pointer\n
 *  2.Initializes a semaphore and attaches to the linked list's node\n
 *
 *  \return
 *  Pointer to the linked list.
 *
 *
 */
INT UT_ListInit(SLinkedList * pstLinkedList)
{
	if (UT_MutexInit(&pstLinkedList->stSem) < 0)
	{
		UT_ErrorLog(UT, "UT_ListInit : MutexInit failed so List can not be created");
		goto err;
	}

	pstLinkedList->stHead.pstNext = &pstLinkedList->stHead;
	pstLinkedList->stHead.pstPrev = &pstLinkedList->stHead;

	return 0;

      err:
	return -1;
}


/***************************************************************************
 * __UT_AddToFront: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function adds an element in front of a prelocked  linked list.\n
 *
 *  \return
 *   None\n
 *
 *  \param pstLinkedList    Pointer to the Linked List.
 *  \param pstNode          Pointer to the node to be added.
 */
void __UT_AddToFront(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode)
{
	pstNode->pstNext = pstLinkedList->stHead.pstNext;
	pstNode->pstPrev = &pstLinkedList->stHead;
	pstLinkedList->stHead.pstNext = pstNode;
	pstNode->pstNext->pstPrev = pstNode;
}

/***************************************************************************
 * UT_AddToFront: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function adds an element in front of the linked list.\n
 *  2.It lock the list then calls __UT_AddToFront to add the node to the list \n
 *
 *  \return
 *   None\n
 *
 *  \param pstLinkedList    Pointer to the Linked List.
 *  \param pstNode          Pointer to the node to be added.
 */
void UT_AddToFront(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode)
{
	UT_MutexLock(&pstLinkedList->stSem);

	__UT_AddToFront(pstLinkedList, pstNode);

	UT_MutexUnLock(&pstLinkedList->stSem);
}

/***************************************************************************
 * __UT_AddToTail: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function adds an element to the tail of a prelocked  linked list.\n
 *
 *  \return
 *   None\n
 *
 *  \param pstLinkedList    Pointer to the Linked List.
 *  \param pstNode          Pointer to the node to be added.
 */
void __UT_AddToTail(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode)
{
	pstNode->pstNext = &pstLinkedList->stHead;
	pstNode->pstPrev = pstLinkedList->stHead.pstPrev;
	pstLinkedList->stHead.pstPrev = pstNode;
	pstNode->pstPrev->pstNext = pstNode;
}

/***************************************************************************
 * UT_AddToTail: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function adds an element to the tail of the linked list.\n
 *  2.It locks the list then calls __UT_AddToTail to add the node to the list \n
 *
 *  \return
 *   None\n 
 *
 *  \param pstLinkedList    Pointer to the Linked List.
 *  \param pstNode          Pointer to the node to be added.
 */
void UT_AddToTail(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode)
{
	UT_MutexLock(&pstLinkedList->stSem);

	__UT_AddToTail(pstLinkedList, pstNode);

	UT_MutexUnLock(&pstLinkedList->stSem);
}

/***************************************************************************
 * __UT_ListAddBefore: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function adds an element before defined one of a prelocked linked list.\n
 *
 *  \return
 *   None\n
 *
 *  \param pstBaseNode		Pointer to the defined node before which pstNode to be added.
 *  \param pstNode     	Pointer to the node to be added.
 */
void __UT_ListAddBefore(SLstNode * pstBaseNode, SLstNode * pstNode)
{
	pstNode->pstNext = pstBaseNode;
	pstNode->pstPrev = pstBaseNode->pstPrev;
	pstBaseNode->pstPrev->pstNext = pstNode;
	pstBaseNode->pstPrev = pstNode;
}

/***************************************************************************
 * UT_ListAddBefore: The function does the following things -
 ***************************************************************************/
/*!
 *  1. This function adds an element before defined one of the linked list.\n
 *  2. It locks the list then calls __UT_AddBefore to add the node to the list \n
 *
 *  \return
 *   None\n 
 *
 *  \param pstLinkedList	Pointer to the Linked List.
 *  \param pstBaseNode		Pointer to the defined node before which pstNode to be added.
 *  \param pstNode          	Pointer to the node to be added.
 */
void UT_ListAddBefore(IN SLinkedList * pstLinkedList, IN SLstNode * pstBaseNode, IN SLstNode * pstNode)
{
	UT_MutexLock(&pstLinkedList->stSem);

	__UT_ListAddBefore(pstBaseNode, pstNode);

	UT_MutexUnLock(&pstLinkedList->stSem);
}

/***************************************************************************
 * __UT_ListAddAfter: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function adds an element after defined one of a prelocked linked list.\n
 *
 *  \return
 *   None\n
 *
 *  \param pstBaseNode		Pointer to the defined node after which pstNode to be added.
 *  \param pstNode		Pointer to the node to be added.
 */
void __UT_ListAddAfter(SLstNode * pstBaseNode, SLstNode * pstNode)
{
	pstNode->pstNext = pstBaseNode->pstNext;
	pstNode->pstPrev = pstBaseNode;
	pstBaseNode->pstNext->pstPrev = pstNode;
	pstBaseNode->pstNext = pstNode;
}

/***************************************************************************
 * UT_ListAddAfter: The function does the following things -
 ***************************************************************************/
/*!
 *  1. This function adds an element after defined one of the linked list.\n
 *  2. It locks the list then calls __UT_AddAfter to add the node to the list \n
 *
 *  \return
 *   None\n
 *
 *  \param pstLinkedList	Pointer to the Linked List.
 *  \param pstBaseNode		Pointer to the defined node after which pstNode to be added.
 *  \param pstNode		Pointer to the node to be added.
 */
void UT_ListAddAfter(IN SLinkedList * pstLinkedList, IN SLstNode * pstBaseNode, IN SLstNode * pstNode)
{
	UT_MutexLock(&pstLinkedList->stSem);

	__UT_ListAddAfter(pstBaseNode, pstNode);

	UT_MutexUnLock(&pstLinkedList->stSem);
}

/***************************************************************************
 * __UT_RemoveNode: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function removes an element from the prelocked linked list.\n
 *
 *  \return
 *   None\n
 *
 *  \param pstLinkedList    Pointer to the Linked List.
 *  \param pstNode          Pointer to the node to be added.
 */
void __UT_RemoveNode(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode)
{
	pstNode->pstNext->pstPrev = pstNode->pstPrev;
	pstNode->pstPrev->pstNext = pstNode->pstNext;
	pstNode->pstNext = NULL;
	pstNode->pstPrev = NULL;
}

/***************************************************************************
 * UT_RemoveNode: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function removes an element from the linked list.\n
 *  2.It locks the list then calls __UT_RemoveNode to remove the node to the list \n
 *
 *  \return
 *   None\n 
 *
 *  \param pstLinkedList    Pointer to the Linked List.
 *  \param pstNode          Pointer to the node to be added.
 */
void UT_RemoveNode(IN SLinkedList * pstLinkedList, IN SLstNode * pstNode)
{
	UT_MutexLock(&pstLinkedList->stSem);

	__UT_RemoveNode(pstLinkedList, pstNode);

	UT_MutexUnLock(&pstLinkedList->stSem);
}

/***************************************************************************
 * UT_GetFrontNode: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function gets the front node of the linked list.\n
 *  2.If the linked list is empty then it returns NULL\n
 *
 *  \return
 *   Pointer to the first node in case it is present.\n
 *   NULL   in case linked list is empty.\n
 *
 *  \param pstLinkedList    Pointer to the Linked List.
 */
SLstNode *__UT_GetFrontNode(IN SLinkedList * pstLinkedList)
{
	if (__UT_IsListEmpty(pstLinkedList))
		return NULL;
	else
		return pstLinkedList->stHead.pstNext;
}

SLstNode *UT_GetFrontNode(IN SLinkedList * pstLinkedList)
{
	SLstNode *pstTempNode;

	UT_MutexLock(&pstLinkedList->stSem);

	pstTempNode = __UT_GetFrontNode(pstLinkedList);

	UT_MutexUnLock(&pstLinkedList->stSem);

	return pstTempNode;
}

/***************************************************************************
 * UT_IsListEmpty: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function checks if the Linked list is empty or not.\n
 *
 *  \return
 *   True   If the linked list is empty.\n
 *   False  If the linked list is not empty.\n
 *
 *  \param pstLinkedList    Pointer to the Linked List.
 */
Boolean __UT_IsListEmpty(IN SLinkedList * pstLinkedList)
{
	if (pstLinkedList->stHead.pstNext == &pstLinkedList->stHead)
		return True;
	else
		return False;
}


Boolean UT_IsListEmpty(IN SLinkedList * pstLinkedList)
{
	Boolean rc;

	UT_MutexLock(&pstLinkedList->stSem);

	rc = __UT_IsListEmpty(pstLinkedList);

	UT_MutexUnLock(&pstLinkedList->stSem);

	return rc;
}

/***************************************************************************
 * UT_ListDestroy: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function deletes the linked list.\n
 *  2.It also frees the memory allocated to the semaphore.\n
 *
 *  \return
 *   None\n
 *
 *  \param pstLinkedList    Pointer to the Linked List.
 */
INT UT_ListDestroy(IN SLinkedList * pstLinkedList)
{
	return UT_MutexDestroy(&pstLinkedList->stSem);
}

/***************************************************************************
 * UT_CreateSortedArray: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function creates a sorted array\n
 *  2. Initializes the semaphore assigned to it.\n
 *
 *  \return
 *   Pointer to the newly created sorted array.\n
 *
 *  \param uiSize   Size of the Array to be created. 
 */
SSortArray *UT_CreateSortedArray(IN UINT uiSize)
{
	SSortArray *pstTempArr;

	pstTempArr = UT_AllocMem(sizeof(SSortArray));
	if (pstTempArr == NULL)
	{
		goto err0;
	}

	pstTempArr->pastStart = UT_Calloc(uiSize, sizeof(SArrayNode *));
	if (pstTempArr->pastStart == NULL)
	{
		goto err1;
	}

	pstTempArr->uiSize = uiSize;
	pstTempArr->uiNumElement = 0;

	if (UT_MutexInit(&pstTempArr->stSem) < 0)
	{
		UT_ErrorLog(UT, "UT_CreateSortedArray : SemInit failed so array can not be created");
		goto err2;
	}

	return pstTempArr;

      err2:
	UT_FreeMem(pstTempArr->pastStart);

      err1:
	UT_FreeMem(pstTempArr);

      err0:
	return NULL;
}

/***************************************************************************
 * __UT_SortArray: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function bubble sorts the array pssed to it. 
 *
 *  \return
 *   None\n
 *
 *  \param  pstArray    Pointer to the array.
 */
static void __UT_SortArray(IN SSortArray * pstSortArray)
{
	INT iCount1, iCount2;
	SArrayNode *pstTempNode, **pstTempArr = pstSortArray->pastStart;

	for (iCount1 = 0; iCount1 < pstSortArray->uiNumElement - 1; iCount1++)
	{
		for (iCount2 = 0; iCount2 < pstSortArray->uiNumElement - 1 - iCount1; iCount2++)
		{
			if (!pstTempArr[1 + iCount2])
				continue;

			if ((pstTempArr[iCount2] == NULL) || pstTempArr[1 + iCount2]->uiKey < pstTempArr[iCount2]->uiKey)
			{
				pstTempNode = pstTempArr[iCount2];
				pstTempArr[iCount2] = pstTempArr[1 + iCount2];
				pstTempArr[1 + iCount2] = pstTempNode;
			}
		}
	}
}

/***************************************************************************
 * UT_AddElement: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function adds the node in the array.\n
 *  2.Checks if the node passed to is empty or array is not present.\n
 *  3.If the size of the array exceeds the memory allocated to it then it\n
 *    returns the failure.\n
 *  4.It adds the element in the end and then bubble sorts the array.\n
 *
 *  \return
 *   SUCCESS    if the node is successfully added.\n
 *   FAILURE    if the node couldn't be entered.\n
 *
 *  \param  pstArray    Pointer to the array.
 *  \param  pstArrNode  Pointer to the node to be added to the node.
 */
INT UT_AddElement(IN SSortArray * pstArray, IN SArrayNode * pstArrNode)
{
	UT_MutexLock(&pstArray->stSem);

	if (pstArray->uiSize <= pstArray->uiNumElement)
	{
		UT_Log(UT, WARN, "Array passed is already full");
		goto err_unlock;
	}

	pstArray->pastStart[pstArray->uiNumElement] = pstArrNode;
	pstArray->uiNumElement++;

	__UT_SortArray(pstArray);

	UT_MutexUnLock(&pstArray->stSem);

	return 0;

      err_unlock:
	UT_MutexUnLock(&pstArray->stSem);
	return -1;
}

/***************************************************************************
 * __UT_SearchElementIndex: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function searches a node using its key in the array.\n
 *
 *  \return
 *   Pointer to the node in case it is present.\n
 *   NULL in case of array is not present.\n
 *
 *  \param  uiKey       Key to be searched in the Array.
 *  \param  pstArray    Pointer to the array.
 */
static INT __UT_SearchElementIndex(IN SSortArray * pstArray, IN U32 uiKey)
{
	INT iLow, iHigh, iMid;

	/*Binary Search */
	iLow = 0;
	iHigh = pstArray->uiNumElement - 1;

	while (iLow <= iHigh)
	{
		iMid = (iLow + iHigh) / 2;
		if (uiKey == pstArray->pastStart[iMid]->uiKey)
			goto match;

		if (uiKey > pstArray->pastStart[iMid]->uiKey)
		{
			iLow = iMid + 1;
		}
		else
		{
			if (!iMid)
				break;

			iHigh = iMid - 1;
		}
	}

	return -1;

      match:
	return iMid;
}


/***************************************************************************
 * UT_RemElement: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function removes a node from the array.\n
 *  2.It assigns some very big value to the key and then bubble sorts\n
 *    the array as a reult it reaches the end of the array.\n
 *
 *  \return
 *   SUCCESS    if the node is successfully added.\n
 *   FAILURE    if the node couldn't be entered.\n
 *
 *  \param  pstArray    Pointer to the array.
 *  \param  pstArrNode  Pointer to the node to be added to the node.
 */
INT UT_RemElement(IN SSortArray * pstArray, IN SArrayNode * pstArrNode)
{
	INT iIndex;

	UT_MutexLock(&pstArray->stSem);

	iIndex = __UT_SearchElementIndex(pstArray, pstArrNode->uiKey);

	if (iIndex < 0)
	{
		UT_ErrorLog(UT, "UT_RemElement: Element not found\n");
		goto err_unlock;
	}

	pstArray->pastStart[iIndex] = NULL;

	__UT_SortArray(pstArray);

	pstArray->uiNumElement--;

	UT_MutexUnLock(&pstArray->stSem);

	return 0;

err_unlock:
	UT_MutexUnLock(&pstArray->stSem);

	return -1;
}


/***************************************************************************
 * UT_SearchElement: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function searches a node using its key in the array.\n
 *
 *  \return
 *   Pointer to the node in case it is present.\n
 *   NULL in case of array is not present.\n
 *
 *  \param  uiKey       Key to be searched in the Array.
 *  \param  pstArray    Pointer to the array.
 */
SArrayNode *UT_SearchElement(IN SSortArray * pstArray, IN U32 uiKey)
{
	SArrayNode *pstArrNode;
	INT iIndex;

	UT_Log(UT, INFO, "UT_SearchElement Entry\n");

	UT_MutexLock(&pstArray->stSem);

	iIndex = __UT_SearchElementIndex(pstArray, uiKey);

	if (iIndex < 0)
	{
		UT_Log(UT, INFO, "UT_SearchElement Exit - Failure\n");
		pstArrNode = NULL;
	}
	else
	{
		pstArrNode = pstArray->pastStart[iIndex];

		UT_Log(UT, INFO, "UT_SearchElement Exit - SUCCESS\n");
	}

	UT_MutexUnLock(&pstArray->stSem);

	return pstArrNode;
}

/***************************************************************************
 * UT_DeleteSortedArray: The function does the following things -
 ***************************************************************************/
/*!
 *  1.This function deletes the semaphore assigned to the array.\n
 *  2.Then it deletes the array pointer by freeing the memory.\n
 *  
 *  \return
 *   None\n
 *
 *  \param  pstArray    Pointer to the array.
 */
void UT_DeleteSortedArray(IN SSortArray * pstArray)
{
	UT_MutexDestroy(&pstArray->stSem);
	UT_FreeMem(pstArray->pastStart);
	UT_FreeMem(pstArray);
}

#ifndef VAPI_NODBG
void UT_Log(IN U8 ucModule, IN U8 ucSeverity, IN char *pstrFormat, ...)
{
	va_list args;

	if (ucSeverity < aucSeverity_g[ucModule])
		return;

	UT_va_start(args, pstrFormat);
	UT_VErrorLog(ucModule, pstrFormat, args);
	UT_va_end(args);
}
#endif

void UT_ErrorLog(IN U8 ucModule, IN char *pstrFormat, ...)
{
	va_list args;
	if (ut_state != UT_INITIALIZED)
		return;
	UT_va_start(args, pstrFormat);
	UT_VErrorLog(ucModule, pstrFormat, args);
	UT_va_end(args);
}

void UT_SetDebugLevel(U8 ucVapiDebugLevel, U8 ucAppDebugLevel)
{
	if (ut_state != UT_INITIALIZED)
		return;

	UT_MutexLock(&stDebugSem_g);

	switch (ucVapiDebugLevel)
	{
	case NO_DEBUG:
		aucSeverity_g[0] = NODBG;
		aucSeverity_g[1] = NODBG;
		aucSeverity_g[2] = NODBG;
		aucSeverity_g[3] = NODBG;
		aucSeverity_g[4] = ucAppDebugLevel;
		break;

		/* For ERRORS we have different Error path i.e UT_ErrorLog() 
		   case ALL_ERRORS:
		   aucSeverity_g[0] = ERRORS;
		   aucSeverity_g[1] = ERRORS;
		   aucSeverity_g[2] = ERRORS;
		   aucSeverity_g[3] = ERRORS; 
		   aucSeverity_g[4] = ucAppDebugLevel;
		   break;
		 */
	case API_LEVEL:
		aucSeverity_g[0] = INFO;

		aucSeverity_g[1] = NODBG;
		aucSeverity_g[2] = NODBG;
		aucSeverity_g[3] = NODBG;
		aucSeverity_g[4] = ucAppDebugLevel;
		break;

	case API_PACKET_LEVEL:
		aucSeverity_g[0] = INFO;
		aucSeverity_g[1] = PACKET;
		aucSeverity_g[2] = NODBG;
		aucSeverity_g[3] = NODBG;
		aucSeverity_g[4] = ucAppDebugLevel;
		break;

	case VAPI_INFO:
		aucSeverity_g[0] = INFO;
		aucSeverity_g[1] = INFO;
		aucSeverity_g[2] = NODBG;
		aucSeverity_g[3] = NODBG;
		aucSeverity_g[4] = ucAppDebugLevel;
		break;

	case VAPI_GTL_INFO:
		aucSeverity_g[0] = INFO;
		aucSeverity_g[1] = INFO;
		aucSeverity_g[2] = INFO;
		aucSeverity_g[3] = NODBG;
		aucSeverity_g[4] = ucAppDebugLevel;
		break;

	case VAPI_UT:
		aucSeverity_g[0] = INFO;
		aucSeverity_g[1] = NODBG;
		aucSeverity_g[2] = NODBG;
		aucSeverity_g[3] = INFO;
		aucSeverity_g[4] = ucAppDebugLevel;
		break;

	case ALL_INFO:
		aucSeverity_g[0] = INFO;
		aucSeverity_g[1] = INFO;
		aucSeverity_g[2] = INFO;
		aucSeverity_g[3] = INFO;
		aucSeverity_g[4] = ucAppDebugLevel;
		break;
	}

	UT_MutexUnLock(&stDebugSem_g);
}

#ifdef MEMDBG
void *UT_AllocMemDbg(char *strFileName, INT iLineNo, INT iMemSize)
{
#ifdef MEMDBG_LEAKS
	SMallocInfo *pstMem;
	INT i;
#else
	void *pstMem;
#endif
	INT iAllocSize = iMemSize;

	if (iMemSize == 0)
	{
		UT_ErrorLog(UT, "UT_AllocMemDbg: 0 bytes being allocated from %s[%d]\n", strFileName, iLineNo);
		return NULL;
	}

	if (ut_state != UT_INITIALIZED)
		return NULL;
#ifdef MEMDBG_SYNC
	UT_MutexLock(&semMalloc_g);
#endif

#ifdef MEMDBG_CORRUPT
	iAllocSize += MEM_PAD_SIZE;
#endif

#ifdef MEMDBG_LEAKS
	iAllocSize += sizeof(SMallocInfo);
#endif

	pstMem = __UT_malloc(iAllocSize, strFileName, iLineNo);
	if (pstMem == NULL)
		goto out;

#ifdef MEMDBG_LEAKS
	iAllocCtr_g += iMemSize;
	for (i = 0; i < MEMINFO_ARR_SIZE; i++)
	{
		if (apstMemInfo_g[i] == NULL)
		{
			apstMemInfo_g[i] = pstMem;
			pstMem->iSize = iMemSize;
			pstMem->iIdx = i;
			pstMem->iLineNo = iLineNo;
			strncpy(pstMem->strFileName, strFileName, MAX_SRC_NAME - 1);
			pstMem->strFileName[MAX_SRC_NAME - 1] = 0;
			if (i > iMaxIdx_g)
				iMaxIdx_g = i;

			break;
		}
	}
	pstMem = (SMallocInfo *) (((U8 *) pstMem) + sizeof(SMallocInfo));
#endif

#ifdef MEMDBG_CORRUPT
	UT_MemSet(((U8 *) pstMem) + iMemSize, MEM_PAD_PATTERN, MEM_PAD_SIZE);
#endif

      out:
#ifdef MEMDBG_SYNC
	UT_MutexUnLock(&semMalloc_g);
#endif

	return pstMem;
}

void UT_FreeMemDbg(char *strFileName, S32 iLineNo, void *pvMem)
{
	void **ptr = (void **)pvMem;
#ifdef MEMDBG_LEAKS
	SMallocInfo *pstMem;
	INT i;
#endif
	if (ut_state != UT_INITIALIZED)
		return;

	if (*ptr == NULL)
		return;

#ifdef MEMDBG_LEAKS
	pstMem = (SMallocInfo *) (((U8 *) (*ptr)) - sizeof(SMallocInfo));
#endif

#ifdef MEMDBG_SYNC
	UT_MutexLock(&semMalloc_g);
#endif

#ifdef MEMDBG_LEAKS
	for (i = 0; i < MEMINFO_ARR_SIZE; i++)
	{
		if (pstMem == apstMemInfo_g[i])
		{
			iFreeCtr_g += pstMem->iSize;
			apstMemInfo_g[pstMem->iIdx] = NULL;
#ifdef MEMDBG_CORRUPT
			if (UT_MemCmp(((U8 *) pstMem) + sizeof(SMallocInfo) + pstMem->iSize,
				   aucMemPattern_g, MEM_PAD_SIZE) != 0)
			{
				UT_ErrorLog(UT, "Memory Block Corrupted: Allocated at %s[%d]\n",
					    pstMem->strFileName, pstMem->iLineNo);
			}
			else
			{
				UT_Log(UT, INFO, "Valid block being freed 0x%X from %s[%d]\n",
				       pstMem, strFileName, iLineNo);
			}
#endif
			__UT_free(&pstMem, strFileName, iLineNo);
			pstMem = NULL;
			break;
		}
	}
#else

#ifdef MEMDBG_CORRUPT
	if (UT_MemCmp
	    (((U8 *) (*ptr)) + (((*(((U32 *) (*ptr) - 1))) * 2 - 8) - MEM_PAD_SIZE), aucMemPattern_g,
	     MEM_PAD_SIZE) != 0)
	{
		int j;
		UT_ErrorLog(UT, "Corrupt block %#x getting freed at %s[%d]\n", *ptr, strFileName, iLineNo);
		for (j = 0; j < ((*(((U32 *) (*ptr) - 1))) * 2 - 8); j++)
		{
			UT_ErrorLog(UT, "%X ", *(((U8 *) (*ptr)) + j));
		}
		UT_ErrorLog(UT, "\n");
		*((U32 *) (0xFFFFFFFF)) = 0x0;
	}
#endif				/* MEMDBG_CORRUPT */

	__UT_free(pvMem, strFileName, iLineNo);
#endif				/* MEMDBG_LEAKS */

#ifdef MEMDBG_SYNC
	UT_MutexUnLock(&semMalloc_g);
#endif				/* MEMDBG_SYNC */

#ifdef MEMDBG_LEAKS
	if (pstMem != NULL)
	{
		UT_ErrorLog(UT, "Invalid block being freed 0x%X from %s[%d]\n", pstMem, strFileName, iLineNo);
	}
#endif				/* MEMDBG_LEAKS */
}

#ifdef MEMDBG_LEAKS

void UT_DumpMemInfo()
{
	INT i;

	UT_ErrorLog(UT, "Alloc counter = %d\n", iAllocCtr_g);
	UT_ErrorLog(UT, "Free counter = %d\n", iFreeCtr_g);
	UT_ErrorLog(UT, "Max Index = %d\n", iMaxIdx_g);

	for (i = 0; i < MEMINFO_ARR_SIZE; i++)
	{
		if (apstMemInfo_g[i] != NULL)
		{
			UT_ErrorLog(UT, "Block not freed = %d\n", i);
			UT_ErrorLog(UT, "Block Size = %d\n", apstMemInfo_g[i]->iSize);
			UT_ErrorLog(UT, "Allocated at %s[%d]\n", apstMemInfo_g[i]->strFileName,
				    apstMemInfo_g[i]->iLineNo);
		}
	}
}
#endif				/* MEMDBG_LEAKS */

#endif				/* MEMDBG */

void *UT_Calloc(INT num, INT size)
{
	void *value = UT_AllocMem(num * size);
	if (NULL != value)
		UT_MemSet(value, 0, num * size);	/* Initialise to 0 */

	return value;
}
