/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include <errno.h>
#include "ut.h"

static SSem *pstTimerSem;

static SLinkedList stTimerLinkedList;
static SThread *pstThread = NULL;
static char ShutdownThread = 0;

#define TIME_DIFF_MS(t1, t2) ((t1)->tv_sec - (t2)->tv_sec) * 1000 + ((t1)->tv_nsec - (t2)->tv_nsec) / 1000000

static int TimerAddSorted(SLinkedList *pstLinkedList, SLstNode *pstBaseNode, STimer *pstTimer)
{
	STimer *pstBaseTimer;

	pstBaseTimer = UT_ContainerOf(pstBaseNode, STimer, stNode);
	if (TIME_DIFF_MS(&pstTimer->stTime, &pstBaseTimer->stTime) >= 0)
	{
		__UT_ListAddAfter(pstBaseNode, &pstTimer->stNode);
		return SUCCESS;
	}	

	return FAILURE;
}

int __TimerRemove(STimer *pstTimer)
{
	if (eSCHEDULE_T == pstTimer->eState)
	{
		__UT_RemoveNode(&stTimerLinkedList, &pstTimer->stNode);
		pstTimer->eState = eIDLE_T;
		return SUCCESS;
	}

	return FAILURE;
}

static void *UT_TimerThread(void *pv)
{
	S32 iTimeOut;	
	STimer *pstTimer;
	SLstNode *pstNode;
	STimeSpec stCurrentTime;

	while (1)
	{
		if (UT_IsListEmpty(&stTimerLinkedList))
			UT_SemWait(pstTimerSem, INFINITE_TIMEOUT);

		if (ShutdownThread) {
			ShutdownThread = 0;
			break;
		}

		UT_MutexLock(&stTimerLinkedList.stSem);

		pstNode = __UT_GetFrontNode(&stTimerLinkedList);
		if (NULL == pstNode)
		{
			UT_MutexUnLock(&stTimerLinkedList.stSem);
			continue;
		}

		pstTimer = UT_ContainerOf(pstNode, STimer, stNode);
		UT_GetTimeOfDay(&stCurrentTime);
		iTimeOut = TIME_DIFF_MS(&pstTimer->stTime, &stCurrentTime);

		if (iTimeOut <= 0) 
		{
			__TimerRemove(pstTimer);
			pstTimer->eState = eRUNNING_T;
			
			UT_MutexLock(&pstTimer->stMutex); 
			UT_MutexUnLock(&stTimerLinkedList.stSem);

			pstTimer->pfnHdlrFunc(pstTimer->pvArg);

			if (eRUNNING_T == pstTimer->eState)
				pstTimer->eState = eIDLE_T;

			UT_MutexUnLock(&pstTimer->stMutex); 
		}
		else 
		{
			UT_MutexUnLock(&stTimerLinkedList.stSem);
			UT_SemWait(pstTimerSem, iTimeOut);
		}

		if (ShutdownThread) {
			ShutdownThread = 0;
			break;
		}
	}

	return NULL;
}

/***************************************************************************
 * UT_TimerModuleInit: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	Initialize linked list, timer's semafore, timer's thread \n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*
*	\n \b Returns:
*	\li SUCCESS
*	\li FAILURE
*	\li VAPI_ERR_DEVICE_NOT_INITIALIZED
*	\li VAPI_ERR_INVALID_CONNID
*	\li VAPI_ERR_DEV_IS_NOT_UP
*	\li VAPI_ERR_NOMEM
*
*/
SThread *UT_TimerModuleInit(char *tname)
{
	if (UT_ListInit(&stTimerLinkedList) != SUCCESS)
	{
		UT_ErrorLog(UT, "UT_TimerModuleInit: Unable to initialize linked list for timers");
		goto out;
	}

	pstTimerSem = UT_SemInit(0);
	if (pstTimerSem == NULL)
	{
		UT_ErrorLog(UT, "UT_TimerModuleInit: Unable to initialize timer's semafore");
		goto err1;
	}

	pstThread = UT_ThreadCreate(tname, UT_TimerThread, NULL);
	if (pstThread == NULL)
	{
		UT_ErrorLog(UT, "UT_TimerModuleInit: Unable to initialize timer's thread");
		goto err2;
	}

	return pstThread;

err2:
	UT_SemDestroy(pstTimerSem);
err1:
	UT_ListDestroy(&stTimerLinkedList);
out:
	return NULL;
}

/***************************************************************************
 * UT_TimerShutDown: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*	1. Stop all timers.\n
*	2. Free all datastrucutres.\n
*	3. Destroy timer's semafore, linked list.\n
*	4. Stop timer's thread.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*
*	\n \b Returns:
*	None. \n
*
*/
void UT_TimerShutDown()
{
	int i;
	SLstNode *pstNode;
	STimer *pstTimer = NULL;

	UT_MutexLock(&stTimerLinkedList.stSem);

	while((pstNode = __UT_GetFrontNode(&stTimerLinkedList)) != NULL)
	{
		pstTimer = UT_ContainerOf(pstNode, STimer, stNode);
		__UT_TimerDelete(pstTimer);
	}

	UT_MutexUnLock(&stTimerLinkedList.stSem);

	ShutdownThread = 1;
	UT_SemPost(pstTimerSem);
	for (i = 0; i < 3 && ShutdownThread != 0; i++)
		sleep(1);

	UT_SemDestroy(pstTimerSem);
	UT_ListDestroy(&stTimerLinkedList);

	UT_FreeMem(pstThread);
}

/***************************************************************************
 * UT_TimerStartMS: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	1. Connects the timer to the time handler function to be invoked 
* 	on expiry of timer.\n
* 	2. Sets the timer to number of msecs passed in the arguement.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTimer</td>
*		<td style="vertical-align: top;">pointer to timer sructure.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiMSec</td>
*		<td style="vertical-align: top;">time specified in ms after that timer to be expired.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnTimerFunc</td>
*		<td style="vertical-align: top;">pointer to handler to be run after timer having expired.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pvArg</td>
*		<td style="vertical-align: top;">argument to be passed for handler.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*  	 None.\n
*/
void UT_TimerStartMS(STimer *pstTimer, UINT uiMSec, PFNTimerFunc pfnTimerFunc, void *pvArg)
{
	STimeSpec stTime;
	STimer *pstBaseTimer;
	SLstNode *pstNode;
	
	UT_GetTimeOfDay(&stTime);
	stTime.tv_sec += uiMSec / 1000;
	stTime.tv_nsec += (uiMSec % 1000) * 1000 * 1000;

	pstTimer->stTime = stTime;	
	pstTimer->pfnHdlrFunc = pfnTimerFunc;
	pstTimer->pvArg = pvArg;

	UT_MutexLock(&stTimerLinkedList.stSem);

	__UT_TimerStop(pstTimer);

	if (__UT_IsListEmpty(&stTimerLinkedList))
		__UT_AddToFront(&stTimerLinkedList, &pstTimer->stNode);
	else 
	{
		/* front node */
		pstBaseTimer = UT_ContainerOf(stTimerLinkedList.stHead.pstNext, STimer, stNode);
		if (TIME_DIFF_MS(&pstTimer->stTime, &pstBaseTimer->stTime) <= 0)
		{
			__UT_AddToFront(&stTimerLinkedList, &pstTimer->stNode);
			goto out;
		}

		/* tail node */
		pstBaseTimer = UT_ContainerOf(stTimerLinkedList.stHead.pstPrev, STimer, stNode);		
		if (TIME_DIFF_MS(&pstTimer->stTime, &pstBaseTimer->stTime) >= 0)
		{
			__UT_AddToTail(&stTimerLinkedList, &pstTimer->stNode);
			goto out;
		}

		__UT_ListForEachReversed(pstNode, &stTimerLinkedList.stHead)
		{
			if (SUCCESS == TimerAddSorted(&stTimerLinkedList, pstNode, pstTimer))	
				goto out;
		}
	}

out:
	pstTimer->eState = eSCHEDULE_T;
	UT_MutexUnLock(&stTimerLinkedList.stSem);
	UT_Log(UT, INFO, "UT_TimerStartMS: signaling to timer thread\n");
	UT_SemPost(pstTimerSem);
}

/***************************************************************************
 * UT_TimerStart: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	1. Connects the timer to the time handler function to be invoked\n
* 	on expiry of timer.\n
* 	2. Sets the timer to number of secs passed in the arguement.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTimer</td>
*		<td style="vertical-align: top;">pointer to timer sructure.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">uiMSec</td>
*		<td style="vertical-align: top;">time specified in sec after that timer to be expired.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pfnTimerFunc</td>
*		<td style="vertical-align: top;">pointer to handler to be run after timer having expired.</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pvArg</td>
*		<td style="vertical-align: top;">argument to be passed for handler.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*  	 None.\n
*/
void UT_TimerStart(STimer *pstTimer, UINT uiSec, PFNTimerFunc pfnTimerFunc, void *pvArg)
{
	UT_TimerStartMS(pstTimer, uiSec*1000, pfnTimerFunc, pvArg);
}

/***************************************************************************
 * __UT_TimerStop: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	Stops the specified timer by removing from pre-locked list.\n  
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTimer</td>
*		<td style="vertical-align: top;">pointer to timer sructure.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*  	 None.\n
*/
void __UT_TimerStop(STimer *pstTimer)
{
	UT_Log(UT, INFO, "UT_TimerStop: ENTRY\n");

	if (SUCCESS == __TimerRemove(pstTimer))
	{
		UT_Log(UT, INFO, "UT_TimerStop: signaling to timer thread\n");
		UT_SemPost(pstTimerSem);
	}

	UT_Log(UT, INFO, "UT_TimerStop: EXIT\n");
}

/***************************************************************************
 * UT_TimerStop: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	Stops the specified timer using UT_RemoveNode from list.\n  
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTimer</td>
*		<td style="vertical-align: top;">pointer to timer sructure.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*  	 None.\n
*/
void UT_TimerStop(STimer *pstTimer)
{
	UT_MutexLock(&stTimerLinkedList.stSem);

	__UT_TimerStop(pstTimer);

	UT_MutexUnLock(&stTimerLinkedList.stSem);
}

/***************************************************************************
 * __UT_TimerStopSync: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	Stops the specified timer by removing from pre-locked list\n
*	and returns once timer is stopped.\n  
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTimer</td>
*		<td style="vertical-align: top;">pointer to timer sructure.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*  	 None.\n
*/
void __UT_TimerStopSync(STimer *pstTimer)
{
	UT_MutexLock(&pstTimer->stMutex);

	__UT_TimerStop(pstTimer);

	UT_MutexUnLock(&pstTimer->stMutex);
}

/***************************************************************************
 * UT_TimerStopSync: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	Stops the specified timer by removing from list\n
*	and returns once timer is stopped.\n  
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTimer</td>
*		<td style="vertical-align: top;">pointer to timer sructure.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*  	 None.\n
*/
void UT_TimerStopSync(STimer *pstTimer)
{
	UT_MutexLock(&pstTimer->stMutex);

	__UT_TimerStopSync(pstTimer);

	UT_MutexUnLock(&pstTimer->stMutex);
}

/***************************************************************************
 * UT_TimerCreate: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	1. Creates a allocate a timer structure.\n
*  	2. Initialize the newly created timer.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*
*	\n \b Returns:
*   	\li Pointer to the newly created timer's struct.\n
*   	\li NULL in case create timer returned an error.\n
*/
STimer *UT_TimerCreate()
{
	STimer *pstTimer = NULL;

	pstTimer = (STimer *) UT_Calloc(1, sizeof(STimer));
	if (NULL == pstTimer)
	{
		UT_ErrorLog(UT, "UT_CreateTime: No memory avilable");
		return NULL;
	}

	UT_MutexInit(&pstTimer->stMutex);

	return pstTimer;
}

/***************************************************************************
 * UT_TimerInit: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	Initialisze timer's structure to be passed as argument.\n
*  
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTimer</td>
*		<td style="vertical-align: top;">Pointer to timer structure</td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTimer</td>
*		<td style="vertical-align: top;">pointer to timer sructure.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*  	 None.\n
*/
void UT_TimerInit(STimer *pstTimer)
{
	UT_MemSet(pstTimer, 0, sizeof(STimer));
}

/***************************************************************************
 * __UT_TimerDelete: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	1. Stops timer using __UT_RemElement from pre-locked list.\n
*	2. Delete the specified timer using free allocated memory.\n
*
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTimer</td>
*		<td style="vertical-align: top;">pointer to timer sructure.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*  	 None.\n
*/
void __UT_TimerDelete(STimer *pstTimer)
{
	UT_Log(UT, INFO, "UT_TimerDelete: ENTRY\n");

	__UT_TimerStop(pstTimer);
	UT_MutexDestroy(&pstTimer->stMutex);
	UT_FreeMem(pstTimer);
	
	UT_Log(UT, INFO, "UT_TimerDelete: EXIT\n");
}

/***************************************************************************
 * UT_TimerDelete: The function does the following things -
 ***************************************************************************/
/*!
*	\n \b Description: \n
*  	1. Stops timer using __UT_RemElement from list.\n
*	2. Delete the specified timer using free allocated memory.\n
*  
*	\n 
*	<table style="text-align: left; width: 640px" border="0" cellpadding="2" cellspacing="0">
*	<tr>
*		<td style="background-color: rgb(213, 225, 232);"><b>Inputs-Outputs</b></td>
*		<td style="background-color: rgb(213, 225, 232);"><b></b></td>
*	</tr>
*	<tr>
*		<td style="vertical-align: top;">pstTimer</td>
*		<td style="vertical-align: top;">pointer to timer sructure.</td>
*	</tr>
*	</table>
*
*	\n \b Returns:
*  	 None.\n
*/
void UT_TimerDelete(STimer *pstTimer)
{
	UT_MutexLock(&stTimerLinkedList.stSem);

	__UT_TimerDelete(pstTimer);

	UT_MutexUnLock(&stTimerLinkedList.stSem);
}

