/*! \file ut.c 
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include "avl.h"
#include "ut.h"

/* Local Functions*/

/* Swing to the left
 * Warning: no balance maintainance
 */
static void __AVL_SwingLeft(SAVL ** ppstRoot)
{
	SAVL *pstA = *ppstRoot;
	SAVL *pstB = pstA->pstRight;
	*ppstRoot = pstB;
	pstA->pstRight = pstB->pstLeft;
	pstB->pstLeft = pstA;
}

/* Swing to the right
 * Warning: no balance maintainance
 */
static void __AVL_SwingRight(SAVL ** ppstRoot)
{
	SAVL *pstA = *ppstRoot;
	SAVL *pstB = pstA->pstLeft;
	*ppstRoot = pstB;
	pstA->pstLeft = pstB->pstRight;
	pstB->pstRight = pstA;
}

/* Balance maintainance after swings
 */
static void __AVL_Balance(SAVL * pstRoot)
{
	switch (pstRoot->cBalance)
	{
	case -1:
		pstRoot->pstLeft->cBalance = 0;
		pstRoot->pstRight->cBalance = 1;
		break;
	case 1:
		pstRoot->pstLeft->cBalance = -1;
		pstRoot->pstRight->cBalance = 0;
		break;
	case 0:
		pstRoot->pstLeft->cBalance = 0;
		pstRoot->pstRight->cBalance = 0;
	}

	pstRoot->cBalance = 0;
}

/* Public methods */

/* Insert element a into the AVL tree t
 * returns 1 if the depth of the tree has grown
 * Warning: do not insert elements already present
 */
INT __AVL_Insert(SAVL_Tree *pstTree, SAVL *pstA)
{
	/* initialize */
	pstA->pstLeft = 0;
	pstA->pstRight = 0;
	pstA->cBalance = 0;

	/* insert into an empty tree */
	if (!pstTree->pstRoot)
	{
		pstTree->pstRoot = pstA;

		return 1;
	}

	if (pstTree->pfnCompare(pstTree->pstRoot, pstA) > 0)
	{
		/* insert into the left subtree */
		if (pstTree->pstRoot->pstLeft)
		{
			SAVL_Tree stLeftSubTree;
			stLeftSubTree.pstRoot = pstTree->pstRoot->pstLeft;
			stLeftSubTree.pfnCompare = pstTree->pfnCompare;

			if (__AVL_Insert(&stLeftSubTree, pstA))
			{
				switch (pstTree->pstRoot->cBalance--)
				{
				case 1:
					return 0;
				case 0:
					return 1;
				}

				if (pstTree->pstRoot->pstLeft->cBalance < 0)
				{
					__AVL_SwingRight(&(pstTree->pstRoot));
					pstTree->pstRoot->cBalance = 0;
					pstTree->pstRoot->pstRight->cBalance = 0;
				}
				else
				{
					__AVL_SwingLeft(&(pstTree->pstRoot->pstLeft));
					__AVL_SwingRight(&(pstTree->pstRoot));
					__AVL_Balance(pstTree->pstRoot);
				}
			}
			else
				pstTree->pstRoot->pstLeft = stLeftSubTree.pstRoot;

			return 0;
		}
		else
		{
			pstTree->pstRoot->pstLeft = pstA;

			if (pstTree->pstRoot->cBalance--)
				return 0;

			return 1;
		}
	}
	else
	{
		/* insert into the right subtree */
		if (pstTree->pstRoot->pstRight)
		{
			SAVL_Tree stRightSubTree;
			stRightSubTree.pstRoot = pstTree->pstRoot->pstRight;
			stRightSubTree.pfnCompare = pstTree->pfnCompare;

			if (__AVL_Insert(&stRightSubTree, pstA))
			{
				switch (pstTree->pstRoot->cBalance++)
				{
				case -1:
					return 0;
				case 0:
					return 1;
				}

				if (pstTree->pstRoot->pstRight->cBalance > 0)
				{
					__AVL_SwingLeft(&(pstTree->pstRoot));
					pstTree->pstRoot->cBalance = 0;
					pstTree->pstRoot->pstLeft->cBalance = 0;
				}
				else
				{
					__AVL_SwingRight(&(pstTree->pstRoot->pstRight));
					__AVL_SwingLeft(&(pstTree->pstRoot));
					__AVL_Balance(pstTree->pstRoot);
				}
			}
			else
				pstTree->pstRoot->pstRight = stRightSubTree.pstRoot;

			return 0;
		}
		else
		{
			pstTree->pstRoot->pstRight = pstA;

			if (pstTree->pstRoot->cBalance++)
				return 0;

			return 1;
		}
	}
}

/* Remove an element a from the AVL tree t
 * returns -1 if the depth of the tree has shrunk
 * Warning: if the element is not present in the tree,
 *          returns 0 as if it had been removed succesfully.
 */
INT __AVL_Remove(SAVL_Tree *pstTree, SAVL *pstA)
{
	INT b;

	if (pstTree->pstRoot == pstA)
	{
		return __AVL_RemoveRoot(pstTree);
	}

	b = pstTree->pfnCompare(pstTree->pstRoot, pstA);
	if (b >= 0)
	{
		/* remove from the left subtree */
		int ch;
		SAVL_Tree stLeftSubTree;
		if ((stLeftSubTree.pstRoot = pstTree->pstRoot->pstLeft) != NULL)
		{
			stLeftSubTree.pfnCompare = pstTree->pfnCompare;
			ch = __AVL_Remove(&stLeftSubTree, pstA);
			pstTree->pstRoot->pstLeft = stLeftSubTree.pstRoot;
			if (ch)
			{
				switch (pstTree->pstRoot->cBalance++)
				{
				case -1:
					return -1;
				case 0:
					return 0;
				}

				switch (pstTree->pstRoot->pstRight->cBalance)
				{
				case 0:
					__AVL_SwingLeft(&(pstTree->pstRoot));
					pstTree->pstRoot->cBalance = -1;
					pstTree->pstRoot->pstLeft->cBalance = 1;
					return 0;
				case 1:
					__AVL_SwingLeft(&(pstTree->pstRoot));
					pstTree->pstRoot->cBalance = 0;
					pstTree->pstRoot->pstLeft->cBalance = 0;
					return -1;
				}

				__AVL_SwingRight(&(pstTree->pstRoot->pstRight));
				__AVL_SwingLeft(&(pstTree->pstRoot));
				__AVL_Balance(pstTree->pstRoot);

				return -1;
			}
		}
	}

	if (b <= 0)
	{
		/* remove from the right subtree */
		int ch;
		SAVL_Tree stRightSubTree;
		if ((stRightSubTree.pstRoot = pstTree->pstRoot->pstRight) != NULL)
		{
			stRightSubTree.pfnCompare = pstTree->pfnCompare;
			ch = __AVL_Remove(&stRightSubTree, pstA);
			pstTree->pstRoot->pstRight = stRightSubTree.pstRoot;
			if (ch)
			{
				switch (pstTree->pstRoot->cBalance--)
				{
				case 1:
					return -1;
				case 0:
					return 0;
				}

				switch (pstTree->pstRoot->pstLeft->cBalance)
				{
				case 0:
					__AVL_SwingRight(&(pstTree->pstRoot));
					pstTree->pstRoot->cBalance = 1;
					pstTree->pstRoot->pstRight->cBalance = -1;

					return 0;

				case -1:
					__AVL_SwingRight(&(pstTree->pstRoot));
					pstTree->pstRoot->cBalance = 0;
					pstTree->pstRoot->pstRight->cBalance = 0;

					return -1;
				}

				__AVL_SwingLeft(&(pstTree->pstRoot->pstLeft));
				__AVL_SwingRight(&(pstTree->pstRoot));
				__AVL_Balance(pstTree->pstRoot);

				return -1;
			}
		}
	}

	return 0;
}

/* Remove the root of the AVL tree t
 * Warning: dumps core if pstTree is empty
 */
INT __AVL_RemoveRoot(SAVL_Tree *pstTree)
{
	INT ch;
	SAVL *pstA;

	if (!pstTree->pstRoot->pstLeft)
	{
		if (!pstTree->pstRoot->pstRight)
		{
			pstTree->pstRoot = 0;
			return -1;
		}
		pstTree->pstRoot = pstTree->pstRoot->pstRight;

		return -1;
	}

	if (!pstTree->pstRoot->pstRight)
	{
		pstTree->pstRoot = pstTree->pstRoot->pstLeft;

		return -1;
	}

	if (pstTree->pstRoot->cBalance < 0)
	{
		/* remove from the left subtree */
		pstA = pstTree->pstRoot->pstLeft;
		while (pstA->pstRight)
			pstA = pstA->pstRight;
	}
	else
	{
		/* remove from the right subtree */
		pstA = pstTree->pstRoot->pstRight;
		while (pstA->pstLeft)
			pstA = pstA->pstLeft;
	}

	ch = __AVL_Remove(pstTree, pstA);

	pstA->pstLeft = pstTree->pstRoot->pstLeft;
	pstA->pstRight = pstTree->pstRoot->pstRight;
	pstA->cBalance = pstTree->pstRoot->cBalance;
	pstTree->pstRoot = pstA;

	if (pstA->cBalance == 0)
		return ch;

	return 0;
}

SAVL *__AVL_Search(SAVL_Tree *pstTree, SAVL *pstA)
{
	INT iComp;
	SAVL_Tree stTemp;

	if (pstTree == NULL)
	{
		return NULL;
	}

	if (!(pstTree)->pstRoot)
	{
		return NULL;
	}

	UT_MemCopy(&stTemp, pstTree, sizeof(SAVL_Tree));

	while (stTemp.pstRoot != NULL)
	{
		if ((iComp = stTemp.pfnCompare(stTemp.pstRoot, pstA)) == 0)
		{
			return stTemp.pstRoot;
		}

		if (iComp > 0)
		{
			stTemp.pstRoot = stTemp.pstRoot->pstLeft;
			stTemp.pfnCompare = stTemp.pfnCompare;
		}

		if (iComp < 0)
		{
			stTemp.pstRoot = stTemp.pstRoot->pstRight;
			stTemp.pfnCompare = stTemp.pfnCompare;
		}
	}

	return NULL;
}
