/*! \file tmap_map.c 
 *
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#ifdef _VXWORKS_
#include <vxWorks.h>
#endif

#include "ut.h"
#include "gtl.h"
#include "tmap_devinfo.h"

#define PRINT_CSME_MSG(MsgType, pstMsg)\
do { \
	UT_Log(GTL, INFO, \
	"index:    0x%02x, length:  0x%02x\n" \
	"cmdclass: 0x%02x, cmdtype: 0x%02x\n" \
	"fcode:    0x%04x\n", \
	((MsgType *)(pstMsg))->index, ((MsgType *)(pstMsg))->length, \
	((MsgType *)(pstMsg))->cmd_class, ((MsgType *)(pstMsg))->cmd_type, \
	((MsgType *)(pstMsg))->func_code); \
} while(0)

GTL_READ_CB gtl_read_cb;
static int bGtlInit = 0;
SSortArray *pstDeviceArray = NULL;

static U32 uiGtlMaxInterfaces = 0;

SGtlConfigData stGtlConfigData;

void print_devinfo(SGtlDeviceInfo * dev);

#if !defined(VAPI_NODBG)
void GTL_PrintCsmeMsg(IN gtl_msg_t *pstMsg)
{
	int sOffset = 0;
	U8 ucLength;
	
	if (pstMsg->type & BOOT)
	{	if (pstMsg->fifo_size < sizeof(SBootHdr))
			UT_ErrorLog(GTL, "The message fifo size is less then 0x%x bytes \n", sizeof(SBootHdr));	
		else
			PRINT_CSME_MSG(SBootHdr, (U8 *)pstMsg->fifo);
	}
	else if (pstMsg->type & CSME)
	{
		/* while loop to dump multi command (if any) */
		while ((sOffset + sizeof(SApiHdr)) <= pstMsg->fifo_size)
		{
			ucLength = ((U8 *) pstMsg->fifo)[sOffset];
			if (ucLength < sizeof(SApiHdr))
			{
				UT_ErrorLog(GTL, "The packet length is less then 0x%x bytes \n", sizeof(SApiHdr));
				break;
			}
			
			PRINT_CSME_MSG(SApiHdr, (U8 *)pstMsg->fifo + sOffset);
			sOffset += (ucLength + 3) & ~0x3;
		}		
	}
	else
		UT_ErrorLog(GTL, "The message type is neither BOOT nor CSME 0x%02x\n", pstMsg->type);
}
#endif

void GTL_PrintMessage(IN gtl_msg_t * pstMsg, IN U8 layer, IN Boolean bIsTx)
{
	int iCnt;
	char dest[BUFFER_SIZE];
	char *dst = dest;

	if (bIsTx)
	{
		UT_Sprintf(dst, "TX :");
	}
	else
	{
		UT_Sprintf(dst, "RX :");
	}

	dst += UT_StrLen(dst);

	if ((pstMsg->type & PCI) == PCI)
	{
		if (pstMsg->type & BOOT)
			UT_Sprintf(dst, "%04X,%04X,%04X,%04X\n", pstMsg->mailbox.reg3, pstMsg->mailbox.reg2,
							pstMsg->mailbox.reg1, pstMsg->mailbox.reg0);
		else
			UT_Sprintf(dst, "%04X,%04X,%04X,%04X\n", MAILBOX_BMR_CMD, pstMsg->fifo_size,
							pstMsg->channel, 0);
	}

	else if ((pstMsg->type & CSME) == CSME)
	{
		UT_Sprintf(dst, "%04X,%04X,%04X\n", *(((U16 *) (pstMsg->fifo)) + 0), pstMsg->fifo_size, pstMsg->channel);
	}

	dst += UT_StrLen(dst);
	if (pstMsg->fifo != NULL)
	{
		UT_Sprintf(dst, "Fifo(%d):", pstMsg->fifo_size);
		dst += UT_StrLen(dst);
	
		if(pstMsg->fifo == NULL)
			goto out;
	
		for (iCnt = 0; iCnt < pstMsg->fifo_size && iCnt < 100; iCnt++)
		{
			UT_Sprintf(dst, " %02X", *(((U8 *) pstMsg->fifo) + iCnt));
			dst += 3;
	
		}
	}
out:
	UT_Sprintf(dst, "\n");
	UT_Log(layer, PACKET, "%s", dest);
}

/************************** GTL_FreeInterfaces *********************************/
/*!
 *    This function should be called by VAPI Core module at shutdown \n
 *    It performs the following. fns\n 
 *    1. frees the individual interface adapters' .\n
 *
 */
/***************************************************************************/

void GTL_FreeInterfaces(U32 uNumInterfaces)
{
	U32 uCount;

	UT_Log(GTL, INFO, "GTL_FreeInterfaces: ENTRY\n");

	for (uCount = 0; uCount < uNumInterfaces; uCount++)
	{
		UT_FreeMem(astItfList[uCount].pstAdapterFns);
	}

	UT_Log(GTL, INFO, "GTL_FreeInterfaces: EXIT\n");
}

/************************** GTL_ShutdownInterfaces *********************************/
/*!
 *    This function should be called by VAPI Core module at shutdown \n
 *    It performs the following. fns\n 
 *    1. shut down interface adapters' .\n
 *
 */
/***************************************************************************/

void GTL_ShutdownInterfaces(U32 uNumInterfaces)
{
	U32 uCount;

	UT_Log(GTL, INFO, "GTL_ShutdownInterfaces: ENTRY\n");

	for (uCount = 0; uCount < uNumInterfaces; uCount++)
	{
		astItfList[uCount].pstAdapterFns->pfnAdapterShutdown();
	}

	UT_Log(GTL, INFO, "GTL_ShutdownInterfaces: EXIT\n");
}

/************************** GTL_FreeAllInterfaces *********************************/
/*!
 *    This function should be called by VAPI Core module at shutdown \n
 *    It performs the following. fns\n 
 *    1. frees all adapters' .\n
 *
 */
/***************************************************************************/

int GTL_FreeAllInterfaces(void)
{
	int Status = SUCCESS;

	UT_Log(GTL, INFO, "GTL_FreeAllInterfaces: ENTRY\n");

	if (bGtlInit == 0)
	{
		UT_ErrorLog(GTL, "GTL_FreeAllInterfaces: GTL not initialized\n");
		Status = GTL_ERR_NOT_INITIALIZED;
		goto out;
	}	

	GTL_FreeInterfaces(uiGtlMaxInterfaces);

out:
	UT_Log(GTL, INFO, "GTL_FreeAllInterfaces: EXIT\n");
	return Status;	
}

/************************** GTL_ShutdownAllInterfaces *********************************/
/*!
 *    This function should be called by VAPI Core module at shutdown \n
 *    It performs the following. fns\n
 *    1. shut down all adapters.\n
 *
 */
/***************************************************************************/

int GTL_ShutdownAllInterfaces(void)
{
	int Status = SUCCESS;

	UT_Log(GTL, INFO, "GTL_ShutdownAllInterfaces: ENTRY\n");

	if (bGtlInit == 0)
	{
		UT_ErrorLog(GTL, "GTL_ShutdownAllInterfaces: GTL not initialized\n");
		Status = GTL_ERR_NOT_INITIALIZED;
		goto out;
	}

	GTL_ShutdownInterfaces(uiGtlMaxInterfaces);

out:
	UT_Log(GTL, INFO, "GTL_ShutdownAllInterfaces: EXIT\n");
	return Status;
}

/************************** GTL_FreeDevices *********************************/
/*!
 *    This function should be called by VAPI Core module at shutdown \n
 *    It performs the following. fns\n 
 *    1. frees the individual device adapters' .\n
 *
 */
/***************************************************************************/

void GTL_FreeDevices(U32 uNumDevices)
{
	SArrayNode *pstArrayNode;
	SGtlDeviceInfo *pstDevice;
	U32 uCount;

	UT_Log(GTL, INFO, "GTL_FreeDevices: ENTRY\n");

	for (uCount = 0; uCount < uNumDevices; uCount++)
	{
		pstDevice = &pastDevList[uCount];

		pstArrayNode = UT_SearchElement(pstDeviceArray, pstDevice->devId);

		if (pstArrayNode != NULL)
			UT_RemElement(pstDeviceArray, pstArrayNode);

		if (pstDevice->eDevItfType == ePCI_ITF)
		{
			UT_FreeMem(((SGtlPciItf *) pstDevice->pvDevAddrInfo)->pciDevName);
			UT_FreeMem(pstDevice->pvDevAddrInfo);
		}
		else if (pstDevice->eDevItfType == eCSM_ITF)
		{
			UT_FreeMem(((SGtlCsmeItf *) pstDevice->pvDevAddrInfo)->pcEthName);
			UT_FreeMem(pstDevice->pvDevAddrInfo);
		}
		else if (pstDevice->eDevItfType == ePOS_ITF)
		{
			UT_FreeMem(((SGtlPosItf *) pstDevice->pvDevAddrInfo)->pcEthName);
			UT_FreeMem(pstDevice->pvDevAddrInfo);
		}
	}

	UT_Log(GTL, INFO, "GTL_FreeDevices: EXIT\n");
}

/************************** GTL_Init *************************************/
/*!
 *    This function should be called by VAPI Core module at start up \n
 *    before issuing any other command. It performs the following. fns\n 
 *    1. Registers and initializes the individual interface adapters.\n
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_ERR_ALREADY_INITIALIZED\n
 *                          or GTL_ERR_NULL_POINTER_PASSED\n 
 *                          or GTL_ERR_START.\n
 *    \param    pstDeviceHeaders -> pointer to list of device configuration
 *              descriptors. \n
 *    \param    pfnReadCb -> pointer to the receive callback function. \n
 *              This pointer is passed to the individual interface adapters\n
 *              On reception of a message from MSP, the adapters invokes this\n
 *              callback function to deliver the message to the\n
 *              GTL client.\n
 */
/***************************************************************************/
int GTL_Init(SDeviceHeader *pstDeviceHeaders, GTL_READ_CB pfnReadCb)
{
	U32 uCount, uCount1;
	int retCode;
	SDeviceHeader *pstTempDevHdr;
	SGtlDeviceInfo *pstDevice;
	SGtlPciItf *pstPciItf;
	SGtlCsmeItf *pstCsmeItf;
	SGtlPosItf *pstPosItf;
	UT_Log(GTL, INFO, "GTL_Init: ENTRY\n");

	for (uiGtlMaxInterfaces = 0; astItfList[uiGtlMaxInterfaces].pfnItfInitFn; uiGtlMaxInterfaces++)
	{
	}

	UT_Log(GTL, INFO, "GTL_Init: Number of Interfaces = %d\n", uiGtlMaxInterfaces);

	if (bGtlInit == 1)
	{
		UT_ErrorLog(GTL, "GTL already initialized\n");
		retCode = GTL_ERR_ALREADY_INITIALIZED;
		goto err0;
	}

	if (pfnReadCb == NULL)
	{
		UT_ErrorLog(GTL, " GTL Call back is NULL\n");
		retCode = GTL_ERR_NULL_POINTER_PASSED;
		goto err0;
	}

	/* Get the DeviceList and calculate the max number of devices */
	if (pstDeviceHeaders == NULL)
	{
		UT_ErrorLog(GTL, "Error while getting Device Configuration\n");
		retCode = GTL_ERR_CONFIG;
		goto err0;
	}

	pstTempDevHdr = pstDeviceHeaders;
	uCount = 0;
	do
	{
		pstTempDevHdr = pstTempDevHdr->pvNext;
		uCount++;
	} while (pstTempDevHdr != NULL);

	stGtlConfigData.uiNumDevices = uCount;

	pastDevList = UT_AllocMem(stGtlConfigData.uiNumDevices * sizeof(SGtlDeviceInfo));
	if (pastDevList == NULL)
	{
		UT_ErrorLog(GTL, " GTL_Init : Memory Alloc error while creating device list\n");
		retCode = GTL_ERR_NOMEM;
		goto err0;
	}

	pstDeviceArray = UT_CreateSortedArray(stGtlConfigData.uiNumDevices);
	if (pstDeviceArray == NULL)
	{
		UT_ErrorLog(GTL, " GTL_Init : Memory Alloc error while creating device list\n");
		retCode = GTL_ERR_NOMEM;
		goto err1;
	}

	for (uCount = 0, pstTempDevHdr = pstDeviceHeaders; uCount < stGtlConfigData.uiNumDevices; uCount++, pstTempDevHdr = pstTempDevHdr->pvNext)
	{
		pstDevice = &pastDevList[uCount];

		pstDevice->devId = pstTempDevHdr->uiDevId;
		pstDevice->eDevItfType = pstTempDevHdr->usControlInterface;
		pstDevice->eDevMode = pstTempDevHdr->eDevMode;
		pstDevice->eDevState = eIDLE;
		pstDevice->uItfIdx = 0;
		pstDevice->ucDevType = pstTempDevHdr->ucDevType;

		if (pstTempDevHdr->usControlInterface == ePCI_ITF)
		{
			pstPciItf = UT_AllocMem(sizeof(SGtlPciItf));
			if (pstPciItf == NULL)
			{
				UT_ErrorLog(GTL, " GTL_Init : Memory Alloc error\n");
				retCode = GTL_ERR_NOMEM;
				goto err2;
			}

			pstDevice->pvDevAddrInfo = pstPciItf;

			pstPciItf->pciDevName = UT_AllocMem(MAX_PCI_NAME_SIZE);

			if (pstPciItf->pciDevName == NULL)
			{
				UT_ErrorLog(GTL, " GTL_Init : Memory Alloc error\n");
				UT_FreeMem(pstPciItf);
				retCode = GTL_ERR_NOMEM;
				goto err2;
			}

			UT_Log(GTL, INFO, "Adding a PCI device %d\n", pstDevice->devId);

			pstPciItf->pciDevName[MAX_PCI_NAME_SIZE - 1] = '\0';

			UT_StrnCpy(pstPciItf->pciDevName, ((SPCIUsrData *) pstTempDevHdr)->pucPCIDevName,
					MAX_PCI_NAME_SIZE - 1);

			if (((SPCIUsrData *) pstTempDevHdr)->bUseDefaultMaxChnls)
			{
				pstPciItf->usMaxChannels = GTL_GetDefaultMaxChnls(((SPCIUsrData *) pstTempDevHdr)->ucDevType);
			}
			else
			{
				pstPciItf->usMaxChannels = ((SPCIUsrData *) pstTempDevHdr)->usMaxChannels;
			}

			pstDevice->usMaxChnls = pstPciItf->usMaxChannels;
		}
		else if (pstTempDevHdr->usControlInterface == eCSM_ITF)
		{
			UT_Log(GTL, INFO, "Adding a CSME device %d\n", pstDevice->devId);

			pstCsmeItf = UT_AllocMem(sizeof(SGtlCsmeItf));
			if (NULL == pstCsmeItf)
			{
				UT_ErrorLog(GTL, " GTL_Init : Memory Alloc error\n");
				retCode = GTL_ERR_NOMEM;
				goto err2;
			}

			pstDevice->pvDevAddrInfo = pstCsmeItf;

			pstCsmeItf->pcEthName = UT_AllocMem(MAX_ETH_NAME_SIZE);
			if (pstCsmeItf->pcEthName == NULL)
			{
				UT_ErrorLog(GTL, " GTL_Init : Memory Alloc error\n");
				UT_FreeMem(pstCsmeItf);
				retCode = GTL_ERR_NOMEM;
				goto err2;
			}

			pstCsmeItf->ackRequired = ((SCSMEUsrData *) pstTempDevHdr)->ucIsAckReqd;

			UT_MemCopy(pstCsmeItf->devMac, ((SCSMEUsrData *) pstTempDevHdr)->aucDevMac, 6);

			UT_MemCopy(pstCsmeItf->hostMac, ((SCSMEUsrData *) pstTempDevHdr)->aucHostMac, 6);

			pstCsmeItf->pcEthName[MAX_ETH_NAME_SIZE - 1] = '\0';

			UT_StrnCpy(pstCsmeItf->pcEthName, ((SCSMEUsrData *) pstTempDevHdr)->pucEthDevName,
					MAX_ETH_NAME_SIZE - 1);

			if (((SCSMEUsrData *) pstTempDevHdr)->bUseDefaultMaxChnls)
			{
				pstCsmeItf->usMaxChannels = GTL_GetDefaultMaxChnls(((SCSMEUsrData *) pstTempDevHdr)->ucDevType);
			}
			else
			{
				pstCsmeItf->usMaxChannels = ((SCSMEUsrData *) pstTempDevHdr)->usMaxChannels;
			}

			pstDevice->usMaxChnls = pstCsmeItf->usMaxChannels;
			pstCsmeItf->ucEthBootMode = ((SCSMEUsrData *) pstTempDevHdr)->ucEthBootMode;
			pstDevice->ucEthBootMode = ((SCSMEUsrData *) pstTempDevHdr)->ucEthBootMode;

			/* for the M823XX alternate boot mode the user may have set a unicast address for the device
			to receive the ready frame so use devHardMac as it is.
			For other devices or M823XX other boot mode always open a broadcast target*/
			if (((pstDevice->ucDevType == DEV_TYPE_M823XX) || (pstDevice->ucDevType == DEV_TYPE_M823XX_2)) &&
				(pstDevice->ucEthBootMode != BOOT_MODE_LEGACY))
			{
				UT_MemCopy(pstCsmeItf->devHardMac, ((SCSMEUsrData *) pstTempDevHdr)->aucDevHardMac, 6);
				UT_MemCopy(pstDevice->devHardMac, pstCsmeItf->devHardMac, 6);
			}

		}
		else if (pstTempDevHdr->usControlInterface == ePOS_ITF)
		{
			UT_Log(GTL, INFO, "Adding a POS device %d\n", pstDevice->devId);

			pstPosItf = UT_AllocMem(sizeof(SGtlPosItf));
			if (pstPosItf == NULL)
			{
				UT_ErrorLog(GTL, " GTL_Init : Memory Alloc error\n");
				retCode = GTL_ERR_NOMEM;
				goto err2;
			}

			pstPosItf->pvAdptData = UT_AllocMem(sizeof(PosPathInfo));
			if (pstPosItf->pvAdptData == NULL)
			{
				UT_ErrorLog(GTL, " GTL_Init : Memory Alloc error\n");
				UT_FreeMem(pstPosItf);
				retCode = GTL_ERR_NOMEM;
				goto err2;
			}

			pstDevice->pvDevAddrInfo = pstPosItf;

			pstPosItf->pvAdptData->ucPhyId = ((SPOSUsrData *) pstTempDevHdr)->ucPhyId;

			pstPosItf->pvAdptData->ePathType = ((SPOSUsrData *) pstTempDevHdr)->ePathType;

			pstPosItf->pvAdptData->Tsp3.DevId = ((SPOSUsrData *) pstTempDevHdr)->PathInfo.Tsp3.DevId;

			pstPosItf->ackRequired = ((SPOSUsrData *) pstTempDevHdr)->ucIsAckReqd;

			UT_MemCopy(pstPosItf->devMac, ((SPOSUsrData *) pstTempDevHdr)->aucDevMac, 6);

			UT_MemCopy(pstPosItf->hostMac, ((SPOSUsrData *) pstTempDevHdr)->aucHostMac, 6);

			pstPosItf->pcEthName = UT_AllocMem(MAX_ETH_NAME_SIZE);

			if (pstPosItf->pcEthName == NULL)
			{
				UT_FreeMem(pstPosItf->pvAdptData);
				UT_FreeMem(pstPosItf);
				retCode = GTL_ERR_NOMEM;
				goto err2;
			}

			pstPosItf->pcEthName[MAX_ETH_NAME_SIZE - 1] = '\0';

			UT_StrnCpy(pstPosItf->pcEthName, ((SPOSUsrData *) pstTempDevHdr)->pucEthDevName,
					MAX_ETH_NAME_SIZE - 1);

			if (((SPOSUsrData *) pstTempDevHdr)->bUseDefaultMaxChnls)
			{
				pstPosItf->usMaxChannels = GTL_GetDefaultMaxChnls(((SPOSUsrData *) pstTempDevHdr)->ucDevType);
			}
			else
			{
				pstPosItf->usMaxChannels = ((SPOSUsrData *) pstTempDevHdr)->usMaxChannels;
			}

			pstDevice->usMaxChnls = pstPosItf->usMaxChannels;
		}
		else
		{
			retCode = GTL_ERR_INVAL_CNTRL_ITF;
			goto err2;
		}

		pstDevice->stArrayNode.uiKey = pstDevice->devId;

		UT_Log(GTL, INFO,
			       "GTL_Init: Adding Device" "(DevId %u,Itftype %d) to devlist\n",
			       pstDevice->devId,
			       pstDevice->eDevItfType);

		for (uCount1 = 0; uCount1 < uiGtlMaxInterfaces; uCount1++)
		{
			if (astItfList[uCount1].eItfType == pstDevice->eDevItfType)
			{
				pstDevice->uItfIdx = uCount1;
				break;
			}
			else
			{
				pstDevice->uItfIdx = INVALID_ITF_IDX;
			}
		}

		if (pstDevice->uItfIdx < uiGtlMaxInterfaces)
		{
			if (UT_AddElement(pstDeviceArray, &pstDevice->stArrayNode) < 0)
			{
				UT_ErrorLog(GTL, " GTL_Init : ERROR while adding device\n");

				retCode = GTL_ERR_UNKNOWN;
			}
		}
	}

	for (uCount = 0; uCount < uiGtlMaxInterfaces; uCount++)
	{
		astItfList[uCount].pstAdapterFns = UT_Calloc(1, sizeof(STAdapterFns));
		/*Store the Adapter function pointers in the interface list */
		if (NULL == astItfList[uCount].pstAdapterFns)
		{
			UT_ErrorLog(GTL, " GTL_Init : Memory Alloc error while creating device list\n");
			retCode = GTL_ERR_NOMEM;
			goto err3;
		}

		if (astItfList[uCount].pfnItfInitFn != NULL
		    && (retCode = astItfList[uCount].pfnItfInitFn(pfnReadCb,
								  astItfList[uCount].pstAdapterFns, pastDevList,
								  stGtlConfigData.uiNumDevices)) < 0)
		{
			UT_FreeMem(astItfList[uCount].pstAdapterFns);
			UT_ErrorLog(GTL, "Failed to initialize GTL Adapter\n");
			retCode = GTL_ERR_UNKNOWN;
			goto err3;
		}
	}

	bGtlInit = 1;

	UT_Log(GTL, INFO, "GTL_Init: EXIT\n");

	return SUCCESS;

err3:
	/* free the already allocated interfaces */
	GTL_ShutdownInterfaces(uCount);
	GTL_FreeInterfaces(uCount);
	/* free all the allocated devices */
	uCount = stGtlConfigData.uiNumDevices;
err2:
	/* free the already allocated devices */
	GTL_FreeDevices(uCount);
	UT_DeleteSortedArray(pstDeviceArray);

err1:
	UT_FreeMem(pastDevList);
err0:
	return retCode;
}

/************************** GTL_Open *************************************/

/*!
 *    This function should be called by VAPI Core module for opening \n
 *    a device. It performs the following. fns\n 
 *    1. Finds out the device entry corresponding to the device-id and\n
 *       calls the adapter open function corresponding to the device \n
 *       interface.
 *    2. Changes device state to eACTIVE 
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_ERR_INVALID_DEVID\n
 *                          or GTL_ERR_ALREADY_OPEN\n 
 *                          or GTL_ERR_MAX_DEV_OPEN.\n
 *    \param    deviceid -> Device identifier provided by GTL user \n
 */
/***************************************************************************/
int GTL_Open(S32 deviceid)
{
	SArrayNode *pstArrayNode;
	SGtlDeviceInfo *pstDevice;
	void *pvAdapterSpecInfo = NULL;
	int retCode = SUCCESS;

	UT_Log(GTL, INFO, "GTL_Open: ENTRY\n");

	if (bGtlInit == 0)
	{
		UT_ErrorLog(GTL, "GTL_Open : GTL not initialized\n");
		retCode = GTL_ERR_NOT_INITIALIZED;
		goto out;
	}

	pstArrayNode = UT_SearchElement(pstDeviceArray, deviceid);
	if (pstArrayNode == NULL)
	{

		UT_ErrorLog(GTL, "GTL_Open: Device %d does not exist\n", deviceid);
		retCode = GTL_ERR_INVALID_DEVID;
		goto out;
	}

	pstDevice = UT_ContainerOf(pstArrayNode, SGtlDeviceInfo, stArrayNode);

	if (eACTIVE == pstDevice->eDevState)
	{
		UT_ErrorLog(GTL, "GTL_Open: Device %d is already open\n", deviceid);
		retCode = GTL_ERR_ALREADY_OPEN;
		goto out;
	}

	if (SUCCESS != astItfList[pstDevice->uItfIdx].pstAdapterFns->pfnAdapterOpen(deviceid,
					pstDevice->pvDevAddrInfo, &pvAdapterSpecInfo,
					pstDevice->eDevMode))

	{
		UT_ErrorLog(GTL, "GTL_Open: Error opening Device %d\n", deviceid);
		retCode = GTL_ERR_UNKNOWN;
		goto out;
	}

	UT_Log(GTL, INFO, "GTL_Open: Opened Device %d successfully\n", deviceid);
	pstDevice->eDevState = eACTIVE;
	pstDevice->pvAdapterSpecInfo = pvAdapterSpecInfo;

out:
	UT_Log(GTL, INFO, "GTL_Open: EXIT\n");
	return retCode;
}

/************************** GTL_Close *************************************/

/*!
 *    This function should be called by VAPI Core module for closing \n
 *    a device. It performs the following. fns\n 
 *    1. Finds out the device entry corresponding to the device-id and\n
 *       calls the adapter close function corresponding to the device \n
 *       interface.
 *    2. Changes device state to eIDLE. 
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_ERR_INVALID_DEVID\n
 *                          or GTL_ERR_ALREADY_CLOSED
 *    \param    deviceid -> Device identifier provided by GTL user \n
 */
/***************************************************************************/
int GTL_Close(S32 deviceid)
{
	SArrayNode *pstArrayNode;
	SGtlDeviceInfo *pstDevice;
	int retCode = SUCCESS;

	UT_Log(GTL, INFO, "GTL_Close: ENTRY\n");
	if (bGtlInit == 0)
	{
		UT_ErrorLog(GTL, "GTL_Close : GTL not initialized\n");
		retCode = GTL_ERR_NOT_INITIALIZED;
		goto out;
	}

	pstArrayNode = UT_SearchElement(pstDeviceArray, deviceid);
	if (pstArrayNode == NULL)
	{
		UT_ErrorLog(GTL, "GTL_Close: Device %d does not exist\n", deviceid);
		retCode = GTL_ERR_INVALID_DEVID;
		goto out;
	}

	pstDevice = UT_ContainerOf(pstArrayNode, SGtlDeviceInfo, stArrayNode);

	if ((pstDevice->eDevState ==eIDLE))
	{
		UT_ErrorLog(GTL, "GTL_Close: Device %d is already closed\n", deviceid);
		retCode = GTL_ERR_ALREADY_CLOSED;
		goto out;
	}

	if (SUCCESS != astItfList[pstDevice->uItfIdx].pstAdapterFns->pfnAdapterClose(pstDevice->pvAdapterSpecInfo))
	{
		UT_ErrorLog(GTL, "GTL_Close: Error closing Device %d\n", deviceid);
		retCode = GTL_ERR_UNKNOWN;
		goto out;
	}

	UT_Log(GTL, INFO, "GTL_Close: Closed Device %d successfully\n", deviceid);
	pstDevice->eDevState = eIDLE;
	pstDevice->pvAdapterSpecInfo = NULL;

out:
	UT_Log(GTL, INFO, "GTL_Close: EXIT\n");
	return retCode;
}

/************************** GTL_QueryInterface *****************************/

/*!
 *    This function is called by VAPI Core module for querying the interace\n
 *    type of a device. It performs the following. fns\n 
 *    1. Finds out the device entry corresponding to the device-id and\n
 *       returns the interface details of the device.
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_ERR_INVALID_DEVID\n
 *    \param    deviceid -> Device identifier provided by GTL user \n
 *    \param    pstItfPtr-> Pointer for returning interface information\n 
 */
/***************************************************************************/
int GTL_QueryInterface(S32 deviceid, gtl_phy_itf_t * pstItfPtr)
{
	SArrayNode *pstArrayNode;
	SGtlDeviceInfo *pstDevice;
	int retCode = SUCCESS;

	UT_Log(GTL, INFO, "GTL_QueryInterface: ENTRY\n");
	if (bGtlInit == 0)
	{
		UT_ErrorLog(GTL, "GTL_QueryInterface : GTL not initialized\n");
		retCode = GTL_ERR_NOT_INITIALIZED;
		goto out;
	}

	if (NULL == pstItfPtr)
	{
		UT_ErrorLog(GTL, " GTL_QueryInterface : The interface pointer is NULL \n");
		retCode = GTL_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	pstArrayNode = UT_SearchElement(pstDeviceArray, deviceid);
	if (pstArrayNode == NULL)
	{
		UT_ErrorLog(GTL, "GTL_QueryInterface: Device %d does not exist\n", deviceid);
		retCode = GTL_ERR_INVALID_DEVID;
		goto out;
	}

	pstDevice = UT_ContainerOf(pstArrayNode, SGtlDeviceInfo, stArrayNode);

	pstItfPtr->pvItfInfo = (GtlInterface *) pstDevice->pvDevAddrInfo;
	pstItfPtr->type = pstDevice->eDevItfType;
	pstItfPtr->eDevMode = pstDevice->eDevMode;
	pstItfPtr->ucDevType = pstDevice->ucDevType;
	pstItfPtr->usMaxChnls = pstDevice->usMaxChnls;
	pstItfPtr->ucEthBootMode = pstDevice->ucEthBootMode;

out:
	UT_Log(GTL, INFO, "GTL_QueryInterface : EXIT DevId %d\n", deviceid);

	return retCode;
}

/************************** GTL_Reset *****************************/

/*!
 *    This function is called by VAPI Core module for reseting the \n
 *    device. It performs the following. fns\n 
 *    1. Finds out the device entry corresponding to the device-id and\n
 *       calls the interface reset function.
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_ERR_INVALID_DEVID\n
 *    \param    deviceid -> Device identifier provided by GTL user \n
 */
/***************************************************************************/
int GTL_Reset(S32 deviceid)
{
	SArrayNode *pstArrayNode;
	SGtlDeviceInfo *pstDevice;
	int retCode = SUCCESS;

	UT_Log(GTL, INFO, "GTL_Reset: ENTRY\n");
	if (bGtlInit == 0)
	{
		UT_ErrorLog(GTL, "GTL_Reset : GTL not initialized\n");
		retCode = GTL_ERR_NOT_INITIALIZED;
		goto out;
	}

	pstArrayNode = UT_SearchElement(pstDeviceArray, deviceid);
	if (pstArrayNode == NULL)
	{
		UT_ErrorLog(GTL, "GTL_Reset: Device %d does not exist\n", deviceid);
		retCode = GTL_ERR_INVALID_DEVID;
		goto out;
	}

	pstDevice = UT_ContainerOf(pstArrayNode, SGtlDeviceInfo, stArrayNode);

	/* no reset required in Mqter mode (no MAC addresses to change/update)*/
	if (pstDevice->eDevMode == eMASTER)
		goto out;

	if (SUCCESS != astItfList[pstDevice->uItfIdx].pstAdapterFns->
			pfnAdapterReset(pstDevice->pvDevAddrInfo, pstDevice->pvAdapterSpecInfo))
	{
		UT_ErrorLog(GTL, "GTL_Reset: Error resetting Device %d\n", deviceid);
		retCode = GTL_ERR_UNKNOWN;
		goto out;
	}

	UT_Log(GTL, INFO, "GTL_Reset: Reset on  Device %d successful\n", deviceid);
	/*((SGtlDeviceInfo *)(pstDevice->pvPointer))->eDevState = eIDLE;  */
out:
	UT_Log(GTL, INFO, "GTL_Reset : EXIT\n");
	return retCode;
}

/************************** GTL_Write *****************************/

/*!
 *    This function is called by VAPI Core module for writing to the \n
 *    device. It performs the following. fns\n 
 *    1. Finds out the device entry corresponding to the device-id and\n
 *       calls the interface write function.
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_ERR_INVALID_DEVID\n
 *    \param    deviceid -> Device identifier provided by GTL user \n
 *    \param    pstMsg   -> Message to be written to the device \n
 */
/***************************************************************************/

int GTL_Write(S32 deviceid, gtl_msg_t * pstMsg)
{
	SArrayNode *pstArrayNode;
	SGtlDeviceInfo *pstDevice;
	int retCode = SUCCESS;
	
	UT_Log(GTL, INFO, "GTL_Write : ENTRY\n");

	if (bGtlInit == 0)
	{
		UT_ErrorLog(GTL, "GTL_Write : GTL not initialized\n");
		retCode = GTL_ERR_NOT_INITIALIZED;
		goto out;
	}

	if (NULL == pstMsg)
	{
		UT_ErrorLog(GTL, " GTL_Write : The message pointer is NULL \n");
		retCode = GTL_ERR_NULL_POINTER_PASSED;
		goto out;
	}

	pstArrayNode = UT_SearchElement(pstDeviceArray, deviceid);
	if (pstArrayNode == NULL)
	{
		UT_ErrorLog(GTL, "GTL_Write: Device %d does not exist\n", deviceid);
		retCode = GTL_ERR_INVALID_DEVID;
		goto out;
	}

	pstDevice = UT_ContainerOf(pstArrayNode, SGtlDeviceInfo, stArrayNode);

	if ((eACTIVE != pstDevice->eDevState))
	{
		UT_ErrorLog(GTL, "GTL_Write: Device %d is not open\n", deviceid);
		retCode = GTL_ERR_ALREADY_CLOSED;
		goto out;
	}

	if (!astItfList[pstDevice->uItfIdx].pstAdapterFns->pfnAdapterWrite)
	{
		UT_ErrorLog(GTL, "GTL_Write: Error astItfList[%i].pstAdapterFns->pfnAdapterWrite == NULL\n", pstDevice->uItfIdx);
		retCode = GTL_ERR_UNKNOWN;
		goto out;
	}

	if (SUCCESS != astItfList[pstDevice->uItfIdx].pstAdapterFns->pfnAdapterWrite(pstDevice->pvAdapterSpecInfo, pstMsg))
	{
		UT_ErrorLog(GTL, "GTL_Write: Error writing to device %d\n", deviceid);
		retCode = GTL_ERR_UNKNOWN;
		goto out;
	}

out:
	UT_Log(GTL, INFO, "GTL_Write : EXIT\n");
	return retCode;
}

/************************** GTL_Shutdown *********************************/

/*!
 *    This function should be called by VAPI Core module at shutdown \n
 *    It performs the following. fns\n 
 *    1. Calls the individual interface adapters' shutdown fns.\n
 *
 *    \return   On successful completion, it returns SUCCESS\n
 *              On failure, it returns GTL_ERR_NOT_INITIALIZED\n
 */
/***************************************************************************/

int GTL_Shutdown()
{
	int retCode = SUCCESS;

	UT_Log(GTL, INFO, "GTL_Shutdown: ENTRY\n");
	if (bGtlInit == 0)
	{
		UT_ErrorLog(GTL, "GTL_Shutdown : GTL not initialized\n");
		retCode = GTL_ERR_NOT_INITIALIZED;
		goto out;
	}

	/* Here we close general csme thread, so let it be for 0 adapter */
	if (SUCCESS != astItfList[0].pstAdapterFns->pfnAdapterShutdown())
	{
		UT_ErrorLog(GTL, "GTL_Shutdown: Error shutdown adapter\n");
		retCode = GTL_ERR_UNKNOWN;
		goto out;
	}

	/* Free All Interfaces */
	GTL_FreeAllInterfaces();

	GTL_FreeDevices(stGtlConfigData.uiNumDevices);

	UT_DeleteSortedArray(pstDeviceArray);

	UT_FreeMem(pastDevList);

	bGtlInit = 0;

out:
	UT_Log(GTL, INFO, "GTL_Shutdown: EXIT\n");
	return retCode;
}

/****************************************************************************
 * GTL_GetDefaultMaxChnls : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      - Get max channels depending on passed device type
 *
 *  \return 
 *  Default max channels
 *  
 */
U16 GTL_GetDefaultMaxChnls(U8 ucDevType)
{
	switch (ucDevType)
	{
	case DEV_TYPE_M83XXX:
		return MAX_CHANNELS_M83XXX;

	case DEV_TYPE_M821XX:
		return MAX_CHANNELS_M821XX;

	case DEV_TYPE_M823XX:
		return MAX_CHANNELS_M823XX;

	case DEV_TYPE_M825XX:
		return MAX_CHANNELS_M825XX;

	case DEV_TYPE_M826XX:
		return MAX_CHANNELS_M826XX;

	case DEV_TYPE_M827XX:
		return MAX_CHANNELS_M827XX;

	case DEV_TYPE_M828XX:
		return MAX_CHANNELS_M828XX;

	case DEV_TYPE_M829XX:
		return MAX_CHANNELS_M829XX;

	case DEV_TYPE_M823XX_2:
		return MAX_CHANNELS_M823XX_2;

	default:
		UT_ErrorLog(VCORE, "GTL_GetDefaultMaxChnls: Wrong device type (%u)\n", ucDevType);
	}
	return 0;
}

/****************************************************************************
 * GTL_GetDefaultMaxPartPerConf : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      - Get max participants per conference depending on passed device type
 *
 *  \return 
 *  Default max participants per conf
 *  
 */
U16 GTL_GetDefaultMaxPartPerConf(U8 ucDevType)
{
	switch (ucDevType)
	{
	case DEV_TYPE_M83XXX:
		return MAX_PART_PER_CONF_M83XXX;

	case DEV_TYPE_M821XX:
		return MAX_PART_PER_CONF_M821XX;

	case DEV_TYPE_M823XX:
		return MAX_PART_PER_CONF_M823XX;

	case DEV_TYPE_M825XX:
		return MAX_PART_PER_CONF_M825XX;

	case DEV_TYPE_M826XX:
		return MAX_PART_PER_CONF_M826XX;

	case DEV_TYPE_M827XX:
		return MAX_PART_PER_CONF_M827XX;

	case DEV_TYPE_M828XX:
		return MAX_PART_PER_CONF_M828XX;

	case DEV_TYPE_M829XX:
		return MAX_PART_PER_CONF_M829XX;

	case DEV_TYPE_M823XX_2:
		return MAX_PART_PER_CONF_M823XX_2;


	default:
		UT_ErrorLog(VCORE, "GTL_GetDefaultMaxPartPerConf: Wrong device type (%u)\n", ucDevType);
	}
	return 0;
}

/****************************************************************************
 * GTL_GetDefaultMaxConfPerDevice : The function does the following things -
 ***************************************************************************/
/*! 
 *  - Implementation
 *      - Get max participants per conference depending on passed device type
 *
 *  \return 
 *  Default max participants per conf
 *  
 */
U16 GTL_GetDefaultMaxConfPerDevice(U8 ucDevType)
{
	switch (ucDevType)
	{
	case DEV_TYPE_M83XXX:
		return MAX_CONF_PER_DEV_M83XXX;

	case DEV_TYPE_M821XX:
		return MAX_CONF_PER_DEV_M821XX;

	case DEV_TYPE_M823XX:
		return MAX_CONF_PER_DEV_M823XX;

	case DEV_TYPE_M825XX:
		return MAX_CONF_PER_DEV_M825XX;

	case DEV_TYPE_M826XX:
		return MAX_CONF_PER_DEV_M826XX;

	case DEV_TYPE_M827XX:
		return MAX_CONF_PER_DEV_M827XX;

	case DEV_TYPE_M828XX:
		return MAX_CONF_PER_DEV_M828XX;

	case DEV_TYPE_M829XX:
		return MAX_CONF_PER_DEV_M829XX;

	case DEV_TYPE_M823XX_2:
		return MAX_CONF_PER_DEV_M823XX_2;

	default:
		UT_ErrorLog(VCORE, "GTL_GetDefaultMaxConfPerDevice: Wrong device type (%u)\n", ucDevType);
	}
	return 0;
}

