/*! \file gtl.h 
 *
 * Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#ifndef _TMAP_MAP_H
#define _TMAP_MAP_H

#ifdef __cplusplus
extern "C" {
#endif

#include "ut.h"
#include "msp.h"

#ifndef _VAPI_H
typedef U32 DEVID;
#endif

#define MAX_PARAM   744
#define MAX_CHANNEL 128
#define SUPVSR_CHANNEL 0xFFFF

#define MAX_PART_PER_CONF_M821XX	16
#define MAX_PART_PER_CONF_M83XXX	24
#define MAX_PART_PER_CONF_M823XX	36
#define MAX_PART_PER_CONF_M825XX	64
#define MAX_PART_PER_CONF_M825XX_2	64
#define MAX_PART_PER_CONF_M826XX	64
#define MAX_PART_PER_CONF_M827XX	128
#define MAX_PART_PER_CONF_M828XX	64
#define MAX_PART_PER_CONF_M829XX	192
#define MAX_PART_PER_CONF_M823XX_2	36

#define MAX_CONF_PER_DEV_M821XX	8
#define MAX_CONF_PER_DEV_M83XXX	8
#define MAX_CONF_PER_DEV_M823XX	36
#define MAX_CONF_PER_DEV_M825XX	64
#define MAX_CONF_PER_DEV_M825XX_2	24
#define MAX_CONF_PER_DEV_M826XX	64
#define MAX_CONF_PER_DEV_M827XX	128
#define MAX_CONF_PER_DEV_M828XX	32
#define MAX_CONF_PER_DEV_M829XX	192
#define MAX_CONF_PER_DEV_M823XX_2	36

#define MAX_CHANNELS_M821XX		16
#define MAX_CHANNELS_M83XXX		24
#define MAX_CHANNELS_M823XX		160
#define MAX_CHANNELS_M825XX		128
#define MAX_CHANNELS_M825XX_2		64
#define MAX_CHANNELS_M826XX		256
#define MAX_CHANNELS_M827XX		404
#define MAX_CHANNELS_M828XX		64
#define MAX_CHANNELS_M829XX		640
#define MAX_CHANNELS_M823XX_2		160



/* Data or Command/Response Indication */
#define DCR_BIT             0x0002
#define DCR_DATA            0x0000
#define DCR_CMD_RESP        0x0002

/* Boot Loader Message Indication */
#define BLM_BIT             	0x0001
#define MAAS_ASSIGN_CMD_TYPE	0x1b

enum {
	DEV_TYPE_M825XX,
	DEV_TYPE_M826XX,
	DEV_TYPE_M827XX,
	DEV_TYPE_M828XX,
	DEV_TYPE_M829XX,
	DEV_TYPE_M823XX,
	DEV_TYPE_M821XX,
	DEV_TYPE_M83XXX,
	DEV_TYPE_M823XX_2,
	NUM_DEV_TYPES
};

#define MAX_ETH_NAME_SIZE	17
#define MAX_PCI_NAME_SIZE	50
#define MAX_IFS			64

#define MAX_FIFO_PCI_SIZE	256

#define BOOT_MODE_LEGACY	0
#define BOOT_MODE_STANDARD	1
#define BOOT_MODE_ALTERNATE	2

#define BOOT	0x01
#define PCI	0x02
#define CSME	0x04
#define POS	0x08
#define DATA	0x10

typedef struct comcerto_api_hdr SApiHdr;

typedef struct comcerto_api_hdr_boot SBootHdr;

/** Structure of GTL PCI Mailbox for boot operation*/
typedef struct {
	U16 reg3;
	U16 reg2;
	U16 reg1;
	U16 reg0;
} gtl_mailbox_t;

/** Structure of GTL Message */
typedef struct {
	gtl_mailbox_t mailbox;		/**< PCI mailbox descriptor*/
	unsigned int fifo_size;		/**< Size of Fifo */
	void *fifo;	 		/**< Fifo */
	U8 type;			/**< type of the message (boot or not) | interface type (PCI, ETH, POS)*/
	U16 channel;			/**< Channel to send the message to */
	U16 max_fifo_size;		/**< Max number of byte that the allocated fifo can hold */
} gtl_msg_t;

/** Enumeration of Interface types */
typedef enum {
	eCSM_ITF = 0,
		  /**< CSME  Interface*/
	ePCI_ITF = 1,
		  /**< PCI Interface*/
	ePOS_ITF = 2
		  /**< POS Interface*/
} EDevItfType;

/** Enumeration of Device mode  */
typedef enum {
	eMASTER = 0,
		 /**< Master mode*/
	eSLAVE = 1
	       /**< Slave mode*/
} EDevMode;

/** Enumeration of Device States */
typedef enum {
	eIDLE = 0,
	       /**< Device IDLE*/
	eACTIVE = 1
		/**< Device ACTIVE*/
} EDeviceState;

/** Structure describing Device Information*/
typedef struct {
	S32 devId;			/**<Device ID, unique across VAPI Library*/
	EDevItfType eDevItfType;	/**<Interface Type Identifier*/
	U32 uItfIdx;			/**<Index to interface in the interface list*/
	EDevMode eDevMode;		/**<Device Mode*/
	U8 devMac[6];			/**<Device MAC Address*/
	EDeviceState eDevState;		/**<State of Device */
	U16 usMaxChnls;			/**<Max channels supported by device*/
	U8 ucDevType;			/**<Device Type*/
	void *pvDevAddrInfo;		/**<Device Addressing Info, Interface dependent*/
	void *pvAdapterSpecInfo;	/**<Adapter Specific device information */
	SArrayNode stArrayNode;
	U8 devHardMac[6];		/**< M823xx only preconfigured Device MAC Address */
	U8 ucEthBootMode;		/**< M823xx only Boot mode (legacy, standard or alt) */
} SGtlDeviceInfo;

/** Structure describing the CSME Addr. Info.*/
typedef struct {
	U8 devMac[6];		/**< Device MAC Address*/
	U8 hostMac[6];		/**< Host MAC Address*/
	U8 devHardMac[6];	/**< M823xx only, preconfigured Device MAC Address */
	U8 ucEthBootMode;	/**< M823xx only, Boot mode (legacy, standard or alt) */
	int ackRequired;	/**< Ack required or not*/
	U16 usMaxChannels;	/** It is need here again as only this structure is passed
					to the adapter Max channels supported by device*/
	char *pcEthName;	/**< Interface for comm. with device */

	void *pvAdptData;	/**< Pointer to hold adapter specific information, 
					unusable for the layers above */
} SGtlCsmeItf;

/** Structure describing the PCI Addr. Info.*/
typedef struct {
	char *pciDevName;	 /**< Device Name*/
	U16 usMaxChannels; /** Max channels supported by device*/
} SGtlPciItf;

/** Enumeration of POS Path Type */
typedef enum {
	ePOSoTSP3 = 0,
	ePOSoPCICM = 1
} EPosPathType;

#define POS_PATH_INFO_MAGIC		0xA31CC13A		/* whatever */
typedef struct {
	int magic;
    U8 ucPhyId;  /**< Physical Identifier*/
    EPosPathType ePathType;
    /**< Physical Path Type*/
    struct {
		U8 DevId;
		U32 MuxChannID;
		U32 DemuxChannID;
		U8 *pDummyGtlMsgHdr;
    } Tsp3;		/* Path Type ePosoTsp3 */
    struct {
		S32 pciDevId;  /**<Device ID, unique across VAPI Library*/
		int pciFd;
    } PciCM;	/* Path Type ePOSoPCICM */
} PosPathInfo;

/** Structure describing the POS Addr. Info.*/
typedef struct {
	U8 devMac[6];
		   /**< Device MAC Address*/
	U8 hostMac[6];
		    /**< Host MAC Address*/
	U8 devHardMac[6];	/**< M823xx only, preconfigured Device MAC Address */
	U8 ucEthBootMode;	/**< M823xx only, Boot mode (legacy, standard or alt) */
	int ackRequired;
		     /**< Ack required or not*/
	U16 usMaxChannels;
		       /** Max channels supported by device*/
	char *pcEthName;	/*Interface for comm. with device */
	PosPathInfo *pvAdptData;	/* Pointer to hold adapter specific information, usable anywhere */
} SGtlPosItf;

/** Union contains the data exchanging interfaces */
typedef union _GtlInterface
{
	SGtlCsmeItf csme;	/**<Contains the PCI Addr. info */
	SGtlPciItf pci;		/**<Contains the CSME Addr. info */
	SGtlPosItf pos;		/**<Contains the POS Addr. info */
	/* add new interface description here */
} GtlInterface;

/** Structure describing the physical interface Info.*/
typedef struct {
	EDevItfType type;	/**<Enumerated data type for device interface CSME/PCI , etc*/
	EDevMode eDevMode;	/**<Mode of device Master/Slave*/
	U8 ucDevType;		/**<Device Type*/
	U16 usMaxChnls;		/**<Max Chnls*/
	U8 ucEthBootMode;	/**< ethernet boot mode for the M823xx*/
	GtlInterface *pvItfInfo; 
} gtl_phy_itf_t;

/** Structure listing the funcptrs of adapters */
typedef struct {
	int (*pfnAdapterOpen) (S32 deviceid, void *pvDevAddrInfo, 
		void **pvAdapterSpecInfo, EDevMode eMode);	/**< Open Fn*/

	int (*pfnAdapterClose) (void *pvDevAddrInfo);		/**< Close Fn*/

	int (*pfnAdapterReset) (void *pvDevAddrInfo, void *pvAdapterSpecInfo);	/**< Reset Fn*/

	int (*pfnAdapterWrite) (void *pvDevAddrInfo, gtl_msg_t * pstMsg);	/**< Write Fn*/

	int (*pfnAdapterShutdown) (void);				/**< Shutdown Fn*/
} STAdapterFns;

/** Desription: This function is used by GTL  to send messages
                to GTL client.GTL invokes this callback to deliver
                the message to the client. */
typedef void (*GTL_READ_CB) (S32 deviceid, gtl_msg_t * msg);

/** Structure describing Interface Information*/
typedef struct {
	EDevItfType eItfType;
			  /**< Interface type,ePCI_ITF or eCSM_ITF */
	int (*pfnItfInitFn) (GTL_READ_CB, STAdapterFns *, SGtlDeviceInfo *, U32);
									     /**< Interface Init fn*/
	STAdapterFns *pstAdapterFns;
} SItfDesc;

/* To be used internally by the GTL to retrieve struct type */
typedef struct _SDeviceHeader {
	void *pvNext;		/* Pointer to the next device struct */
	U16 usControlInterface;	/* Type of Data struct in pvNext */

	DEVID uiDevId;		/* Device Id */
	U32 uiVersion;		/* Version Info */
	EDevMode eDevMode;	/* eMASTER or eSLAVE */
	U8 ucDevType;		/* DEV_TYPE_M825XX,DEV_TYPE_M827XX,DEV_TYPE_M826XX etc */
	Boolean bUseDefaultMaxChnls;	/* 1 = device default, 0 = custom */
	U16 usMaxChannels;	/* Ignored if bUseDefaultMaxChnls = 1 */
} SDeviceHeader;

typedef struct _SPCIUsrData {
	/*Common info */
	void *pvNext;		/* Pointer to the next device struct */
	U16 usControlInterface;	/* Type of Data struct in pvNext */

	DEVID uiDevId;		/* Device Id */
	U32 uiVersion;		/* Version Info */
	EDevMode eDevMode;	/* eMASTER or eSLAVE */
	U8 ucDevType;		/* DEV_TYPE_M825XX,DEV_TYPE_M827XX,DEV_TYPE_M826XX etc */
	Boolean bUseDefaultMaxChnls;	/* 1 = device default, 0 = custom */
	U16 usMaxChannels;	/* Ignored if bUseDefaultMaxChnls = 1 */

	/*Interface Specific info */
	char *pucPCIDevName;	/* PCI Dev name */
} SPCIUsrData;

typedef struct _SCSMEUsrData {
	/*Common info */
	void *pvNext;		/* Pointer to the next device struct */
	U16 usControlInterface;	/* Type of Data struct in pvNext */

	DEVID uiDevId;		/* Device Id */
	U32 uiVersion;		/* Version Info */
	EDevMode eDevMode;	/* eMASTER or eSLAVE */
	U8 ucDevType;		/* DEV_TYPE_M825XX,DEV_TYPE_M827XX,DEV_TYPE_M826XX etc */
	Boolean bUseDefaultMaxChnls;	/* 1 = device default, 0 = custom */
	U16 usMaxChannels;	/* Ignored if bUseDefaultMaxChnls = 1 */

	/*Interface Specific info */
	U8 aucDevMac[6];	/* Device MAC Address */
	U8 aucHostMac[6];	/* Host MAC Address (used in MAAS ASSIGN only
				   for CSME Slave) */
	char *pucEthDevName;	/* Interface name on which communication with
				   device is takin place eg ETH0 or ETH1 */
	U8 ucIsAckReqd;		/* 1 : ACK is required
				   0 : ACK not required */

	U8 aucDevHardMac[6];	/* Device preconfigured M823XX MAC Address */
	U8 ucEthBootMode;	/* define the ethernet boot mode for slave device (M823XX only)
				   0 : Backward compatible mode (default)
				   1 : Standard mode
				   2 : Alternate mode */
} SCSMEUsrData;

typedef struct _SPOSUsrData {
	/*Common info */
	void *pvNext;		/* Pointer to the next device struct */
	U16 usControlInterface;	/* Type of Data struct in pvNext */

	DEVID uiDevId;		/* Device Id */
	U32 uiVersion;		/* Version Info */
	EDevMode eDevMode;	/* eMASTER or eSLAVE */
	U8 ucDevType;		/* DEV_TYPE_M825XX,DEV_TYPE_M827XX,DEV_TYPE_M826XX etc */
	Boolean bUseDefaultMaxChnls;	/* 1 = device default, 0 = custom */
	U16 usMaxChannels;	/* Ignored if bUseDefaultMaxChnls = 1 */

	/*Interface Specific info */
	U8 ucPhyId;	    /**< Physical Identifier*/
	EPosPathType ePathType;
			    /**< Physical Path Type*/
	U8 aucDevMac[6];	/* Device MAC Address */
	U8 aucHostMac[6];	/* Host MAC Address (used in MAAS ASSIGN only
				   for CSME Slave) */
	char *pucEthDevName;	/* Interface name on which communication with
				   device is taking place eg pos0 */
	U8 ucIsAckReqd;		/* 1 : ACK is required
				   0 : ACK not required */
	union {
		struct {
			U8 DevId;
		} Tsp3;		/* Path Type ePosoTsp3 */
		struct {
			S32 pciDevId;  /**<Device ID, unique across VAPI Library*/
			int pciFd;
		} PciCM;	/* Path Type ePOSoPCICM */
	} PathInfo;
	int ether_unit;
} SPOSUsrData;

typedef struct _SGtlConfigData {
	U32 uiNumDevices;
} SGtlConfigData;

/** Desription: This function is called by GTL clients to send messages
                to the MSP. */
int GTL_Write(IN S32 deviceid, IN gtl_msg_t * msg);

/** Desription: This function should be called by GTL clients at
                start up before issuing any other command. It
                initializes the GTL internal data structures, sets the
                client callback function  and spawns the receiving
                thread.*/
int GTL_Init(SDeviceHeader *, GTL_READ_CB);

/** Desription: This function will provide interface details to the GTL client.*/
int GTL_QueryInterface(S32 deviceid, gtl_phy_itf_t *);

/** Desription: This function will reset the device. This function is board specific.
                Different boards may have different board dependent mechanism.
                If no such mechanism is available then GTL Reset can be empty.*/
int GTL_Reset(S32 deviceid);

/** Desription: This function activates the device. Before call to this 
                function, GTL client will not be able to send and recive 
                any message to and from MSP.*/
int GTL_Open(S32 deviceid);

/** Desription: This function de-activates the device.*/
int GTL_Close(S32 deviceid);

/** Desription: This function should be called by GTL clients to
                shutdown the various GTL Adapters*/
int GTL_Shutdown(void);

int GTL_FreeAllInterfaces(void);

/* Get max channels depending on passed device type*/
U16 GTL_GetDefaultMaxChnls(U8 ucDevType);

/* Get default max participants per conf */
U16 GTL_GetDefaultMaxPartPerConf(U8 ucDevType);

/* Get default max conference per device */
U16 GTL_GetDefaultMaxConfPerDevice(U8 ucDevType);

/* This function can be used for any cleanup tasks like freeing memory etc */
int QueryDeviceCleanup(void);

void GTL_PrintMessage(IN gtl_msg_t * pstMsg, U8 leyer, IN Boolean bIsTx);
#if !defined(VAPI_NODBG)
void GTL_PrintCsmeMsg(IN gtl_msg_t *pstMsg);
#else
#define GTL_PrintCsmeMsg(msg)
#endif

/*
 * List of error codes
 */

#define GTL_ERROR_BASE     66000
#define GTL_ERR_START    -(GTL_ERROR_BASE)
#define GTL_ERR_INVALID_DEVID      -(GTL_ERROR_BASE + 1)
#define GTL_ERR_INVALID_CONNID     -(GTL_ERROR_BASE + 2)
#define GTL_ERR_NOMEM          -(GTL_ERROR_BASE + 3)
#define GTL_ERR_INVALID_PARAM      -(GTL_ERROR_BASE + 4)
#define GTL_ERR_NULL_POINTER_PASSED    -(GTL_ERROR_BASE + 5)
#define GTL_ERR_MSP_NO_RESPONSE    -(GTL_ERROR_BASE + 6)
#define GTL_ERR_INVALID_STATISTICS_TYPE -(GTL_ERROR_BASE + 7)
#define GTL_ERR_UNKNOWN        -(GTL_ERROR_BASE + 8)
#define GTL_ERR_ALREADY_INITIALIZED    -(GTL_ERROR_BASE + 9)
#define GTL_ERR_MAX_DEV_OPEN       -(GTL_ERROR_BASE + 10)
#define GTL_ERR_ALREADY_OPEN    -(GTL_ERROR_BASE + 11)
#define GTL_ERR_ALREADY_CLOSED    -(GTL_ERROR_BASE + 12)
#define GTL_ERR_DATA_MSG    -(GTL_ERROR_BASE + 13)
#define GTL_ERR_NOT_INITIALIZED    -(GTL_ERROR_BASE + 14)
#define GTL_ERR_END    -(GTL_ERROR_BASE + 15)
#define GTL_ERR_INVAL_CNTRL_ITF		-(GTL_ERROR_BASE + 16)
#define GTL_ERR_CONFIG      -(GTL_ERROR_BASE + 17)


#ifdef __cplusplus
};
#endif

#endif	/* _TMAP_MAP_H */

