/*! \file vxcompat.c 
	\brief This file contains some VxWorks compatibilities functions
*/

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include <vxWorks.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sysLib.h>
#include <taskLib.h>
#include "vxcompat.h"
#include "vapi_san_type.h"

static int argp_find_option(const struct argp_option *options, int key)
{
	while (options->name != NULL)
	{
		if (options->key == key)
			return 0;
		options++;
	}

	return -1;
}

static void argp_show_help(const struct argp_option *options)
{
	while (options->name != NULL)
	{
		PDEBUG(DEBUG_INFO, "-%c [%s]\t%s", options->key, options->arg, options->doc);
		options++;
	}
	PDEBUG(DEBUG_INFO, "\n-[?|h]\tThis help message\n" "-V\tVersion information");
}

int argp_parse(struct argp *argp, char *args)
{
	char c, *option_arg_start = NULL;
	int state = 0, option = 0;

	for (; *args; args++)
	{
		c = *args;
		switch (state)
		{
		case 0:	/* default state */
			/* skipping spaces and tabs, ranting otherwise */
			if (c == ' ' || c == '\t')
				break;

			if (c == '-')
			{
				state = 1;
				break;
			}

			PDEBUG(DEBUG_INFO, "Invalid arguments, pass '-?' or '-h' for help");
			return -1;

		case 1:	/* '-' found, trying short option name */
			if (c == '?' || c == 'h')
			{
				argp_show_help(argp->options);
				return 1;
			}
			if (c == 'V')
			{
				extern char *argp_program_version;

				PDEBUG(DEBUG_INFO, "%s", argp_program_version);
				return 1;
			}

			if (argp_find_option(argp->options, c))
			{
				PDEBUG(DEBUG_INFO, "unknown option: '-%c'", c);
				return -1;
			}

			option = c;
			state = 2;
			break;

		case 2:	/* searching option argument, assuming that all options require that */
			if (c == ' ' || c == '\t')
				break;
			if (c == '=')
			{
				state = 3;
				break;
			}

			/* everything else is option argument start */
			option_arg_start = args;
			state = 3;
			break;

		case 3:	/* scanning option argument */
			if (c == ' ' || c == '\t')
			{
				/* that's the end of the argument */
				*args = 0;
				if (argp->parser(option, option_arg_start, NULL))
					return -1;
				state = 0;
			}
		}
	}

	/* don't forget about the last option */
	if (state == 3)
		if (argp->parser(option, option_arg_start, NULL))
			return -1;

	return 0;
}

int strcasecmp(const char *s1, const char *s2)
{
	char c1, c2;

	if (!s1 || !s2)
		abort();

	while (1)
	{
		c1 = tolower(*s1++);
		c2 = tolower(*s2++);

		if (c1 < c2)
			return -1;
		if (c1 > c2)
			return 1;

		if (!c1 && !c2)
			break;
	}

	return 0;
}

char *strdup(const char *s)
{
	int len;
	char *dublicate;

	if (!s)
		abort();

	len = strlen(s);
	if (len < 0)
		abort();

	dublicate = (char *)malloc(len + 1);
	if (dublicate)
		memcpy(dublicate, s, len + 1);

	return dublicate;
}
