/*!	\file vapi_san_util.c
	\brief This file contains some utilities functions
*/

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#if !defined(_VXWORKS_)
#include <vapi/msp.h>
#include <vapi/vapi_build_option.h>
#include <vapi/comcerto-ud-types.h>
#else
#include <msp.h>
#include <comcerto-ud-types.h>
#include <vapi_build_option.h>
#endif

#include <errno.h>
#include <string.h>

#include "readcfg.h"
#include "vapi_san_type.h"
#include "vapi_san_var.h"
#include "vapi_san_func.h"


/* default values used in default_device_configuration */
#define DEVICE_CONTROL_MAC		{0x00,0x11,0x22,0x33,0x44,0x55}
#define CSME_HOST_CONTROL_INTERFACE	"eth0"
#define PCI_HOST_CONTROL_INTERFACE	"/dev/comcerto0"
#define MAX_STRING_INTERFACE_NAME	20

/* these function code are missing in msp.h*/
#define FC_GET_ARM_CODE_VERSION 	UT_CPU2LE16(0x0021)
#define FC_GET_SPU_CODE_VERSION 	UT_CPU2LE16(0x0022)
#define FC_GET_DEVICE_TYPE 		UT_CPU2LE16(0x0031)

static char control_interface_name[MAX_STRING_INTERFACE_NAME];
static U16 device_model = 0;

/** Structure to define the default values for the device configuration
 * in case he parameters are not found in the configuration file
 */
SCSMEUsrData default_csme_device_configuration = {
	.pvNext = NULL,			/**< pointer to next device */
	.usControlInterface = eCSM_ITF,	/**< control interface to use */
	.uiDevId = 0,			/**< device ID */
	.uiVersion = 0,			/**< Version infor (not used) */
	.eDevMode = eMASTER,		/**< Slave / Master mode */
	.ucDevType = DEV_TYPE_M823XX,	/**< Device type */
	.bUseDefaultMaxChnls = True,	/**< Default or custom max channels */
	.usMaxChannels = 0,		/**< Max Channels if above flag is custom */
	.pucEthDevName = (char *)CSME_HOST_CONTROL_INTERFACE,	/**< host interface used to control the device */

	.aucDevMac = DEVICE_CONTROL_MAC,		/**< MAC address assigned to the device for control over csmencaps */
	.aucHostMac = {0x00,0x00,0x00,0x00,0x00,0x00},	/**< Mac address of the network interface of the host controlling the device */
	.ucIsAckReqd = 1				/**< csme ack required */
};

SPCIUsrData default_pci_device_configuration = {
	.pvNext = NULL,			/**< pointer to next device */
	.usControlInterface = ePCI_ITF,	/**< control interface to use */
	.uiDevId = 0,			/**< device ID */
	.uiVersion = 0,			/**< Version infor (not used) */
	.eDevMode = eSLAVE,		/**< Slave / Master mode */
	.ucDevType = DEV_TYPE_M823XX,	/**< Device type */
	.bUseDefaultMaxChnls = True,	/**< Default or custom max channels */
	.usMaxChannels = 0,		/**< Max Channels if above flag is custom */
	.pucPCIDevName = (char *)PCI_HOST_CONTROL_INTERFACE,	/**< host interface used to control the device */
};

static int m823xx_refclk = 25;
static int m823xxv2_amba_clk = 137;

void display_gtl_config(SDeviceHeader *device_cfg)
{
	PDEBUG(SANITY_INIT, "CONTROL_INTERFACE_TYPE  = %d", device_cfg->usControlInterface);
	PDEBUG(SANITY_INIT, "DEVICE_ID    = %d", device_cfg->uiDevId);
	PDEBUG(SANITY_INIT, "DEVICE_MODE  = %d", device_cfg->eDevMode);
	PDEBUG(SANITY_INIT, "DEVICE_TYPE  = %d", device_cfg->ucDevType);
	PDEBUG(SANITY_INIT, "USE_DEFAULT_MAX_CHANNEL = %d", device_cfg->bUseDefaultMaxChnls);
	PDEBUG(SANITY_INIT, "CUSTOM_MAX_CHANNEL = %d", device_cfg->usMaxChannels);
	if(device_cfg->usControlInterface == ePCI_ITF)
		PDEBUG(SANITY_INIT, "HOST_CONTROL_INTERFACE, %s", ((SPCIUsrData *)device_cfg)->pucPCIDevName);
	
	if(device_cfg->usControlInterface == eCSM_ITF)
	{
		PDEBUG(SANITY_INIT, "HOST_CONTROL_INTERFACE, %s", ((SCSMEUsrData *)device_cfg)->pucEthDevName);

		PDEBUG(SANITY_INIT, "ACK_REQUIRED = %d", ((SCSMEUsrData *)device_cfg)->ucIsAckReqd);
		print_mac_addr("DEVICE MAC   = ", ((SCSMEUsrData *)device_cfg)->aucDevMac);
		print_mac_addr("HOST MAC     = ", ((SCSMEUsrData *)device_cfg)->aucHostMac);
	}
}
/*=================================================================================*/
/*!
 *	\brief This function reads the SETUP_CONFIG section parameters from the configuration (passed on command line with the -c option).\n
 *	It uses the macros defined in readcfg (i.e READ_INT).\n
 *	
 *	\section Inputs-Outputs
 *		\param config_file configuration file name
 *		\return void
*/
/*=================================================================================*/
int set_device_configuration(const char *config_file, void *gtl_config, int device_ctrl_itf)
{
	struct _CFG *cfg_info = NULL;
	SDeviceHeader *default_config;
	SDeviceHeader *device_cfg = (SDeviceHeader *)gtl_config;

	if (config_file && *config_file)
		cfg_info = cfg_read(config_file, 0);

	if (device_ctrl_itf == eCSM_ITF)
	{
		default_config = (SDeviceHeader *)&default_csme_device_configuration;
		memset(device_cfg, 0, sizeof(SCSMEUsrData));
	}
	else if (device_ctrl_itf == ePCI_ITF)
	{
		default_config = (SDeviceHeader *)&default_pci_device_configuration;
		memset(device_cfg, 0, sizeof(SPCIUsrData));
	}
	else
	{
		PDEBUG(SANITY_ERROR, "set_device_configuration: Interface control type %d not supported", device_ctrl_itf);
		return FAILURE;
	}

	if (!cfg_info)
	{
		PDEBUG(SANITY_ERROR, "Config read: Can't open config file \"%s\", using default config", config_file);

		if (device_ctrl_itf == eCSM_ITF)
		{
			if(default_config->usControlInterface != eCSM_ITF)
			{
				PDEBUG(SANITY_ERROR, "set_device_configuration: control interface not configured for eCSM_ITF");
				return FAILURE;
			}
			else
			{
				memcpy(device_cfg, default_config, sizeof(SCSMEUsrData));
				display_gtl_config(device_cfg);
				return SUCCESS;
			}
		}
		else if (device_ctrl_itf == ePCI_ITF)
		{
			if(default_config->usControlInterface != ePCI_ITF)
			{
				PDEBUG(SANITY_ERROR, "set_device_configuration: control interface not configured for ePCI_ITF");
				return FAILURE;
			}
			else
			{
				memcpy(device_cfg, default_config, sizeof(SPCIUsrData));
				display_gtl_config(device_cfg);
				return SUCCESS;
			}
		}
	}

	device_cfg->usControlInterface = device_ctrl_itf; /*the control interface type comes from the -d option */

	device_cfg->uiDevId = default_config->uiDevId; /*Don't really care about the value */

	READ_INT(cfg_info, "SETUP_CONFIG", "DEVICE_MODE", device_cfg->eDevMode, default_config->eDevMode);

	READ_INT(cfg_info, "SETUP_CONFIG", "DEVICE_TYPE", device_cfg->ucDevType, default_config->ucDevType);

	/* C300 EVM has 10MHz or 25MHz REFCLK */
	if (device_cfg->ucDevType == DEV_TYPE_M823XX)
		READ_INT(cfg_info, "SETUP_CONFIG", "M823XX_REFCLK", m823xx_refclk, 25);

	/* C300V2 EVM AMBA clk can be 137.5MHz or 150MHz */
	if (device_cfg->ucDevType == DEV_TYPE_M823XX_2)
		READ_INT(cfg_info, "SETUP_CONFIG", "M823XXV2_AMBA_CLK", m823xxv2_amba_clk, 137);

	READ_INT(cfg_info, "SETUP_CONFIG", "USE_DEFAULT_MAX_CHANNEL",device_cfg->bUseDefaultMaxChnls, default_config->bUseDefaultMaxChnls);

	READ_INT(cfg_info, "SETUP_CONFIG", "CUSTOM_MAX_CHANNEL", device_cfg->usMaxChannels, default_config->usMaxChannels);

	if(device_cfg->usControlInterface == ePCI_ITF)
	{
		READ_STRN(cfg_info, "SETUP_CONFIG", "PCI_HOST_CONTROL_INTERFACE", control_interface_name, 
						MAX_STRING_INTERFACE_NAME, default_pci_device_configuration.pucPCIDevName);
		((SPCIUsrData *)device_cfg)->pucPCIDevName = (char *)control_interface_name;
	}
	else if(device_cfg->usControlInterface == eCSM_ITF)
	{
		READ_INT(cfg_info, "SETUP_CONFIG", "ACK_REQUIRED", ((SCSMEUsrData *)device_cfg)->ucIsAckReqd, default_csme_device_configuration.ucIsAckReqd);

		READ_MAC(cfg_info, "SETUP_CONFIG", "DEVICE_CONTROL_MAC", ((SCSMEUsrData *)device_cfg)->aucDevMac, default_csme_device_configuration.aucDevMac);

		READ_STRN(cfg_info, "SETUP_CONFIG", "CSME_HOST_CONTROL_INTERFACE", control_interface_name, 
						MAX_STRING_INTERFACE_NAME, default_csme_device_configuration.pucEthDevName);
		((SCSMEUsrData *)device_cfg)->pucEthDevName = (char *)control_interface_name;

		UT_GetInterfaceMac(((SCSMEUsrData *)device_cfg)->pucEthDevName, (char *)((SCSMEUsrData *)device_cfg)->aucHostMac);

	}

	cfg_clean(cfg_info);

	display_gtl_config(device_cfg);

	return SUCCESS;
}

/*=================================================================================*/
/*!
*	\brief	This function set an event for the endpoint
*	It is used from the state machine or the asynchronous callback function (comcerto_response_handler).\n
*	
*	\section Inputs-Outputs
*		\param event_type event to set
*		\param index index od the endpoint in the global endpoints array.
*		\return void
*/
/*=================================================================================*/
void set_endpoint_event(int event_type, int index)
{
	endpoints[index].event = event_type;
	PDEBUG(SANITY_FUNCTION, "Endpoint %d set event %d (state %d)", index, endpoints[index].event,
	       endpoints[index].state);
}

/*=================================================================================*/
/*!
*	\brief	This function set a state for the endpoint
*	It is used from the state machine.\n
*	
*	\section Inputs-Outputs
*		\param state event to set
*		\param index index od the endpoint in the global endpoints array.
*		\return void
*/
/*=================================================================================*/
void set_endpoint_state(int state, int index)
{
	endpoints[index].state = state;
	PDEBUG(SANITY_FUNCTION, "Endpoint %d, set state %d (event %d)", index, endpoints[index].state,
	       endpoints[index].event);
}

/*=================================================================================*/
/*!
*	\brief This function handle the endpoint timed connection.
*	It is called when the endpoint timer has expired \n
*	
*	\section Inputs-Outputs
*		\param timer_data
*		\return void
*/
/*=================================================================================*/
void timed_connection (void *timer_data)
{
	struct _ENDPOINT_DESC *this_endpoint = (struct _ENDPOINT_DESC*)timer_data;
	
	PDEBUG(SANITY_FUNCTION, "Endpoint %d, on time connection timed out", this_endpoint->index);
	UT_TimerStop(this_endpoint->this_timer);
	
	set_endpoint_event(ENDPOINT_EVENT_CONNECTION_TIMED, this_endpoint->index);
}
/*=================================================================================*/
/*!
 *	\brief This function reads the firmware (.axf or .elf) and store it to a buffer.\n
 *	
 *	\section Inputs-Outputs
 *		\param filename
 *		\param buffer address
 *		\param buffer size of the buffer
 *		\return
 *			0 if OK \n
 *			-1 if error \n
 */
/*=================================================================================*/
int comcerto_read_firmware_file(char *filename, unsigned char **buf, int *size)
{
	FILE *fp;

	fp = fopen(filename, "r");

	/* if the firmware file can't be open, 
	   we consider is it a valid situation and skip the boot sequence.
	   we assume we are in master mode config */
	if (fp == NULL)
	{
		perror(filename);
		goto err0;
	}

	/* Figure out how big the size of the file and allocate that much space in memory */
	fseek(fp, 0, SEEK_END);
	*size = ftell(fp);

	*buf = (unsigned char *)malloc(*size);
	if (*buf == NULL)
	{
		PDEBUG(SANITY_ERROR, "error allocating memory for CSM code\n");
		goto err1;
	}

	fseek(fp, 0, SEEK_SET);

	if (fread(*buf, sizeof(unsigned char), *size, fp) != *size)
	{
		PDEBUG(SANITY_ERROR, "error reading CSM code file %s\n", filename);
		goto err2;
	}

	fclose(fp);

	return 0;

err2:
	free(*buf);

err1:
	fclose(fp);

err0:
	return -1;
}

/*=================================================================================*/
/*!
*	\brief	This function boots the device.
*		it calls:
*		- comcerto_read_firmware_file to read the firmware file
*		- VAPI_AssignBootMAC
*		- VAPI_BootDevice
*	
*	\section Inputs-Outputs
*		\param firmware buffer containing the firmware.
*		\return
*			0 if OK \n
*			error if not 0 \n
*/
/*=================================================================================*/
int boot_device(const char *firmware)
{
	int result = 0, i;
	unsigned char *firmware_buf;
	int firmware_size;
	SHwCfg hw_cfg, *hw_cfg_param = NULL;


	memset(&hw_cfg, 0, sizeof(hw_cfg));

	/* specify refclk for the C300 EVM */
	if (gtl_device_configuration->ucDevType == DEV_TYPE_M823XX)
	{
		hw_cfg.usDeviceRefclk = m823xx_refclk;
		hw_cfg.usHwParamsMask = VAPI_HWCFG_REFCLK_MASK;

		hw_cfg_param = &hw_cfg;
		PDEBUG(SANITY_INFO, "C300 EVM REFCLK: %dMHz", m823xx_refclk);
	}

	/* specify AMBA clk for the C300V2 EVM */
	if (gtl_device_configuration->ucDevType == DEV_TYPE_M823XX_2)
	{
		hw_cfg.usDeviceAmbaClkM823V2 = m823xxv2_amba_clk;
		hw_cfg.usHwParamsMask |= VAPI_HWCFG_M823V2_HIGH_SPEED_MASK;

		hw_cfg_param = &hw_cfg;
		PDEBUG(SANITY_INFO, "C300V2 EVM AMBA CLK: %dMHz", m823xxv2_amba_clk);
	}

	if (!firmware || !*firmware)
	{
		PDEBUG(SANITY_ERROR, "No firmware file specified for SLAVE mode boot\n");
		return FAILURE;
	}

	/* Send MAAS_ASSIGN only for CSME*/
	if (gtl_device_configuration->usControlInterface == eCSM_ITF)
	{

		PDEBUG(SANITY_INIT, "Trying to assign boot MAC");
		for (i = 0; i < 1; i++)
		{
			result = VAPI_AssignBootMAC(device_id, NULL);
	
			if (result == SUCCESS)
				break;
	
			if (result == VAPI_ERR_MSP_NO_RESPONSE)
				PDEBUG(SANITY_ERROR, "Timeout try number %d\n", i);
			else
				PDEBUG(SANITY_ERROR, "Error %d (attempt %d)\n", result, i);
		}
		
		if (result != SUCCESS)
			return result;
	}

	PDEBUG(SANITY_INIT, "Trying to boot device");

	if (boot_option_file == 0) {
		result = comcerto_read_firmware_file((char *)firmware, &firmware_buf, &firmware_size);
		if (result)
			return FAILURE;

		PDEBUG(SANITY_INIT, "Firmware size = %d (%s)\n", firmware_size, firmware);

		result = VAPI_BootDevice(device_id, firmware_buf, firmware_size, hw_cfg_param, NULL);

		free(firmware_buf);
	}
	else
		result = VAPI_BootDeviceFile(device_id, (char *)firmware, hw_cfg_param, NULL);

	return result;
}

/*=================================================================================*/
/*!
*	\brief	This function initializes the TDM buses.
*	The parameters are comming from the DEVICE_TDM_CONFIG section 
*	in the configuration file (_c option) \n
*	It calls the VAPI_SetTDMParams.\n
*	Note the 4 TDM buses have the same configurations.\n
*	
*	\section Inputs-Outputs
*		\param config_file confiuration filename (-c option)
*		\return
*			SUCCESS \n
*			or status returned VAPI_SetTDMParams \n
*/
/*=================================================================================*/
int tdm_parameters_initialisation(const char *config_file)
{
	struct _CFG *cfg_info = NULL;
	STdmSetupParams tdm_parameters;
	int result;

	if (config_file && *config_file)
		cfg_info = cfg_read(config_file, 0);

	if (!cfg_info)
	{
		PDEBUG(SANITY_ERROR, "Config read: Can't open config file \"%s\"", config_file);
		return -1;
	}
	    /**< 1 : SINGLE_BUS_MODE */
	    /**< 2 : TWO_BUS_MODE */
	    /**< 4 : FOUR_BIUMODE */
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TDM_BUS_MODE", tdm_parameters.usMode, 1);

	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TDM_BUS_TO_ENABLE", tdm_parameters.usNoOfBus, 1);

	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "BUS_ID", tdm_parameters.astTdmBusParam[0].ucBusId, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TX_ORDER", tdm_parameters.astTdmBusParam[0].ucTxOrder, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "RX_POLARITY", tdm_parameters.astTdmBusParam[0].ucRxPolarity, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TX_POLARITY", tdm_parameters.astTdmBusParam[0].ucTxPolarity, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "RX_CLOCK_MODE", tdm_parameters.astTdmBusParam[0].ucRxClkMode, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TX_CLOCK_MODE", tdm_parameters.astTdmBusParam[0].ucTxClkMode, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "RX_CLOCK_EDGE", tdm_parameters.astTdmBusParam[0].ucRxClkEdge, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "TX_CLOCK_EDGE", tdm_parameters.astTdmBusParam[0].ucTxClkEdge, 1);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "FRAME_EDGE", tdm_parameters.astTdmBusParam[0].ucFrameEdge, 1);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "NUMBER_OF_TS", tdm_parameters.astTdmBusParam[0].usNumOfTS, 0x20);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "INVERTED_FRAME_SIGNAL",
		 tdm_parameters.astTdmBusParam[0].ucInvertedFrmSig, 1);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "BIT_ORDER_RCV", tdm_parameters.astTdmBusParam[0].ucBitOrderRcv, 1);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "NUM_TDM_CLK_CYCLES_DELAY_TX",
		 tdm_parameters.astTdmBusParam[0].usNumTdmClkCyclesDelayTx, 0);
	READ_INT(cfg_info, "DEVICE_TDM_CONFIG", "NUM_TDM_CLK_CYCLES_DELAY_RX",
		 tdm_parameters.astTdmBusParam[0].usNumTdmClkCyclesDelayRx, 0);

	cfg_clean(cfg_info);

	/* uses the same parameters for 4 buses */
	memcpy(&tdm_parameters.astTdmBusParam[1], &tdm_parameters.astTdmBusParam[0], sizeof(STdmBusParams));
	memcpy(&tdm_parameters.astTdmBusParam[2], &tdm_parameters.astTdmBusParam[0], sizeof(STdmBusParams));
	memcpy(&tdm_parameters.astTdmBusParam[3], &tdm_parameters.astTdmBusParam[0], sizeof(STdmBusParams));
	tdm_parameters.astTdmBusParam[1].ucBusId = 1;
	tdm_parameters.astTdmBusParam[2].ucBusId = 2;
	tdm_parameters.astTdmBusParam[3].ucBusId = 3;

	switch (tdm_parameters.usNoOfBus)
	{
	case 4:
		max_timeslots = tdm_parameters.astTdmBusParam[3].usNumOfTS;
	case 3:
		max_timeslots += tdm_parameters.astTdmBusParam[2].usNumOfTS;
	case 2:
		max_timeslots += tdm_parameters.astTdmBusParam[1].usNumOfTS;
	case 1:
		max_timeslots += tdm_parameters.astTdmBusParam[0].usNumOfTS;
		break;
	default:
		PDEBUG(SANITY_ERROR, "Error in number of TDM bus to enable");
		return -1;
	}

	/* FIXME: we should check the result */
	result = VAPI_SetTDMParams(device_id, &tdm_parameters, NULL);
	if (result == -0x4C)
		result = SUCCESS;
	return result;
}

/*=================================================================================*/
/*!
*	\brief This function initializes the network parameters for RTP traffic.
*	it calls :
*		- VAPI_SetEthMac
*		- VAPI_SetDeviceIPAddr
*	The MAC and IP addresses are comming from the DEVICE_NETWORK_CONFIG section of the configuration file.
*
*	\section Inputs-Outputs
*		\param config_file confiuration filename (-c option)
*		\return
*			SUCCESS \n
*			or status returned VAPI_SetEthMac or VAPI_SetDeviceIPAddr \n
*/
/*=================================================================================*/
int network_parameters_initialisation(const char *config_file, U8 ip_flag)
{
	int result;
	struct _CFG *cfg_info = NULL;
	SIpAddrInfo ip_info;

	if (config_file && *config_file)
		cfg_info = cfg_read(config_file, 0);

	if (!cfg_info)
	{
		PDEBUG(SANITY_ERROR, "Config read: Can't open config file \"%s\"", config_file);
		return -1;
	}

	PDEBUG(SANITY_INIT, "Init IPv4 device address");
	/* read the IPV4 address if  */
	READ_IP(cfg_info, "DEVICE_NETWORK_CONFIG", "DEVICE_IP_ADDRESS", device_ip_address, "192.168.1.2");
	print_ip_addr("DEVICE IPv4 = ", device_ip_address);

	PDEBUG(SANITY_INIT, "Init device MAC address");
	READ_MAC(cfg_info, "DEVICE_NETWORK_CONFIG", "DEVICE_MAC_ADDRESS", device_mac_addr, def_mac_addr);
	print_mac_addr("DEVICE MAC = ", device_mac_addr);

	PDEBUG(SANITY_INIT, "Init host MAC address");
	READ_MAC(cfg_info, "DEVICE_NETWORK_CONFIG", "HOST_MAC_ADDRESS", host_mac_addr, def_mac_addr);
	print_mac_addr("HOST MAC   = ", host_mac_addr);

	result = VAPI_SetEthMac(device_id, CMD_LEVEL_DEVICE, device_mac_addr, host_mac_addr, NULL);

	if (result != SUCCESS)
		PDEBUG(SANITY_ERROR, ">>>>>>>>>>> error VAPI_SetEthMac: %d, status %d", device_id, result);
	else if (ip_flag)
	{
		ip_info.ucNumOfSrcIpAddr = 1;
		ip_info.bIsMultipleMode = 0;
		ip_info.ucEncapsulation = 2;
		ip_info.ucChkSumVerification = 1;

		memcpy(ip_info.auiDevIPAddress, device_ip_address, sizeof(ip_info.auiDevIPAddress));

		result = VAPI_SetDeviceIPAddr(device_id, &ip_info, NULL);
		if (result != SUCCESS)
			PDEBUG(SANITY_ERROR, ">>>>>>>>>>> error VAPI_SetDeviceIPAddr: %d, status %d", device_id, result);
	}

	cfg_clean(cfg_info);

	return result;
}

/*=================================================================================*/
/*!
*	\brief This function set the IP/UDP header for the VoIP channel.
*	it calls VAPI_SetConnIpParams.
*
*	\section Inputs-Outputs
*		\param endpoint_id 
*		\param mode Sync/Async
*		\param endpoint_state used as request id
*		\return status returned by VAPI_SetConnIpParams
*/
/*=================================================================================*/
int set_ip_udp_parameters(int endpoint_id, int mode, int endpoint_state)
{
	int result;
	SRequest request;
	SIpParams ip_parameters;

	request.uiReqId = endpoint_state;

	if (mode == ASYNC)
		request.pfnIoCompCallback = &comcerto_response_handler;
	else
		request.pfnIoCompCallback = NULL;

	PDEBUG(SANITY_FUNCTION, "set ip parameters:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

	/* same ip address for source and destination */
	memcpy(&ip_parameters.uiDestIpAddr, device_ip_address, sizeof(ip_parameters.uiDestIpAddr));
	memcpy(&ip_parameters.uiSrcIpAddr, device_ip_address, sizeof(ip_parameters.uiSrcIpAddr));

	/* cross the UDP ports */
	ip_parameters.usSrcUdpPort = htons(endpoints[endpoint_id].rtp_port);
	ip_parameters.usDestUdpPort = htons(endpoints[endpoints[endpoint_id].peer_index].rtp_port);
	PDEBUG(SANITY_FUNCTION, "endpoints[%d].src rtp_port %d", endpoint_id, ip_parameters.usSrcUdpPort);
	PDEBUG(SANITY_FUNCTION, "endpoints[%d].dst rtp_port %d", endpoint_id, ip_parameters.usDestUdpPort);

	ip_parameters.ucIPServiceId = 0; /* RTP */

	result = VAPI_SetConnIpParams(endpoint_id, &ip_parameters, &request);

	if ((result != SUCCESS) && (result != VAPI_ERR_PENDING))
		PDEBUG(SANITY_ERROR, ">>>>>>>>>>> error VAPI_SetConnIpParams: %d, status %d", endpoint_id, result);

	return result;
}

/*=================================================================================*/
/*!
*	\brief This function starts the RTP traffic.
*	It calls VAPI_EnableConnection.
*
*	\section Inputs-Outputs
*		\param endpoint_id 
*		\param mode Sync/Async
*		\param endpoint_state used as request id
*		\return status returned by VAPI_EnableConnection
*/
int start_voip_endpoint(int endpoint_id, int mode, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;

	if (mode == ASYNC)
		request.pfnIoCompCallback = &comcerto_response_handler;
	else
		request.pfnIoCompCallback = NULL;

	PDEBUG(SANITY_FUNCTION, "Start connection on endpoint:%d (request id = 0x%04x)", endpoint_id, request.uiReqId);

	result = VAPI_EnableConnection(endpoint_id, &request);
	if ((result != SUCCESS) && (result != VAPI_ERR_PENDING))
		PDEBUG(SANITY_ERROR, ">>>>>>>>>>> error VAPI_EnableConnection: %d, status %d", endpoint_id, result);

	return result;
}

/*=================================================================================*/
/*!
*	\brief	This function creates a endpoint.
*	It calls VAPI_CreateConnection \n
*	The coding law is forced to ulaw, the timeslot used = the endpoint_id.
*
*	\section Inputs-Outputs
*		\param endpoint_id 
*		\param mode Sync/Async
*		\param endpoint_state used as request id
*		\return status returned by VAPI_CreateConnection
*/
int create_voip_endpoint(int endpoint_id, int mode, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;

	if (mode == ASYNC)
		request.pfnIoCompCallback = &comcerto_response_handler;
	else
		request.pfnIoCompCallback = NULL;

	PDEBUG(SANITY_FUNCTION, "create VoIP channel:%d type %d timeslot %d (request id = 0x%04x)", endpoint_id, eVOIP,
	       endpoint_id, request.uiReqId);

	if (endpoints[endpoint_id].timeslot > max_timeslots)
	{
		PDEBUG(SANITY_ERROR, ">>>>>>>>>>> Timeslot %d out of range", endpoints[endpoint_id].timeslot);
		return -1;
	}

	result = VAPI_CreateConnection(device_id, endpoint_id,	/* use the endpoint_id as connection_id */
				       eVOIP,	/* type 2 */
				       endpoints[endpoint_id].timeslot, U_LAW, &request, NULL);
	if ((result != SUCCESS) && (result != VAPI_ERR_PENDING))
		PDEBUG(SANITY_ERROR, ">>>>>>>>>>> error VAPI_CreateConnection: %d, status %d", endpoint_id, result);

	return result;
}

/*=================================================================================*/
/*!
*	\brief This function destroys a endpoint
*	It calls VAPI_DestroyConnection \n
*
*	\section Inputs-Outputs
*		\param endpoint_id 
*		\param mode Sync/Async
*		\param endpoint_state used as request id
*		\return status returned by VAPI_DestroyConnection
*/
int destroy_endpoint(int endpoint_id, int mode, int endpoint_state)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;

	if (mode == ASYNC)
		request.pfnIoCompCallback = &comcerto_response_handler;
	else
		request.pfnIoCompCallback = NULL;

	PDEBUG(SANITY_FUNCTION, "destroy VoIP channel:%d  (request id = 0x%04x)", endpoint_id, request.uiReqId);

	result = VAPI_DestroyConnection(endpoint_id, &request);

	if ((result != SUCCESS) && (result != VAPI_ERR_PENDING))
		PDEBUG(SANITY_ERROR, ">>>>>>>>>>> error VAPI_DestroyConnection: %d, status %d", endpoint_id, result);

	return result;
}

/*=================================================================================*/
/*!
*	\brief This function creates a particpant.
*	It calls VAPI_CreateTdmSideParticipant or VAPI_CreateIpSideParticipant\n
*	The coding law is forced to ulaw, the timeslot used = the endpoint_id (
*	(for VAPI_CreateTdmSideParticipant).
*
*	\section Inputs-Outputs
*		\param endpoint_id 
*		\param mode Sync/Async
*		\param endpoint_state used as request id
*		\param conference_id Conference to which the participant is created on.
*		\return status returned by VAPI_CreateTdmSideParticipant orVAPI_CreateIpSideParticipant 
*/
int create_participant(int endpoint_id, int mode, int endpoint_state, int conference_id, int participant_type)
{
	int result;
	SRequest request;

	request.uiReqId = endpoint_state;

	if (mode == ASYNC)
		request.pfnIoCompCallback = &comcerto_response_handler;
	else
		request.pfnIoCompCallback = NULL;

	if (endpoints[endpoint_id].timeslot > max_timeslots)
	{
		PDEBUG(SANITY_ERROR, ">>>>>>>>>> Timeslot %d out of range", endpoints[endpoint_id].timeslot);
		return -1;
	}

	if (participant_type == TDM_SIDE)
	{
		PDEBUG(SANITY_INFO, "create TDM Participant: %d conference %d timeslot %d (request id = 0x%04x)",
		       endpoint_id, conference_id, endpoints[endpoint_id].timeslot, request.uiReqId);

		result = VAPI_CreateTdmSideParticipant(conference_id,
						       endpoint_id,
						       endpoints[endpoint_id].timeslot, U_LAW, &request, NULL);
	}

	else
	{
		PDEBUG(SANITY_INFO, "create IP Participant: %d conference %d (request id = 0x%04x)", endpoint_id,
		       conference_id, request.uiReqId);
		result = VAPI_CreateIpSideParticipant(conference_id, endpoint_id, &request, NULL);

	}

	if ((result != SUCCESS) && (result != VAPI_ERR_PENDING))
		PDEBUG(SANITY_ERROR, ">>>>>>>>>>> error VAPI_CreateConnection: %d, status %d", endpoint_id, result);

	return result;
}

/*=================================================================================*/
/*!
*	\brief	This function displays firmware informations
*	Calling Comcerto API using VAPI_PassThru it displays : 
*		- Fimware version
*		- DSP version
*		- device type
*
*	\section Inputs-Outputs
*		NONE
*/
void display_firmware_version(void)
{
	VSTATUS result;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];
	int i;

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return;
	result = VAPI_SetMessage(
			message,
			CMD_CLASS_CONF_DEVICE,
			CMD_TYPE_QUERY,
			FC_SUPVSR_GET_ARM_CODE_VERSION,
			0);

	/* send the command, the response is stored in device_response*/
	if(result == SUCCESS)
	{
		result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);

		if(result == SUCCESS)
		{
			PDEBUG(SANITY_FUNCTION, "FC_GET_ARM_CODE_VERSION: response %u bytes: ", response_len);
			
			for (i = 0; i < response_len; i++)
				PDEBUG(SANITY_FUNCTION, "RESPONSE FIFO %d 0x%02x ", i, device_response[i]);
			
			PDEBUG(SANITY_INIT, "Firmware version :%s\n ", &device_response[8]);
		}
	}

 
	VAPI_InitMessage(message);

	result = VAPI_SetMessage(
			message,
			CMD_CLASS_CONF_DEVICE,
			CMD_TYPE_QUERY,
			FC_SUPVSR_GET_SPU_CODE_VERSION,
			0);

	/* send the command, the response is stored in device_response*/
	if(result == SUCCESS)
	{
		/*re-initialize the len to MAX (it has been changed by the previous command*/
		response_len = DEFAULT_FIFO_MAX_SIZE;
		result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);

		if(result == SUCCESS)
		{
			PDEBUG(SANITY_FUNCTION, "FC_GET_SPU_CODE_VERSION: response %u bytes: ", response_len);
			
			for (i = 0; i < response_len; i++)
				PDEBUG(SANITY_FUNCTION, "RESPONSE FIFO %d 0x%02x ", i, device_response[i]);

			PDEBUG(SANITY_INIT, "SPU version :%s\n ", &device_response[8]);
		}
	}

	VAPI_InitMessage(message);

	result = VAPI_SetMessage(
			message,
			CMD_CLASS_CONF_DEVICE,
			CMD_TYPE_QUERY,
			FC_SUPVSR_GET_DEVICE_TYPE,
			0);

	/* send the command, the response is stored in device_response*/
	if(result == SUCCESS)
	{
		/*re-initialize the len to MAX (it has been changed by the previous command*/
		response_len = DEFAULT_FIFO_MAX_SIZE;
		result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);

		if(result == SUCCESS)
		{
			PDEBUG(SANITY_FUNCTION, "FC_GET_DEVICE_TYPE: response %u bytes: ", response_len);
			
			for (i = 0; i < response_len; i++)
				PDEBUG(SANITY_FUNCTION, "RESPONSE FIFO %d 0x%02x ", i, device_response[i]);

			PDEBUG(SANITY_INIT, "Device type: 0x%02x\n ", device_response[8]);
		}
	}

	VAPI_InitMessage(message);

	result = VAPI_SetMessage(
			message,
			CMD_CLASS_CONF_DEVICE,
			CMD_TYPE_QUERY,
			FC_SUPVSR_GET_DEVICE_TYPE,
			0);

	/* send the command, the response is stored in device_response*/
	if(result == SUCCESS)
	{
		/*re-initialize the len to MAX (it has been changed by the previous command*/
		response_len = DEFAULT_FIFO_MAX_SIZE;
		result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);

		if(result == SUCCESS)
		{
			PDEBUG(SANITY_FUNCTION, "FC_GET_DEVICE_TYPE: response %u bytes: ", response_len);
			
			for (i = 0; i < response_len; i++)
				PDEBUG(SANITY_FUNCTION, "RESPONSE FIFO %d 0x%02x ", i, device_response[i]);

			PDEBUG(SANITY_INIT, "Device type: 0x%02x\n ", device_response[8]);
			device_model = device_response[8];
		}
	}

	VAPI_FreeMessage(message);
}

/*=================================================================================*/
/*!
*	\brief	This utility function sends a rtcp statistics query to a connection.
*
*	\section Inputs-Outputs
*		\param connection_id connection to which the query has to be sent
*		\return status returned by send_connection_message_wait_response 
*/
VSTATUS connection_query_rtcpstat(U16 connection_id)
{
	VSTATUS result = FAILURE;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	
	if (message != NULL)
	{
		result = VAPI_SetMessage(
				message,
				CMD_CLASS_CONF_CHANNEL,
				CMD_TYPE_QUERY,
				FC_VOIP_RTCPSTAT,
				0);

		/* send the command, the response is stored in device_response*/
		if(result == SUCCESS)
		{
			result = VAPI_SendConnectionMessage(connection_id, (SMsg *)message, NULL, device_response, &response_len);

			if(result != SUCCESS)
				PDEBUG(SANITY_ERROR, ">>>>>>> Issue in connection_query_rtcpstat config %d on connection %d", result,
						connection_id);
			else
				PDEBUG(SANITY_INFO, "connection_query_rtcpstat successfull on connection %d", connection_id);
		}
	}

	VAPI_FreeMessage(message);
	
	return result;
}

/*=================================================================================*/
/*!
*	\brief	This utility function sends a rtcp statistics query to a connection using the command type 0x05: 
*
*	\section Inputs-Outputs
*		\param connection_id connection to which the query has to be sent
*		\return status returned by send_connection_message_wait_response 
*/
VSTATUS connection_rtcp_statistics(U16 connection_id)
{
	VSTATUS result = FAILURE;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	
	if (message != NULL)
	{
		result = VAPI_SetMessage(
				message,
				CMD_CLASS_STAT_CHANNEL,
				CMD_TYPE_STAT_VOIP_RTCP,
				FC_RESERVED,
				0);

		/* send the command, the response is stored in device_response*/
		if(result == SUCCESS)
		{
			result = VAPI_SendConnectionMessage(connection_id, (SMsg *)message, NULL, device_response, &response_len);

			if(result != SUCCESS)
				PDEBUG(SANITY_ERROR, ">>>>>>>> Issue in connection_rtcp_statistics config %d on connection %d", result,
						connection_id);
			else
				PDEBUG(SANITY_INFO, "connection_rtcp_statistics successfull on connection %d", connection_id);
		}
	}

	VAPI_FreeMessage(message);
	
	return result;
}


int check_passthru(void)
{
	VSTATUS result;
	SMsg MyMsg;
	U8 OutBuff[MAX_FIFO_SIZE];
	U32 OutBuffLen = MAX_FIFO_SIZE;

/* expected status SUCCESS*/
U8 multi_query[] = {
	 /*GET_IP_HDR*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x90, 0x00, 0x00,
	/*VOPENA*/	0x08, 0x00, 0x01, 0x02, 0x00, 0x80, 0x00, 0x00, 
	/*VCEOPT*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x80, 0x00, 0x00, 
	/*ECHOCAN*/	0x08, 0x00, 0x01, 0x02, 0x06, 0x80, 0x00, 0x00, 
	/*DTMFOPT*/	0x08, 0x00, 0x01, 0x02, 0x03, 0x80, 0x00, 0x00, 
	/*PTMNG*/	0x08, 0x00, 0x01, 0x02, 0x18, 0x80, 0x00, 0x00, 
	/*JBOPT*/	0x08, 0x00, 0x01, 0x02, 0x90, 0x80, 0x00, 0x00
	};

/* expected status SUCCESS*/
U8 multi_mixed[] = {
	 /*GET_IP_HDR*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x90, 0x00, 0x00,
	/*VOPENA*/	0x08, 0x00, 0x01, 0x02, 0x00, 0x80, 0x00, 0x00, 
	/*VCEOPT*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x80, 0x00, 0x00, 
	/*ECHOCAN*/	0x08, 0x00, 0x01, 0x02, 0x06, 0x80, 0x00, 0x00, 
	/*DTMFOPT CMD*/	0x0a, 0x00, 0x00, 0x02, 0x03, 0x80, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 
	/*PTMNG*/	0x08, 0x00, 0x01, 0x02, 0x18, 0x80, 0x00, 0x00, 
	/*JBOPT*/	0x08, 0x00, 0x01, 0x02, 0x90, 0x80, 0x00, 0x00
	};

/* expected status VAPI_ERR_UNEXPECTED_MSP_RESPONSE*/
U8 multi_wrong_padding[] = {
	 /*GET_IP_HDR*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x90, 0x00, 0x00,
	/*VOPENA*/	0x08, 0x00, 0x01, 0x02, 0x00, 0x80, 0x00, 0x00, 
	/*VCEOPT*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x80, 0x00, 0x00, 
	/*ECHOCAN*/	0x08, 0x00, 0x01, 0x02, 0x06, 0x80, 0x00, 0x00, 
	/*DTMFOPT CMD*/	0x0a, 0x00, 0x00, 0x02, 0x03, 0x80, 0x00, 0x00, 0x03, 0x00,
	/*PTMNG*/	0x08, 0x00, 0x01, 0x02, 0x18, 0x80, 0x00, 0x00, 
	/*JBOPT*/	0x08, 0x00, 0x01, 0x02, 0x90, 0x80, 0x00, 0x00
	};

/* expected status VAPI_ERR_UNEXPECTED_MSP_RESPONSE*/
U8 multi_wrong_size[] = {
	 /*GET_IP_HDR*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x90, 0x00, 0x00,
	/*VOPENA*/	0x08, 0x00, 0x01, 0x02, 0x00, 0x80, 0x00, 0x00, 
	/*VCEOPT*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x80, 0x00, 0x00, 
	/*ECHOCAN*/	0x08, 0x00, 0x01, 0x02, 0x06, 0x80, 0x00, 0x00, 
	/*DTMFOPT CMD*/	0x0a, 0x00, 0x00, 0x02, 0x03, 0x80, 0x00, 0x00,
	/*PTMNG*/	0x08, 0x00, 0x01, 0x02, 0x18, 0x80, 0x00, 0x00, 
	/*JBOPT*/	0x08, 0x00, 0x01, 0x02, 0x90, 0x80, 0x00, 0x00
	};

/* expected status -3*/
U8 multi_single_nack[] = {
	 /*GET_IP_HDR*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x90, 0x00, 0x00,
	/*VCEOPT*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x80, 0x00, 0x00, 
	/*ECHOCAN*/	0x08, 0x00, 0x01, 0x02, 0x06, 0x80, 0x00, 0x00, 
	/*DTMFOPT CMD*/	0x08, 0x00, 0x01, 0x02, 0x03, 0x80, 0x00, 0x00, 
	/*PTMNG*/	0x08, 0x00, 0x01, 0x02, 0x18, 0x80, 0x00, 0x00, 
	/*JBOPT*/	0x08, 0x00, 0x01, 0x02, 0x90, 0x80, 0x00, 0x00,
	/*VOPDIS*/	0x0a, 0x00, 0x00, 0x02, 0x00, 0x96, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/*VOPENA*/	0x16, 0x00, 0x00, 0x02, 0x00, 0x80, 0x00, 0x00, 
			0x01, 0x00, 0x80, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x12, 0x34, 0x56, 0x78, 0x00, 0x00 /*nack 0x03-> CNF_ERROR_VOPENA_PT*/
	};

/* expected status -65453 */
U8 multi_double_nack[] = {
	 /*GET_IP_HDR*/	0x08, 0x00, 0x01, 0x02, 0x01, 0x90, 0x00, 0x00,
	/*VCEOPT*/	0x10, 0x00, 0x00, 0x02, 0x01, 0x80, 0x00, 0x00,
			0x39, 0x40, 0x00, 0x7c, 0x04, 0x00, 0x00, 0x00, /*nack 0x33-> CNF_ERROR_PARAM_INVALID_VALUE*/ 
	/*ECHOCAN*/	0x08, 0x00, 0x01, 0x02, 0x06, 0x80, 0x00, 0x00, 
	/*DTMFOPT CMD*/	0x08, 0x00, 0x01, 0x02, 0x03, 0x80, 0x00, 0x00, 
	/*PTMNG*/	0x08, 0x00, 0x01, 0x02, 0x18, 0x80, 0x00, 0x00, 
	/*JBOPT*/	0x08, 0x00, 0x01, 0x02, 0x90, 0x80, 0x00, 0x00,
	/*VOPDIS*/	0x0a, 0x00, 0x00, 0x02, 0x00, 0x96, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/*VOPENA*/	0x16, 0x00, 0x00, 0x02, 0x00, 0x80, 0x00, 0x00, 
			0x08, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x12, 0x34, 0x56, 0x78, 0x00, 0x00 /*nack 0xFFAD ERR_UNKNOWN_FUNCTION*/
	};

U8 complex_tone[] = {
	 /*CADENCE_ENGINE */
			0x18, 0x00, 0x00, 0x02, 0x6f, 0x80, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x40, 0x32, 0x00, 0x00, 0x10,
			0x02, 0x00 ,0xc8, 0x00, 0x2c, 0x01, 0x00, 0x20,
	/* PROGRAM_TONE_ENGINE */
			0x2e, 0x00, 0x00, 0x02, 0x6e, 0x80, 0x00, 0x00,
			0x02, 0x00, 0x6c, 0x07, 0xd3, 0x00, 0x00, 0x00,
			0x00, 0x00 ,0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00 ,0x00, 0x00, 0x78, 0x05, 0xfb, 0x00,
			0x00, 0x00 ,0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00 ,0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* VOPENA_TONEGEN */
			0x0a, 0x00, 0x00, 0x02, 0x00, 0x80, 0x00, 0x00,
			0x0d, 0x00 ,0x00, 0x00
	};

	result = VAPI_CreateConnection(device_id, 0,	/* use the endpoint_id as connection_id */
				       eVOIP,	/* type 2 */
				       0, U_LAW, NULL, NULL);

	MyMsg.sMailbox.usMsgId = 0x000B ;
	MyMsg.sMailbox.usParam1 = 0 ;   /* used internally can be removed also*/

	MyMsg.pusFifo = (U16 *)&multi_query;
	MyMsg.uiFifoSize = (sizeof(multi_query) + 3 ) & ~0x3;
	PDEBUG(SANITY_INFO, "VAPI_PassThru multi_query;\n" );
	result =  VAPI_PassThru(0, CMD_LEVEL_CONN, &MyMsg, NULL, OutBuff, &OutBuffLen);

	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  VAPI_PassThru multi_query");

	MyMsg.pusFifo = (U16 *)&multi_mixed;
	MyMsg.uiFifoSize = (sizeof(multi_mixed) + 3 ) & ~0x3;
	PDEBUG(SANITY_INFO, "VAPI_PassThru multi_mixed;\n" );
	result =  VAPI_PassThru(0, CMD_LEVEL_CONN, &MyMsg, NULL, OutBuff, &OutBuffLen);

	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>> check_passthru VAPI_PassThru multi_mixed");

	MyMsg.pusFifo = (U16 *)&multi_wrong_size;
	MyMsg.uiFifoSize = (sizeof(multi_wrong_size) + 3 ) & ~0x3;
	PDEBUG(SANITY_INFO, "VAPI_PassThru multi_wrong_size;\n" );
	result =  VAPI_PassThru(0, CMD_LEVEL_CONN, &MyMsg, NULL, OutBuff, &OutBuffLen);

	if (result != VAPI_ERR_INVALID_PARAM)
		ERROR_EXIT(">>>>>>>>>>> check_passthru VAPI_PassThru multi_wrong_size");

	MyMsg.pusFifo = (U16 *)&multi_wrong_padding;
	MyMsg.uiFifoSize = (sizeof(multi_wrong_padding) + 3 ) & ~0x3;
	PDEBUG(SANITY_INFO, "VAPI_PassThru multi_wrong_padding;\n" );
	result =  VAPI_PassThru(0, CMD_LEVEL_CONN, &MyMsg, NULL, OutBuff, &OutBuffLen);

	if (result != VAPI_ERR_INVALID_PARAM)
		ERROR_EXIT(">>>>>>>>>>> check_passthru VAPI_PassThru multi_wrong_padding");


	MyMsg.pusFifo = (U16 *)&multi_single_nack;
	MyMsg.uiFifoSize = (sizeof(multi_single_nack) + 3 ) & ~0x3;
	PDEBUG(SANITY_INFO, "VAPI_PassThru multi_single_nack;\n" );
	result =  VAPI_PassThru(0, CMD_LEVEL_CONN, &MyMsg, NULL, OutBuff, &OutBuffLen);

	if (result != -3)
		ERROR_EXIT(">>>>>>>>>>> check_passthru VAPI_PassThru multi_single_nack");

	MyMsg.pusFifo = (U16 *)&multi_double_nack;
	MyMsg.uiFifoSize = (sizeof(multi_double_nack) + 3 ) & ~0x3;
	PDEBUG(SANITY_INFO, "VAPI_PassThru multi_double_nack;\n" );
	result =  VAPI_PassThru(0, CMD_LEVEL_CONN, &MyMsg, NULL, OutBuff, &OutBuffLen);

	if (result != -65453)
		ERROR_EXIT(">>>>>>>>>>> check_passthru VAPI_PassThru multi_double_nack");

	result = VAPI_DestroyConnection(0, NULL);
	PDEBUG(SANITY_INFO, "VAPI_DestroyConnection\n" );
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>> check_passthru VAPI_DestroyConnection");

	result = VAPI_CreateConnection(device_id, 0,	/* use the endpoint_id as connection_id */
				       eComplexTone,	/* type 2 */
				       0, U_LAW, NULL, NULL);

	PDEBUG(SANITY_INFO, "VAPI_CreateConnection eComplexTone\n" );
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>> check_passthru VAPI_CreateConnection eComplexTone");

	MyMsg.pusFifo = (U16 *)&complex_tone;
	MyMsg.uiFifoSize = (sizeof(complex_tone) + 3 ) & ~0x3;
	PDEBUG(SANITY_INFO, "VAPI_PassThru complex_tone;\n" );
	result =  VAPI_PassThru(0, CMD_LEVEL_CONN, &MyMsg, NULL, OutBuff, &OutBuffLen);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>> check_passthru VAPI_PassThru complex_tone");

	result = VAPI_DestroyConnection(0, NULL);

	PDEBUG(SANITY_INFO, "VAPI_DestroyConnectioncomplex_tone \n" );
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>> check_passthru VAPI_DestroyConnection complex_tone");

error_exit:
	return result;
}

int check_cesopsn(void)
{
	VSTATUS result;

	PDEBUG(SANITY_INFO, "check_cesopsn\n" );

	result = VAPI_CreateConnection(device_id, 0, eCSEoPSN, 0, U_LAW, NULL, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>> check_cesopsn VAPI_CreateConnection master");

	result = VAPI_CreateConnection(device_id, 1, eCSEoPSN, 1, U_LAW, NULL, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_cesopsn VAPI_CreateConnection mapped 1");

	result = VAPI_CreateConnection(device_id, 2, eCSEoPSN, 2, U_LAW, NULL, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_cesopsn VAPI_CreateConnection mapped 2");

	result = VAPI_CreateConnection(device_id, 5, eCSEoPSN, 5, U_LAW, NULL, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_cesopsn VAPI_CreateConnection mapped 5");

	PDEBUG(SANITY_INFO, "VAPI_CESoPSN_Mapper eChannelMap  \n" );
	result = VAPI_CESoPSN_Mapper(device_id, NULL, eChannelMap, 0, 1, 4, 0, 1, 2, 5);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_cesopsn VAPI_CESoPSN_Mapper eChannelMap");

	PDEBUG(SANITY_INFO, "VAPI_CESoPSN_Mapper destroy mapped eChannelMap connections\n");
	result = VAPI_DestroyConnection(0, NULL);
	result |= VAPI_DestroyConnection(1, NULL);
	result |= VAPI_DestroyConnection(2, NULL);
	result |= VAPI_DestroyConnection(5, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_cesopsn VAPI_DestroyConnection");

	result = VAPI_CreateConnection(device_id, 0, eCSEoPSN, 0, U_LAW, NULL, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_cesopsn VAPI_CreateConnection master");

	PDEBUG(SANITY_INFO, "VAPI_CESoPSN_Mapper eTSMap  \n" );
	result = VAPI_CESoPSN_Mapper(device_id, NULL, eTSMap, 0, 1, 3, 1, 2, 5);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_cesopsn VAPI_CESoPSN_Mapper eTSMap");

	PDEBUG(SANITY_INFO, "VAPI_CESoPSN_Mapper unlink eTSMap  \n" );
	result = VAPI_CESoPSN_Mapper(device_id, NULL, eTSMap, 0, 0, 0);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_cesopsn VAPI_CESoPSN_Mapper eTSMap");

	PDEBUG(SANITY_INFO, "VAPI_CESoPSN_Mapper destroy mapped eTSMap connections\n");
	result = VAPI_DestroyConnection(0, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_cesopsn VAPI_DestroyConnection");

error_exit:
	return result;
}

int check_play_tone(void)
{
	VSTATUS status;
	U16 usToneId = eCUSTOM_2;
	SPlayToneOpt stPlayToneOpt;
	U16 uiLimitTimestamp = 10;
	U16 uiStartTimestamp = 10;

	/* quad tone parameters */
	U16 usFirstToneFreq = 697;
	U16 usFirstToneAmp = 100;
	U16 usSecondToneFreq = 1209;
	U16 usSecondToneAmp = 100;
	U16 usThirdToneFreq = 1500;
	U16 usThirdToneAmp = 100;
	U16 usFourthToneFreq = 1700;
	U16 usFourthToneAmp = 100;
	U16 usOnTime = 80;
	U16 usOffTime = 80;

	/*dual tone parameters*/
	U16 usPrmToneFreq = 697;
	U16 usPrmToneAmp = 100;
	U16 usSecToneFreq = 1209;
	U16 usSecToneAmp = 100;
	U16 usOnTimeDual = 80;
	U16 usOffTimeDual = 80;

	UT_MemSet(&stPlayToneOpt, 0, sizeof(SPlayToneOpt));
	stPlayToneOpt.usMixModeOn = True;
	stPlayToneOpt.usMixModeOff = True;

	status = VAPI_CreateConnection(device_id, 0, eVOIP, 0, U_LAW, NULL, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_play_tone VAPI_CreateConnection");

	/* quad tone */
	status = VAPI_SetTone(usToneId, eQuadTone, 0, 10,
			usFirstToneFreq, usFirstToneAmp,
			usSecondToneFreq, usSecondToneAmp,
			usThirdToneFreq, usThirdToneAmp,
			usFourthToneFreq, usFourthToneAmp,
			usOnTime, usOffTime);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_play_tone VAPI_SetTone");

	status = VAPI_PlayTone(0, usToneId, eDirToBoth, NULL, 0, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_play_tone VAPI_PlayTone");
	
	status = VAPI_StopTone(0, 0, eBoth, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_play_tone VAPI_StopTone");

	/* dual tone*/
	status = VAPI_SetTone(usToneId, eDualTone, 1, 24, 
			usPrmToneFreq, usPrmToneAmp,
			usSecToneFreq, usSecToneAmp,
			usOnTimeDual, usOffTimeDual,
			usPrmToneFreq, usPrmToneAmp,
			usSecToneFreq, usSecToneAmp,
			usOnTimeDual, usOffTimeDual,
			usPrmToneFreq, usPrmToneAmp,
			usSecToneFreq, usSecToneAmp,
			usOnTimeDual, usOffTimeDual,
			usPrmToneFreq, usPrmToneAmp,
			usSecToneFreq, usSecToneAmp,
			usOnTimeDual, usOffTimeDual);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_play_tone VAPI_SetTone");

	status = VAPI_PlayTone(0, usToneId, eDirToBoth, &stPlayToneOpt, 2, NULL,
				uiLimitTimestamp, uiStartTimestamp);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_play_tone VAPI_PlayTone");
	
	status = VAPI_StopTone(0, 0, eBoth, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_play_tone VAPI_StopTone");

	status = VAPI_DestroyConnection(0, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_play_tone VAPI_DestroyConnection");	

error_exit:
	return status;
}

int check_change_conn_type(void)
{
	VSTATUS status;

	U16 pusParams[3];
	U16 timeslot = 0;
	U16 new_timeslot = 1;
	U16 num_of_timeslots = 1;
	/* define conference parameters */
	SConfParams conference_parameters;
	SChangeConnInfo stChangeConnInfo;

	/* use conference transcoding */
	conference_parameters.bTranscoding = False;
	conference_parameters.bBlockDTMF = False;

	/* allocate VoIP connection with particular timeslot
	then change its type to FoIP with new timeslot + 1*/
	status = VAPI_AllocateConnection(device_id, 0, eVOIP, eNarrowBand, 1, &timeslot, NULL, NULL);
	if (status != SUCCESS)	
		ERROR_EXIT(">>>>>>>>>>>  check_change_conn_type VAPI_AllocateConnection");

	stChangeConnInfo.eConnType = eFOIP;
	stChangeConnInfo.eConnMode = eNarrowBand;
	stChangeConnInfo.ePartType = eNOTPART;

	if ((gtl_device_configuration->ucDevType == DEV_TYPE_M821XX)
		|| (gtl_device_configuration->ucDevType == DEV_TYPE_M83XXX))
		stChangeConnInfo.eChangeOpMode = eOpModeNone;
	else
		stChangeConnInfo.eChangeOpMode = eOpModeIP;

	pusParams[0] = num_of_timeslots;
	pusParams[1] = new_timeslot;

	status = VAPI_ChangeConnectionType(0, &stChangeConnInfo, 2, pusParams, NULL);
	if (status != SUCCESS)	
		ERROR_EXIT(">>>>>>>>>>>  check_change_conn_type VAPI_ChangeConnectionType");
	
	/* create a simple conference 1 on device 0 */
	status = VAPI_CreateConference(device_id, g_conference_id, &conference_parameters, False, NULL, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_change_conn_type VAPI_CreateConference");

	stChangeConnInfo.eConnType = 0;
	stChangeConnInfo.eConnMode = eNarrowBand;
	stChangeConnInfo.ePartType = eRSP;

	if ((gtl_device_configuration->ucDevType == DEV_TYPE_M821XX)
		|| (gtl_device_configuration->ucDevType == DEV_TYPE_M83XXX))
		stChangeConnInfo.eChangeOpMode = eOpModeNone;
	else
		stChangeConnInfo.eChangeOpMode = eOpModeIP;

	pusParams[0] = g_conference_id;
	pusParams[1] = num_of_timeslots;
	pusParams[2] = new_timeslot;

	/* change connection to part type*/
	status = VAPI_ChangeConnectionType(0, &stChangeConnInfo, 3, pusParams, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_change_conn_type VAPI_ChangeConnectionType");

	status = VAPI_DestroyParticipant(0, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_change_conn_type VAPI_DestroyParticipant");

	status = VAPI_DestroyConference(g_conference_id, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_change_conn_type VAPI_DestroyConference");

error_exit:
	return status;

}

int check_modify_conn(void)
{
	VSTATUS status = SUCCESS;
	U16 timeslot = 0;
	U16 connection_id = 0;
	SModifyConnectionInfo stModConInfo[2];

	status = VAPI_AllocateConnection(device_id, connection_id, eVOIP, eNarrowBand, 1, &timeslot, NULL, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_modify_conn VAPI_AllocateConnection");

	/*Set the delay max to 100*/
	stModConInfo[0].usFunctionCode = FC_VOIP_JBOPT;
	stModConInfo[0].usParam = 2;
	stModConInfo[0].ucfield_shift = 0;
	stModConInfo[0].ucfield_width = 16;
	stModConInfo[0].usValue = 100;

	/*Change  the EC tail length to 40ms */
	stModConInfo[1].usFunctionCode = FC_VOIP_ECHOCAN;
	stModConInfo[1].usParam = 1;
	stModConInfo[1].ucfield_shift = 0;
	stModConInfo[1].ucfield_width = 4;
	stModConInfo[1].usValue = 0x04;

	status = VAPI_ModifyConnection(connection_id, 2, stModConInfo, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_modify_conn VAPI_ModifyConnection");	

	status = VAPI_DestroyConnection(connection_id, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_modify_conn VAPI_DestroyConnection");

error_exit:
	return status;
}

int check_recover(void (*pComsertoIndHand)(EEventCode eEventCode, void *pvData))
{
	VSTATUS status = SUCCESS;
	int i;
return SUCCESS;
	
	for (i = 0; i < number_of_endpoints; i++)
	{
		PDEBUG(SANITY_INFO, "VAPI_CreateConnection(%d) SYNC;\n" ,i);
		status = VAPI_CreateConnection(device_id, i,	/* connection Id */
						eVOIP, i,	/* timeslot */
						U_LAW, NULL, NULL);

		if (status != SUCCESS)
			ERROR_EXIT(">>>>>>>>>>>  check_recover_conn VAPI_CreateConnection");	
	}

	status = VAPI_CloseDevice(device_id, ePURGE_ALL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_recover_conn VAPI_CloseDevice");

	status = VAPI_OpenDevice(device_id, pComsertoIndHand);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_recover_conn VAPI_OpenDevice");

	status = VAPI_InitDevice(device_id, VAPI_DEV_OPMODE_RECOVER, VAPI_DEV_PROF_DEFAULT, NULL, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_recover_conn VAPI_InitDevice");	

error_exit:
	return status;
}

int check_conf_transcoding(void)
{
	VSTATUS result;
	/* define conference parameters */
	SConfParams conference_parameters;

	/* use conference transcoding */
	conference_parameters.bTranscoding = True;
	conference_parameters.bBlockDTMF = False;
	
	/* create a simple conference 1 on device 0 */
	result = VAPI_CreateConference(device_id, g_conference_id, &conference_parameters, False, NULL, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_CreateConference");

	/* Create the tanscoding Participant 0  on conference 1*/
	result = VAPI_CreateTranscodingParticipant( g_conference_id, 0, NULL, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_CreateTranscodingParticipant");

	/* Create the tanscoding Participant 1  on conference 1*/
	result = VAPI_CreateTranscodingParticipant( g_conference_id, 1, NULL, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_CreateTranscodingParticipant");

	/*Set participant 1 in G723*/
	result = VAPI_SetCodecType (1, eG723_1_ID, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_SetCodecType");

	result = set_ip_udp_parameters(0, SYNC, 0);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding set_ip_udp_parameters");

	result = set_ip_udp_parameters(1, SYNC, 0);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding set_ip_udp_parameters");

	/*Enable both connections*/ 
	result = VAPI_EnableParticipant(0, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_EnableParticipant");

	result = VAPI_EnableParticipant(1, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_EnableParticipant");

	result = VAPI_DisableParticipant(0, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_DisableParticipant");

	result = VAPI_DisableParticipant(1, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_DisableParticipant");

	result = VAPI_DestroyParticipant(0, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_DestroyParticipant");

	result = VAPI_DestroyParticipant(1, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_DestroyParticipant");

	result = VAPI_DestroyConference(g_conference_id, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_conf_transcoding VAPI_DestroyConference");

error_exit:
	return result;

}
int check_cmd_class_type(void)
{
	VSTATUS result;
	void *message;
	
	/* allocate a message to build the command */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	/* reversed CMD class/type --> unknown CC*/
	result = VAPI_SetMessage(message, CMD_TYPE_CONF_CHANGE, CMD_CLASS_CONF_DEVICE, FC_LATENCY_LEVEL, 1, 0x0000);

	/* expect to get VAPI_ERR_ILLEGAL_COMMAND */
	if (result != VAPI_ERR_ILLEGAL_COMMAND)
		ERROR_EXIT(">>>>>>>>>>>  check_cmd_class_type VAPI_SetMessage VAPI_ERR_ILLEGAL_COMMAND");

	/* Wrong CT for CC */
	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_QUERY_RESP, FC_LATENCY_LEVEL, 1, 0x0000);

	/* expect to get VAPI_ERR_ILLEGAL_CT_FOR_CC */
	if (result != VAPI_ERR_ILLEGAL_CT_FOR_CC)
		ERROR_EXIT(">>>>>>>>>>>  check_cmd_class_type VAPI_SetMessage VAPI_ERR_ILLEGAL_CT_FOR_CC");

	/* received axpected errors so OK*/
	result = SUCCESS;

error_exit:
	VAPI_FreeMessage(message);
	return result;
}

int check_resource_manager(void)
{
	U16 M823x9_mix_mode_res [VAPI_NUM_RESOURCES] = {136,96,66,98,88,60,67,58,58,30,33,46,28,26,32,28,22,20,20,18,16,33,118,24,50,47,48,32,48,48,54,36};
	U16 M823x8_mix_mode_res [VAPI_NUM_RESOURCES] = {121,85,59,87,78,53,59,51,52,27,29,41,25,23,28,25,20,18,18,16,14,29,105,21,44,41,43,28,43,43,48,32};
	U16 M823x6_mix_mode_res [VAPI_NUM_RESOURCES] = {91,64,44,65,59,40,44,38,39,20,22,31,19,17,21,19,15,13,13,12,11,22,79,16,33,32,32,21,32,32,36,24};
	U16 M823x4_mix_mode_res [VAPI_NUM_RESOURCES] = {60,43,29,44,39,27,29,26,26,13,14,20,12,12,14,12,10,9,9,8,7,14,52,11,22,21,21,14,21,21,24,16};
	U16 M823x3_mix_mode_res [VAPI_NUM_RESOURCES] = {45,31,22,33,29,20,22,19,19,10,11,15,9,9,11,9,7,7,7,6,5,11,39,8,17,15,16,11,16,16,18,12};
	U16 M823x2_mix_mode_res [VAPI_NUM_RESOURCES] = {31,21,15,22,20,13,15,13,13,7,7,10,6,6,7,6,5,4,4,4,4,7,26,5,11,11,11,7,11,11,12,8};
	U16 M823x1_mix_mode_res [VAPI_NUM_RESOURCES] = {15,10,7,11,10,7,7,6,6,3,4,5,3,3,4,3,2,2,2,2,2,4,13,3,6,5,5,4,5,5,6,4};

	U16 M82910_mix_mode_res [VAPI_NUM_RESOURCES] = {544,286,168,304,258,156,269,232,216,94,115,176,80,80,128,112,88,80,72,72,64,108,170,88,144,173,136,80,152,144,200,192};

	U16 available_resources[VAPI_NUM_RESOURCES];
	U16 *DevResource = NULL;
	U16 DevResSize = 0;
	int i;
	U8 required_resources[3] = {eResG711_10, eResG729a_20, eResT38};
	U8 my_resource;
	VSTATUS result;

	result = VAPI_SetResourceManager (0, eResourceDisable, NULL);
	if (result == VAPI_ERR_RES_MGR_NOT_SUPPORTED)
		goto out;

	/* expect to get VAPI_ERR_RES_MGR_NOT_ENABLED */
	if (result != VAPI_ERR_RES_MGR_NOT_ENABLED)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager disable VAPI_ERR_RES_MGR_NOT_ENABLED");

	/*Query the current resources when not enabled */
        result  = VAPI_SetResourceManager (0, eResourceQuery, available_resources);
	/* expect to get VAPI_ERR_RES_MGR_NOT_ENABLED */
	if (result != VAPI_ERR_RES_MGR_NOT_ENABLED)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager query VAPI_ERR_RES_MGR_NOT_ENABLED");

	/*Query the current resources without buffer */
        result  = VAPI_SetResourceManager (0, eResourceQuery, NULL);
	/* expect to get VAPI_ERR_RES_MGR_NOT_ENABLED */
	if (result != VAPI_ERR_NULL_POINTER_PASSED)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager NULL Pointer VAPI_ERR_NULL_POINTER_PASSED");

	/* Enable the ressource manager */
        result  = VAPI_SetResourceManager(0, eResourceEnable, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager enable failure");

	switch(device_model)
	{
		case M82910:
			DevResource = M82910_mix_mode_res;
			DevResSize = sizeof(M82910_mix_mode_res);
			break;

		case  M82359:
		case  M82349:
			DevResource = M823x9_mix_mode_res;
			DevResSize = sizeof(M823x9_mix_mode_res);
			break;

		case  M82358:
		case  M82348:
		case  M82318:
		case  M82308:
			DevResource = M823x8_mix_mode_res;
			DevResSize = sizeof(M823x8_mix_mode_res);
			break;

		case  M82356:
		case  M82346:
			DevResource = M823x6_mix_mode_res;
			DevResSize = sizeof(M823x6_mix_mode_res);
			break;

		case  M82354:
		case  M82344:
		case  M82334:
		case  M82324:
			DevResource = M823x4_mix_mode_res;
			DevResSize = sizeof(M823x4_mix_mode_res);
			break;

		case  M82353:
		case  M82343:
			DevResource = M823x3_mix_mode_res;
			DevResSize = sizeof(M823x3_mix_mode_res);
			break;

		case  M82352:
		case  M82332:
		case  M82322:
			DevResource = M823x2_mix_mode_res;
			DevResSize = sizeof(M823x2_mix_mode_res);
			break;

		case  M82351:
		case  M82331:
		case  M82321:
		case  M82311:
			DevResource = M823x1_mix_mode_res;
			DevResSize = sizeof(M823x1_mix_mode_res);
			break;

		default:
			/*If the device model is not supported exit silently*/
			goto out;
	}

	/* Query the ressource manager */
        result  = VAPI_SetResourceManager(0, eResourceQuery, available_resources);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager enable failure");

	for (i = 0; i < DevResSize/2; i++)
	{
		if (DevResource[i] != available_resources [i])
		{
			result = -1;
			ERROR_EXIT(">>>>>>>>>>>  check_resource_manager resource value mismatch for index");
		}
	}
	/* Allocate resource with resource manager not enabled */
	result  = VAPI_SetResourceManager (0, eResourceDisable, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager eResourceDisable failure");

	result  = VAPI_SetConnectionResource(0, 0, 3, required_resources);
	if (result != VAPI_ERR_RES_MGR_NOT_ENABLED)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager VAPI_SetConnectionResource Resource Manager Disabled failure");

	/* Enable the ressource manager */
        result  = VAPI_SetResourceManager(0, eResourceEnable, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager enable failure");

	/* Allocate resource with wrong parameter */
	result  = VAPI_SetConnectionResource(0, 0, 3, NULL);
	if (result != VAPI_ERR_NULL_POINTER_PASSED)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager null pointer check failure");

	/* Allocate resource with invalid res ID */
	my_resource = 33;
	result  = VAPI_SetConnectionResource(0, 0, 1, &my_resource);
	if (result != VAPI_ERR_INVALID_RESOURCE)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager inalid ID check failure");

	result = VAPI_CreateConnection(0, 0, eVOIP, 0, U_LAW, NULL, NULL);
	if (result != VAPI_ERR_NO_RESOURCE_ALLOC)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager no res allocated failure");


	/* Create a connection when resource manager is enabled with resource for resource G711 20ms channel but try to create a T.38 channel*/
	my_resource = eResG711_20;
	result  = VAPI_SetConnectionResource(0, 0, 1, &my_resource);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager enable failure");

	result = VAPI_CreateConnection(0, 0, eFOIP, 0, U_LAW, NULL, NULL);
	if (result != VAPI_ERR_INVALID_RESOURCE)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager Wrong res allocated failure");

	/* Create a connection when resource manager is enabled with resource for resource G711 20ms channel */
	result = VAPI_CreateConnection(0, 0, eVOIP, 0, U_LAW, NULL, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager create connection failure");

	/* try to switch it in T.38 channel (Must not work)*/
	result = VAPI_SwitchToT38(0, NULL, NULL);
	if (result != VAPI_ERR_INVALID_RESOURCE)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager T38 wrong res allocated failure");

	/* try to switch it in G.729 channel (Must not work)*/
	result = VAPI_SetCodecType(0, eG729_A_ID, NULL);
	if (result != VAPI_ERR_INVALID_RESOURCE)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager set codec G729 failure");

	/* try to switch it in 10ms (Must not work)*/
	result = VAPI_SetPacketInterval (0, 10, NULL);
	if (result != VAPI_ERR_INVALID_RESOURCE)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager set 10ms failure");

	 my_resource = eResT38;
	/* Change resource to support T.38 channel */
	result = VAPI_SetConnectionResource(0, 0, 1, &my_resource);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager Wrong res allocated failure");

	/* try to switch it in 10ms (Must work)*/
	result = VAPI_SwitchToT38(0, NULL, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager T38 wrong res allocated failure");

	/* try to set lower reseource than the existing connection requires  (Must not work)*/
	my_resource = eResG711_10;	
	result  = VAPI_SetConnectionResource(0, 0, 1, &my_resource);
	if (result != VAPI_ERR_INVALID_RESOURCE)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager eResG711_10 wrong res allocated failure");

	/* try to set higher reseource than the existing connection requires (Must work)*/
	my_resource = eResG729_1_8K;	
	result  = VAPI_SetConnectionResource(0, 0, 1, &my_resource);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager eResG729_1_8K wrong res allocated failure");

	/* Try to disable the resource manager while a connection exists (Must not work) */
	result  = VAPI_SetResourceManager (0, eResourceDisable, NULL);
	if (result != VAPI_ERR_INVALID_PARAM)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager VAPI_SetResourceManager while connection exists failure");

	/* Try to free the resource while a connection exists (Must not work) */
	result  = VAPI_SetConnectionResource(0, 0, 0, NULL);
	if (result != VAPI_ERR_INVALID_CONNID)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager free ressource while connection exits failure");

	/* Destroy connection */
	result = VAPI_DestroyConnection(0, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager create connection failure");

	/* Try to free the resource while connection destroyed (Must work) */
	result  = VAPI_SetConnectionResource(0, 0, 0, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager free ressource while connection doesn't exit failure");

	PDEBUG(SANITY_INFO, "Check T38 resource %d \n", DevResource[eResT38]);
	my_resource = eResT38;
	for (i = 0; i < DevResource[eResT38] + 1; i++)
	{
		result  = VAPI_SetConnectionResource(0, i, 1, &my_resource);
		PDEBUG(SANITY_INFO, "For Conn %d status = %d \n", i, result);
	}

	/* For the latest attempt (eResT38) it must fails */
	if ((i == DevResource[eResT38] + 1) && (result == VAPI_ERR_NO_RESOURCE))
		PDEBUG(SANITY_INFO, "check max T38 resource %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager for max T38 resource wrong res allocated failure");

	/* Free all resources (Must work) */
	for (i = 0; i < DevResource[eResT38] - 1; i ++)
	{
		result  = VAPI_SetConnectionResource(0, i, 0, NULL);
		if (result != SUCCESS)
			ERROR_EXIT(">>>>>>>>>>>  check_resource_manager free ressource failure");
		
	}
	/* Disable the resource manager while no connection exists (Must work) */
	result  = VAPI_SetResourceManager (0, eResourceDisable, NULL);
	if (result != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_resource_manager VAPI_SetResourceManager failure");

out:
	result = SUCCESS;
error_exit:
	return result;

}

/*=================================================================================*/
/*!
*	\brief
*	This function sets the device for unlimited multiple source IP and check if 
*	more than 10 channel can be created with more than 10 different IP src addresses
*/
/*=================================================================================*/
VSTATUS check_multiple_ip(U16 device_id)
{
	VSTATUS result;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];
	U32 src_ip_address;
	U16 i;
	SIpParams ip_parameters;

	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;
	/*Disable ARP */
	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_ARP_SERVICE_CONFIG, 1, 0x0000);
	if(result != SUCCESS)
		goto err;

	result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);
	if(result != SUCCESS)
		goto err;
	PDEBUG(SANITY_INFO, "check_multiple_ip ARP disabled \n");

	VAPI_InitMessage(message);

	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_IP_SERVICE_CONFIG, 1, 0x0201);
	if(result != SUCCESS)
		goto err;

	/* send the command, the response is stored in device_response*/
	result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);
	if(result != SUCCESS)
		goto err;
	PDEBUG(SANITY_INFO, "check_multiple_ip IP Service configured \n");

	VAPI_InitMessage(message);

	/*IP_OPTION bit 4 can be set to enable unlimited */
	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_IP_OPTIONS, 1, 0x0010);
	if(result != SUCCESS)
		goto err;

	/* send the command, the response is stored in device_response*/
	result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);
	if(result != SUCCESS)
		goto err;
	PDEBUG(SANITY_INFO, "check_multiple_ip IP OPTION unlimited multi IP configured \n");

	memcpy(&src_ip_address, device_ip_address, sizeof(U32));
	src_ip_address = ntohl(src_ip_address);
	memcpy(&ip_parameters.uiDestIpAddr, device_ip_address, sizeof(U32));

	/* Create 15 connections with 15 different IP addresses*/
	for (i = 0; i < 15; i ++)
	{
		result  = VAPI_AllocateConnection(device_id, i, eVOIP, eNarrowBand, 1, &i, NULL, NULL); 
		if(result != SUCCESS)
			goto err;

		ip_parameters.uiSrcIpAddr = htonl(src_ip_address + i);
		ip_parameters.usSrcUdpPort = htons(30000 + (2 * i));
		ip_parameters.usDestUdpPort = htons(30000 + (2 * i));
		ip_parameters.ucIPServiceId = 0; /* RTP */

		result = VAPI_SetConnIpParams(i, &ip_parameters, NULL);
		if(result != SUCCESS)
			goto err;
		PDEBUG(SANITY_INFO, "check_multiple_ip con %d ip src = 0x%08x ip dst = 0x%08x\n", i, ntohl(ip_parameters.uiSrcIpAddr), ntohl(ip_parameters.uiDestIpAddr));
        	result = VAPI_SetConnectionState(i, eActive, NULL);

	}

	sleep(2);

	/* Destroy the 15 connections */
	for (i = 0; i < 15; i ++)
	{
		result  = VAPI_DestroyConnection(i, NULL);
		if(result != SUCCESS)
			goto err;
	}

err:
	VAPI_FreeMessage(message);
	return result;
}

/*=================================================================================*/
/*!
 *	\brief
 *	This function set the NTG (Network Timming Generator) of the M823xx device
 */
/*=================================================================================*/
VSTATUS set_ntg_config(U16 device_id)
{
	VSTATUS result;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	result = VAPI_SetMessage(message, CMD_CLASS_CONF_DEVICE, CMD_TYPE_CONF_CHANGE, FC_NTG_CONFIG, 2, UT_CPU2LE16(0x0001), UT_CPU2LE16(0x0006));
	if(result != SUCCESS)
		goto err;

	/* send the command, the response is stored in device_response*/
	result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);

err:
	VAPI_FreeMessage(message);
	return result;
}


/*=================================================================================*/
/*!
 *	\brief
 *	This function configures the DIAG redirection
 */
/*=================================================================================*/
int configure_diag_redirect(struct redirect_info *redirect)
{
	VSTATUS result = SUCCESS;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];
	struct _REDIRECT_CSM_UDP diag_cmd; /*defined in comcerto-ud-types.h */

	if(redirect == NULL)
		return -1;

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	memset(&diag_cmd, 0, sizeof(struct _REDIRECT_CSM_UDP));

	diag_cmd.param_4.bits.protocol = REDIRECT_CSM_PROTOCOL_OPENDIAGNOSTICS;
	memcpy(diag_cmd.dst_mac, redirect->dst_mac, 6);
	memcpy(diag_cmd.src_mac, redirect->src_mac, 6);

	if(redirect->type == FC_REDIRECT_CSM)
	{
		diag_cmd.packet_type = htons(REDIRECT_CSM_TYPE_DEFAULT); /* csme protocol*/
	
		/* build the command to set the diag redirection*/
		result = VAPI_SetMessageFromBuffer(message, 
						CMD_CLASS_OPEN_DIAG,
						CMD_TYPE_DIAG_CONFIG, 
						FC_REDIRECT_CSM,
						8,		/*8 words in the command*/
						(U16 *)&diag_cmd);
		if(result != SUCCESS)
			goto err;
	}
	else /* assume redirect over IP/UDP*/
	{
		diag_cmd.packet_type = htons(REDIRECT_CSM_UDP_PACKET_TYPE_DEFAULT); /* IP protocol*/
	
		diag_cmd.param_12.bits.ip_hl = REDIRECT_CSM_UDP_IP_HL_DEFAULT;
		diag_cmd.param_12.bits.ip_v = REDIRECT_CSM_UDP_IP_V_DEFAULT;
		diag_cmd.param_12.bits.ip_tos = REDIRECT_CSM_UDP_IP_TOS_DEFAULT;
	
		diag_cmd.ip_len = REDIRECT_CSM_UDP_IP_LEN_DEFAULT;
		diag_cmd.ip_id = REDIRECT_CSM_UDP_IP_ID_DEFAULT;
		diag_cmd.ip_fragment = REDIRECT_CSM_UDP_IP_FRAGMENT_DEFAULT;
		diag_cmd.param_16.word = htons(0x8011); /* UDP , TTL*/
	
		diag_cmd.ip_checksum = REDIRECT_CSM_UDP_IP_CHECKSUM_DEFAULT;
	
		diag_cmd.ip_src = redirect->src_ip;
		diag_cmd.ip_dst = redirect->dst_ip;
		diag_cmd.udp_sport = redirect->src_udp;
		diag_cmd.udp_dport = redirect->dst_udp;
	
		diag_cmd.udp_len = UT_CPU2LE16(REDIRECT_CSM_UDP_UDP_LEN_DEFAULT);
		diag_cmd.udp_checksum = UT_CPU2LE16(REDIRECT_CSM_UDP_UDP_CHECKSUM_DEFAULT);
	
		/* build the command to set the diag redirection*/
		result = VAPI_SetMessageFromBuffer(message, CMD_CLASS_OPEN_DIAG, CMD_TYPE_DIAG_CONFIG, 
			FC_REDIRECT_CSM_UDP, sizeof(struct _REDIRECT_CSM_UDP)/2, (U16 *)&diag_cmd);
	
		if(result != SUCCESS)
			goto err;
	}

	result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, device_response, &response_len);
	if(result != SUCCESS)
		goto err;

err:
	VAPI_FreeMessage(message);
	return result ;
}


/*=================================================================================*/
/*!
 *	\brief
 *	This function configures the DIAG function code
 */
/*=================================================================================*/
int set_diagnostics(unsigned int id,	/* if 0xFFFF device else connection*/
			unsigned char command_class, 
			unsigned char command_type, 
			unsigned short function_code, 
			unsigned char action) /* 1 = enable, 0 = disable */
{
	VSTATUS result = SUCCESS;
	void *message;
	U32 response_len = DEFAULT_FIFO_MAX_SIZE;
	U8 device_response [DEFAULT_FIFO_MAX_SIZE];

	/* allocate a message to query the current options */
	message = VAPI_AllocateMessage(DEFAULT_FIFO_MAX_SIZE);
	if (message == NULL)
		return -1;

	/* build the command to set the diag config to enable or disable */
	result = VAPI_SetMessage(message, command_class, command_type, function_code, 1, action);
	if(result != SUCCESS)
		goto err;

	if(id == 0xFFFF)
		result = VAPI_SendDeviceMessage(device_id, (SMsg *)message, NULL, 
		device_response, &response_len);
	else
		result = VAPI_SendConnectionMessage(id, (SMsg *)message, NULL, 
		device_response, &response_len);

err:
	VAPI_FreeMessage(message);
	return result;
}

/* compile it for Linux only for now*/
#if !defined(_VXWORKS_)

#define SDRAM_SIZE (1024 * 1024 *16) /*16M for SDRAM */
#define IRAM_SIZE (1024 * 8) /*8K for IRAM */
#define ERAM_SIZE (1024 * 1024 *1) /*1M for ERAM */

VSTATUS core_dump(DEVID device_id)
{
	int result;
	U8* core_buffer;
	U32 ulOffset = 0;		/* Offset in the specified memory to get dump*/
	U32 ulDataSize = 0;		/* Number of bytes read */
	Boolean fLastRead = 0;	
	int file_desc = -1;
	char coredump_filename[50];

	PDEBUG(SANITY_INIT, "VAPI is ready to proceed with the core dump\n");

	if (gtl_device_configuration->eDevMode == eMASTER)
	{
		PDEBUG(SANITY_INIT, "The device is in Master mode. Please use msp-control application to run a core dump\n");
		return 0;
	}

	core_buffer = (U8 *) UT_AllocMem(SDRAM_SIZE);    /* Buffer to hold data */
	if(core_buffer == NULL)
	{
		PDEBUG(SANITY_ERROR, "Can't allocate coredump buffer %d", device_id );
		return FAILURE;
	}

	/* Open coredump file sdram */
	strcpy(coredump_filename, "vapi_san_core");
	strcat(coredump_filename, "_sdram.bin");
	file_desc = open (coredump_filename, O_CREAT|O_WRONLY, 0644);
	if (file_desc < 0)
	{
		PDEBUG(SANITY_ERROR, "Can't open corefile %s", coredump_filename );
		return FAILURE;
	}
	
	PDEBUG(SANITY_INIT, "Starting device(%d) coredump procedure for SDRAM(idx0) (Ssync mode) to file %s",device_id, coredump_filename);

	result = VAPI_GetCoreDump(device_id, 0, core_buffer, SDRAM_SIZE, &ulDataSize, ulOffset, &fLastRead, NULL, NULL);

	PDEBUG(SANITY_INFO, "device %d dumped: Offset %d, mem %d\n",device_id, ulOffset, ulDataSize);

	if (write(file_desc, core_buffer, ulDataSize) != ulDataSize) {
		PDEBUG(SANITY_ERROR, "Failed to write corefile %s, errno: %d", coredump_filename, errno);
		close(file_desc);
		return FAILURE;
	}

	close(file_desc);
	result = SUCCESS;
	UT_MemSet(core_buffer, 0, SDRAM_SIZE);

	/* Open coredump file for iram*/
	strcpy(coredump_filename, "vapi_san_core");
	strcat(coredump_filename, "_iram.bin");
	file_desc = open (coredump_filename, O_CREAT|O_WRONLY, 0644);
	if (file_desc < 0)
	{
		PDEBUG(SANITY_ERROR, "Can't open corefile %s", coredump_filename );
		return FAILURE;
	}
	
	PDEBUG(SANITY_INIT, "Starting device(%d) coredump procedure for IRAM(idx1) (Sync mode) to file %s",device_id, coredump_filename);

	ulOffset = 0;
	fLastRead = 0;
	ulDataSize = 0;

	result = VAPI_GetCoreDump(device_id, 1, core_buffer, IRAM_SIZE, &ulDataSize, ulOffset, &fLastRead, NULL, NULL);

	if(result != SUCCESS)
	{
		PDEBUG(SANITY_ERROR, "Coredump device %d Failed\n",device_id);
		result = FAILURE;
		goto err;
	}
	else
	{
		PDEBUG(SANITY_INFO, "device(%d) dumped: Offset %d, mem %d\n",device_id, ulOffset, ulDataSize);
		if (write(file_desc, core_buffer, ulDataSize) != ulDataSize) {
			PDEBUG(SANITY_ERROR, "Failed to write corefile %s, errno: %d", coredump_filename, errno);
			close(file_desc);
			return FAILURE;
		}
		close(file_desc);
		result = SUCCESS;
		UT_MemSet(core_buffer, 0, SDRAM_SIZE);
	}

	/* Open coredump file for eram */
	strcpy(coredump_filename, "vapi_san_core");
	strcat(coredump_filename, "_eram.bin");
	file_desc = open (coredump_filename, O_CREAT|O_WRONLY, 0644);
	if (file_desc < 0)
	{
		PDEBUG(SANITY_ERROR, "Can't open corefile %s", coredump_filename );
		return FAILURE;
	}
	
	ulOffset = 0;
	fLastRead = 0;
	ulDataSize = 0;

	PDEBUG(SANITY_INIT, "Starting device(%d) coredump procedure for ERAM(idx2) (Sync mode) to file %s",device_id, coredump_filename);

	result = VAPI_GetCoreDump(device_id, 2, core_buffer, ERAM_SIZE, &ulDataSize, ulOffset, &fLastRead, NULL, NULL);

	if(result != SUCCESS)
	{
		PDEBUG(SANITY_ERROR, "Coredump device(%d) Failed\n",device_id);
		result = FAILURE;
		goto err;
	}
	else
	{
		PDEBUG(SANITY_INIT, "device(%d) dumped: Offset %d, mem %d\n",device_id, ulOffset, ulDataSize);
		if (write(file_desc, core_buffer, ulDataSize) != ulDataSize) {
			PDEBUG(SANITY_ERROR, "Failed to write corefile %s, errno: %d", coredump_filename, errno);
			close(file_desc);
			return FAILURE;
		}
		close(file_desc);
		result = SUCCESS;
		UT_MemSet(core_buffer, 0, SDRAM_SIZE);
	}
		
err:
	UT_FreeMem(core_buffer);
	close(file_desc);
	
	return result;
}
#endif

int check_allocate_participant(void)
{
	VSTATUS status;
	U16 timeslot1 = 0;
	CONNID connection1 = 1;
	CONNID connection2 = 2;
	CONNID connection3 = 3;
	CONNID connection4 = 4;
	U16 timeslots[4] = {2,3,4,5};

	/* define conference parameters */
	SConfParams conference_parameters;

	/* use conference transcoding */
	conference_parameters.bTranscoding = False;
	conference_parameters.bBlockDTMF = False;

	
	/* create a simple conference 1 on device 0 */
	status = VAPI_CreateConference(device_id, g_conference_id, &conference_parameters, False, NULL, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_allocate_participant VAPI_CreateConference");
	
	status = VAPI_AllocateParticipant(g_conference_id, connection1, eLSP,  1, &timeslot1, NULL, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>>  check_allocate_participant VAPI_AllocateConnection");

	status = VAPI_AllocateParticipant(g_conference_id, connection2, eRSP, 0, NULL, NULL, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>> check_allocate_participant VAPI_AllocateConnection");

	status = VAPI_AllocateParticipant(g_conference_id, connection3, eRSPWB, 0, NULL, NULL, NULL);
	if (status != SUCCESS)
		ERROR_EXIT(">>>>>>>>>>> check_allocate_participant VAPI_AllocateConnection");

	if ((gtl_device_configuration->ucDevType == DEV_TYPE_M821XX) ||
		(gtl_device_configuration->ucDevType == DEV_TYPE_M83XXX))
	{
		status = VAPI_AllocateParticipant(g_conference_id, connection4, eLSPWB, 4, timeslots, NULL, NULL);
		if (status != SUCCESS)
			ERROR_EXIT(">>>>>>>>>>> check_allocate_participant VAPI_AllocateConnection");

		status = VAPI_DestroyParticipant(connection4, NULL);
	}

	status = VAPI_DestroyParticipant(connection1, NULL);
	status = VAPI_DestroyParticipant(connection2, NULL);
	status = VAPI_DestroyParticipant(connection3, NULL);

	status = VAPI_DestroyConference(g_conference_id, NULL);

error_exit:
	return status;
}

