/**! 	\file vapi_san.c 
	\brief This file contains the main functions of the application
 */

/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

/*! \mainpage Introduction 

	<h2>This document describes vapi_san application which uses the VAPI library services</h2>.

	The intention of VAPI sanity application is to test as much as possible the VAPI API's functionality.\n
	Some of them are tested in Synchronous mode only, some others in Synchronous and Asynchronous mode.\n
	Through command line options the user can set :
		- the number of connections to create (options -e, -x or -v)
		- the number of participant to create (options -p)
		- the number of loops to run (options -l)
		- the test level (skip or not some of the operations listed below)  (options -t)
		- the indication flood test  (options -o)
		
	The application is divided in 5 main operations:
	- VAPI and device initialization,  the following API's are exercised:
		- VAPI_RegisterReset
		- VAPI_Init
		- VAPI_OpenDevice
		- VAPI_AssignBootMAC
		- VAPI_BootDevice or VAPI_BootDeviceFile depending on -f or -F option
		- VAPI_SetTDMParams
		- VAPI_InitDevice
		- VAPI_SetEthMac
		- VAPI_SetDeviceIPAddr
		- VAPI_PassThru (device level)
		- VAPI_CESoPSN_Mapper

	ALL THE APIs BELOW ARE TESTED ONLY IF THE TEST LEVEL IS ABOVE 0
	
	- Connection level API's test mainly in Sync mode, the following API's are exercised:
		- VAPI_CreateConnection
		- VAPI_PassThru (connection level)

		if (test level is above 1 (i.e -t2)
			- VAPI_StartCallerId

		- VAPI_SetConnIpParams
		- VAPI_EnableConnection

		if (test level is above 2 (i.e -t3 or -t4)
			- VAPI_TranscodingSession
			- VAPI_SetPacketInterval
			- VAPI_Loopback (Inter channel)
			- VAPI_Loopback (TDM Haipin)

		- VAPI_DestroyConnection
		
	- Connection level API's test in mixed Sync/Async mode, the following API's are exercised:
		- VAPI_CreateConnection or VAPI_AllocateConnection depending on -a option
		- VAPI_SetConnIpParams
		- VAPI_EnableConnection
		- VAPI_SetEthMac (connection level)
		- VAPI_SetPayloadType
		- VAPI_SetCodecType
		- VAPI_SetPacketInterval
		- VAPI_EchoCancellerReset
		- VAPI_SetRtpSsrc
		- VAPI_SendNteEvent
		- VAPI_SwitchToT38
		- VAPI_ConfigureT38Options
		- VAPI_DestroyConnection

	- Conference level API's test in mixed Sync/Async mode, the following API's are exercised:
		- VAPI_CreateConference
		- VAPI_CreateTdmSideParticipant
		- VAPI_CreateIpSideParticipant
		- VAPI_EnableParticipant
		- VAPI_SetPacketInterval
		- VAPI_MuteParticipant
		- VAPI_PutParticipantOnHold
		- VAPI_DestroyParticipant
		- VAPI_DestroyConference

	- VAPI and device shutdown,  the following API's are exercised:
		- VAPI_CloseDevice
		- VAPI_Close
		
	<b>Some notes about configuring this application:</b>\n
	The various MAC/IP addresses for control and RTP/UDP, the TDM parameters, 
	are defined in the vapi_san.conf file and need to be set accordingly 
	the network configuration.

	<b>Some notes about using this application under the Linux:</b>\n
	It requires VAPI to be built with this options :
		- make GTL=CSME ENDIANNESS=BIGENDIAN (if required)
		- make install

	The Makefile provided has to be edited to configured:
		- Set the native or cross compilation options

	<pre>
	Several options are available on the comand line:
	# ./vapi_san --help
	Usage: vapi_san [OPTION...]
	vapi_san - program to run vapi device init then create connections and
	participants.
	
	-a, --connection_create=CONNECTION_CREATE_API
				0=VAPI_CreateConnection, 1=VAPI_AllocateConnection
				(default 0)
	-c, --config_file=CONFIGURATION_FILE
				Name of the configuration file
	-d, --device_itf=DEVICE_ITF   Interface type to control the device CSME (0)
				or PCI (1) (default CSME)
	-e, --num_of_endpoints=NUMBER_OF_ENDPOINTS
				Number of endpoints to create (VoIP and FoIP)
				(default  2)
	-f, --firmware=FIRMWARE_FILE   firmware code filename (using
				VAPI_BootDevice)
	-F, --firmware=FIRMWARE_FILE   firmware code filename (using
				VAPI_BootDeviceFile)
	-g, --enable_diag =ENABLE_DIAG   enable diagnostics
	-l, --loops=NUMBER_OF_LOOPS   Number of loops (default 1)
	-n, --dual_echo_canceller =DUAL_ECHO_CANCELLER
				single/dual echo canceller
	-o, --ontime_connection=ONTIME_CONNECTION
				Connection on time duration (default 0)
	-p, --num_of_participants=NUMBER_OF_PARTICIPANTS
				Number of particiapnts to create (default 4)
	-t, --test_level=TEST_LEVEL   Test level (0 to 4) (default 4)
	-v, --num_of_voice_endpoints=NUMBER_OF_VOICE_ENDPOINTS
				Number of voice endpoints to create (VoIP only)
				(default  2)
	-x, --num_of_complex_endpoints=NUMBER_OF_COMPLEX_ENDPOINTS
				Number of complex endpoints to create (VoIP only)
				(default  2)
	-r, --recover_option=RECOVER_OPTION
				0=VAPI_DEV_OPMODE_DEFAULT
				1=VAPI_DEV_OPMODE_RECOVER
				(default 0)
	-?, --help                 Give this help list
	--usage                Give a short usage message
	-V, --version              Print program version
	
	Mandatory or optional arguments to long options are also mandatory or optional
	for any corresponding short options.
	
	Report bugs to philippe.vivarelli@mindspeed.com.

	</pre>
	\example Usage example :\n
	./vapi_san -d0 -f v5_09_cand6_28613.axf -c m826xx.conf -e20 -p8 -l10 \n
	This will:
	- start the application with the configuration comming from the m826xx.conf file,
	- load the v5_09_cand6_28613.axf, then
	- run 10 loops for 20 endpoints and 10 conference participants.

	<b>Some notes about the -e, -v, -x options:</b>\n
	These 3 options are exclusive, only 1 of the 3 must be entered.\n
	-e10 will create 10 endpoints (G711), then at some point switch the endpoints in eG726_32, then in T38
	The user must not enter a value bigger than the max T38 endpoints supported by the device

	-x10 will create 10 endpoints (G711), then at some point switch the endpoints in eG726_32, but not in T38
	The user must not enter a value bigger than the max complex endpoints supported by the device
 
	-v10 will create 10 endpoints (G711), and keep them in G711
	The user must not enter a value bigger than the max G711 endpoints supported by the device

	<b>Some notes about the -t option:</b>\n
	With this option the user can select the level of the test.
	- level 0: Only the device initialization is performed
	- level 1: Skip CID, Loopbacks and extra configuration test
	- level 2: Skip Loopbacks and extra configuration test
	- level 3: Skip extra configuration test
	- level 4: Complete

	<b>Some notes about the -o option:</b>\n
	With this option the user can choose to keep the connection on for n seconds.\n
	If n > 0 the codec of one of the endpoints involved in the connection is changed.
	Therefore a lot of PTCHNG indications are generated and VAPI should handle them (display indications counter)
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#if !defined(_VXWORKS_)
#include <vapi/msp.h>
#include <vapi/vapi_build_option.h>
#include <getopt.h>
#else
#include <msp.h>
#include <taskLib.h>
#include <vapi_build_option.h>
#endif

#include "readcfg.h"
#include "vapi_san_type.h"
#include "vapi_san_var.h"
#include "vapi_san_func.h"
#include "vxcompat.h"

#if !defined(VAPI_RELEASE) || (VAPI_RELEASE < VAPI_VERSION(2, 11, 0))
#error This version of vapi sanity is for VAPI 2.11.x or above
#endif

#define	PRINT_ERROR											\
	{												\
		PDEBUG(SANITY_ERROR, "Endpoint %d >>>>>>>>>>> error %d", endpoints[i].index, status);	\
		sprintf(error_string, "Endpoint %d >>>>>>>>>>> error %d", endpoints[i].index, status);	\
	}


/* Global structure to store the setup configuration. This structure is defined in gtl.h
 * it will be updated by the set_device_configuration getting the parameters from the configuration file
 */

SPCIUsrData gtl_pci_config;
SCSMEUsrData gtl_csme_config;

SDeviceHeader *gtl_device_configuration;

struct _ENDPOINT_DESC	endpoints[MAX_ENDPOINTS];
SFaxConfigOpts		g_fax_configuration;

/* Caller ID clip message*/
U8 clip_msg[24] = {
	0x80, 0x01, 0x08, 0x31, 0x31, 0x30, 0x38, 0x32,
	0x30, 0x30, 0x30, 0x02, 0x07, 0x39, 0x34, 0x34,
	0x32, 0x38, 0x32, 0x30, 0x08, 0x01, 0x4F
};

/* Caller ID structure */
CidInfo clid_message = {
	.MsgLen = sizeof(clip_msg),
	.pucMsgStr = clip_msg,
	.bIsOnHook = False,
	.eCountryCode = eUSA
};

SCallerIdInfo CallerIdData = {
	.MsgLen = sizeof(clip_msg),
	.pucMsgStr = clip_msg,
	.bIsOnHook = True,	/*Generate OnHook Caller ID*/
	.ucModulType = 1	/* V23 */
	};

SCidDetCtrl CidDetectionCtrl = {
	.usCidEnable = 1,	/* enable */
	.usHookStatus = 2	/* both */
	};

/* define and set global variables default values */

U16	device_id = -1;
int	device_itf = 0;	/* default control itf csme*/
U8	device_mac_addr[6];
U8	host_mac_addr[6];
U8	def_mac_addr[6];
CONFID	g_conference_id;
int	number_of_participants = 4;
int	number_of_endpoints = 2;
int	number_of_loops = 1;
U16	max_timeslots;
U8	boot_option_file;
U8	skip_t38 = 0;
U8	skip_complex = 0;
U8	device_ip_address[4];
U8	test_level = TEST_LEVEL_ALL;
int	ontime_connection = 0;
int	dual_echo_canceller = 0;
static char config_file[MAX_STRING];
static char firmware[MAX_STRING];
U8	alert_received;
int	diag_function_code = 0;
int	connection_create = 0; /*Default is VAPI_CreateConnection*/
int	debug_level = 1;
int	recover_option = 0;
int	loop_counter;

/*array to print error string to be check at exit*/
char error_string[MAX_STRING];

#if !defined(_VXWORKS_)
static void show_help(void)
{
	printf(
		"vapi sanity - program to run vapi device init then create connections and participants\n"
		"Usage: %s [OPTION]...\n"
		"  -e, --num_of_endpoints=NUMBER_OF_ENDPOINTS\n"
		"			Number of endpoints to create (VoIP and FoIP) (default %d)\n\n"
		"  -v, --num_of_voice_endpoints=NUMBER_OF_VOICE_ENDPOINTS\n"
		"			Number of voice endpoints to create (VoIP only) (default  %d)\n\n"
		"  -x, --num_of_complex_endpoints=NUMBER_OF_COMPLEX_ENDPOINTS\n"
		"			Number of complex endpoints to create (VoIP only) (default  %d)\n\n"
		"  -p, --num_of_participants=NUMBER_OF_PARTICIPANTS\n"
		"			Number of particiapnts to create (default %d)\n\n"
		"  -d, --device_itf=DEVICE_ITF\n"
		"			Interface type to control the device CSME (0) or PCI (1) (default %d\n\n"
		"  -a, --connection_create=CONNECTION_CREATE_API\n"
		"			0=VAPI_CreateConnection, 1=VAPI_AllocateConnection (default %d)\n\n"
		"  -c, --config_file=CONFIGURATION_FILE\n"
		"			Name of the configuration file\n\n"
		"  -f, --firmware=FIRMWARE_FILE\n"
		"			Firmware code filename (using VAPI_BootDevice)\n\n"
		"  -F, --firmware=FIRMWARE_FILE\n"
		"			Firmware code filename (using VAPI_BootDeviceFile)\n\n"
		"  -l, --loops=NUMBER_OF_LOOPS\n"
		"			Number of loops (default %d)\n\n"
		"  -t, --test_level=TEST_LEVEL\n"
		"			Test level (0 to 4) (default %d)\n\n"
		"  -o, --ontime_connection=ONTIME_CONNECTION\n"
		"			Connection on time duration (default %d)\n\n"
		"  -n, --dual_echo_canceller=DUAL_ECHO_CANCELLER	single/dual echo canceller\n\n"
		"  -g, --enable_diag=ENABLE_DIAG	enable diagnostics\n\n"
		"  -m, --debug_level=DEBUG_LEVEL\n"
		"			Level of debug (0 error only, 1 +init, 2 +info, 3 +function (default %d)\n\n"
		"  -r, --recover_option=RECOVER_OPTION\n"
		"			0=VAPI_DEV_OPMODE_DEFAULT, 1=VAPI_DEV_OPMODE_RECOVER (default %d)\n\n"
		"      --help		Show this help and exit\n"
		"      --version	Show version and exit\n",
		"vapi_san",
		number_of_endpoints,
		number_of_endpoints,
		number_of_endpoints,
		number_of_participants,
		device_itf,
		connection_create,
		number_of_loops,
		test_level,
		ontime_connection,
		debug_level,
		recover_option
	);
}

static void show_version(void)
{
	printf("vapi sanity test application, %s\n", "vapi-san");
}

static struct option const long_options[] =
{
	{ "num_of_endpoints",		required_argument, NULL, 'e' },
	{ "num_of_voice_endpoints",	required_argument, NULL, 'v' },
	{ "num_of_complex_endpoints",	required_argument, NULL, 'x' },
	{ "num_of_participants",	required_argument, NULL, 'p' },
	{ "device_itf",			required_argument, NULL, 'd' },
	{ "connection_create",		required_argument, NULL, 'a' },
	{ "config_file",		required_argument, NULL, 'c' },
	{ "firmware",			required_argument, NULL, 'f' },
	{ "firmware",			required_argument, NULL, 'F' },
	{ "loops",			required_argument, NULL, 'l' },
	{ "test_level",			required_argument, NULL, 't' },
	{ "ontime_connection",		required_argument, NULL, 'o' },
	{ "dual_echo_canceller",	required_argument, NULL, 'n' },
	{ "enable_diag",		required_argument, NULL, 'd' },
	{ "debug_level",		required_argument, NULL, 'm' },
	{ "recover_option",		required_argument, NULL, 'r' },

	{ "help",			no_argument, NULL, -2 },
	{ "version",			no_argument, NULL, -3 },

	{ NULL, 0, NULL, 0},
};

#else
const char *argp_program_version = "vapi-san";
const char *argp_program_bug_address = "alexander.perekhodko@mindspeed.com";
const char doc[] = "vapi_san - program to run vapi device init then create connections and participants.";

const struct argp_option options[] = {
	{"num_of_endpoints", 'e', "NUMBER_OF_ENDPOINTS", 0, "Number of endpoints to create (VoIP and FoIP) (default  2)"},
	{"num_of_voice_endpoints", 'v', "NUMBER_OF_VOICE_ENDPOINTS", 0, "Number of voice endpoints to create (VoIP only) (default  2)"},
	{"num_of_complex_endpoints", 'x', "NUMBER_OF_COMPLEX_ENDPOINTS", 0, "Number of complex endpoints to create (VoIP only) (default  2)"},
	{"num_of_participants", 'p', "NUMBER_OF_PARTICIPANTS", 0, "Number of particiapnts to create (default 4)"},
	{"device_itf", 'd', "DEVICE_ITF", 0, "Interface type to control the device CSME (0) or PCI (1) (default CSME)"},
	{"connection_create", 'a', "CONNECTION_CREATE_API", 0, "0=VAPI_CreateConnection, 1=VAPI_AllocateConnection (default 0)"},
	{"config_file", 'c', "CONFIGURATION_FILE", 0, "Name of the configuration file"},
	{"firmware", 'f', "FIRMWARE_FILE", 0, "firmware code filename (using VAPI_BootDevice)"},
	{"firmware", 'F', "FIRMWARE_FILE", 0, "firmware code filename (using VAPI_BootDeviceFile)"},
	{"loops", 'l', "NUMBER_OF_LOOPS", 0, "Number of loops (default 1)"},
	{"test_level", 't', "TEST_LEVEL", 0, "Test level (0 to 4) (default 4)"},
	{"ontime_connection", 'o', "ONTIME_CONNECTION", 0, "Connection on time duration (default 0)"},
	{"dual_echo_canceller ", 'n', "DUAL_ECHO_CANCELLER", 0, "single/dual echo canceller"},
	{"enable_diag", 'g', "ENABLE_DIAG", 0, "enable diagnostics"},
	{"debug_level", 'm', "DEBUG_LEVEL", 0, "level of debug (0 error only, 1 +init, 2 +info, 3 +function (default 1)"},
	{"recover_option", 'r', "RECOVER_OPTION", 0, "recover option for VAPI_InitDevice (default 0)"},
	{NULL}
};
#endif

/*=================================================================================*/
/*!	\brief Application shutdown and exit
 *	\section Description
 *	This function proceed with the applcation shutdown, mening closing the device and VAPI.\n
 *	VAPI_CloseDevice() and VAPI_Close().
 *
 *	\section Inputs-Outputs
 *		None
 */	
/*=================================================================================*/
void main_exit(VSTATUS result)
{
	VSTATUS status = result;

	if (result == SUCCESS)
		printf( "TEST LEVEL %d PASSED: exit after %d loops on %d endpoints %d participants\n", 
				test_level, loop_counter, number_of_endpoints, number_of_participants);
	else
		printf( "TEST LEVEL %d FAILED: exit after %d loops on %d endpoints %d participants\n", 
				test_level, loop_counter, number_of_endpoints, number_of_participants);

	PDEBUG(SANITY_INFO, "VAPI_CloseDevice();\n");
	status |= VAPI_CloseDevice(device_id, ePURGE);

	if (status != SUCCESS)
		PDEBUG(SANITY_ERROR, ">>>>>>>>>>> error %d", status);

	PDEBUG(SANITY_INFO, "VAPI_Close();\n");
	status |= VAPI_Close();

	if (status != SUCCESS)
		PDEBUG(SANITY_ERROR, ">>>>>>>>>>> error %d", status);

	PDEBUG(DEBUG_INIT, "Exit from application(%d)", status);
	exit(status);
}

/*=================================================================================*/
/*!	\brief Destroy and endpoint on Onhook event.
 *	\section Description
 *	This function handle the onhook event in several states the endpoints.\n
 *	It sets the endpoint state to ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK.\n
 *	It sets the endpoint event to ENDPOINT_EVENT_NONE.\n
 *	Then destroy the endpoint (VAPI_DtestroyConnection).\n
 *
 *	\section Inputs-Outputs
 * 	\param endpoint_id
 */
 /*=================================================================================*/

void onhook_handler(int endpoint_id)
{
	/* switch to the state WAIT_DESTROY_CONNECTION_ACK to wait the result of destroy_conenction */
	set_endpoint_state(ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK, endpoint_id);
	set_endpoint_event(ENDPOINT_EVENT_NONE, endpoint_id);
	destroy_endpoint(endpoint_id, ASYNC, ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK);
}

/* implement the device reset handler */
VSTATUS comcerto_reset_handler(void *reset_data)
{
	/* platform-specific reset code may be added here */
	return SUCCESS;
}

/*=================================================================================*/
/*!	\brief Callback function registered when a VAPI API is called in Async Mode
 *	\section Description
 *	
 *	This function handles responses comming from the Comcerto device.\n
 *	This function is registered as call back when a function is called in asynchronous mode.\n
 *	
 *	\section Inputs-Outputs
 *		See VAPI documentation
 *
 *	\section Execution
 *	This function is called from a thread created by VAPI.\n
 *	When it is called the request_id is set to the current state of endpoint.\n
 *	This function set an event accordingly the state retrieved thru the request_id.\n
 */
/*=================================================================================*/
void comcerto_response_handler(U32 connection_id,
			       U8 command_level, S32 result, void *response_data, U32 data_length, U32 request_id)
{
	int i = connection_id;
	int status = result;
	PDEBUG(SANITY_INFO, "response received on endpoint id %d, request id = 0x%04x", connection_id, request_id);

	if (result != SUCCESS)
		PRINT_ERROR;

	switch (request_id)
	{
	case REQUEST_ID_DESTROY_LOOPBACK:
		set_endpoint_state(ENDPOINT_STATE_ON_HOOK, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK:
	case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK:
		PDEBUG(SANITY_FUNCTION, "VoIP Endpoint %d created,", connection_id);
		/* The VoIP Endpoint has been succesfully created, post the event */
		set_endpoint_event(ENDPOINT_EVENT_VOIP_CHANNEL_CREATED, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_DESTROY_CONNECTION_ACK:
		PDEBUG(SANITY_FUNCTION, "VoIP Endpoint %d destroyed,", connection_id);
		/* The VoIP Endpoint has been succesfully created, post the event */
		set_endpoint_event(ENDPOINT_EVENT_VOIP_CHANNEL_DESTROYED, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK:
		PDEBUG(SANITY_FUNCTION, "VoIP Endpoint %d IP parameters set,", connection_id);
		/* The VoIP Endpoint IP parameters have been succesfully set, post the event */
		set_endpoint_event(ENDPOINT_EVENT_IP_PARAMETERS_SET, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK:
		PDEBUG(SANITY_FUNCTION, "VoIP Endpoint %d IP parameters set,", connection_id);
		/* The VoIP Endpoint IP parameters have been succesfully set, post the event */
		set_endpoint_event(ENDPOINT_EVENT_IP_PARAMETERS_SET, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK:
		PDEBUG(SANITY_FUNCTION, "VoIP Endpoint %d enabled,", connection_id);
		/* The VoIP Endpoint has been succesfully enabled, post the event */
		set_endpoint_event(ENDPOINT_EVENT_CONNECTION_ENABLED, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK:
		PDEBUG(SANITY_FUNCTION, "VoIP Endpoint %d enabled,", connection_id);
		/* The VoIP Endpoint has been succesfully enabled, post the event */
		set_endpoint_event(ENDPOINT_EVENT_CONNECTION_ENABLED, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_T38_OPTIONS_ACK:
		PDEBUG(SANITY_FUNCTION, "Endpoint %d T38 options configured,", connection_id);
		/* The VoIP Endpoint has been succesfully enabled, post the event */
		set_endpoint_event(ENDPOINT_EVENT_T38_OPTIONS_ACK, connection_id);
		break;

	case ENDPOINT_STATE_FINISH:
		PDEBUG(SANITY_FUNCTION, "Endpoint %d finished", connection_id);
		set_endpoint_event(ENDPOINT_EVENT_FINISH, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_CREATE_TDM_PARTICIPANT_ACK:
		PDEBUG(SANITY_FUNCTION, "TDM participant %d created", connection_id);
		set_endpoint_event(ENDPOINT_EVENT_TDM_PARTICIPANT_CREATED, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_CREATE_IP_PARTICIPANT_ACK:
		PDEBUG(SANITY_FUNCTION, "IP participant %d created", connection_id);
		set_endpoint_event(ENDPOINT_EVENT_IP_PARTICIPANT_CREATED, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_IP_PARAMETERS_IP_PARTICIPANT_ACK:
		PDEBUG(SANITY_FUNCTION, "IP participant %d IP params configured", connection_id);
		set_endpoint_event(ENDPOINT_EVENT_IP_PARAM_IP_PARTICIPANT_CONFIGURED, connection_id);
		break;

	case ENDPOINT_STATE_WAIT_IP_PARTICIPANT_ENABLE_ACK:
		PDEBUG(SANITY_FUNCTION, "IP participant %d enabled", connection_id);
		set_endpoint_event(ENDPOINT_EVENT_IP_PARTICIPANT_ENABLED, connection_id);
		break;

	case ENDPOINT_STATE_CONFERENCE_FINISH:
		PDEBUG(SANITY_FUNCTION, "Participant %d destroyed", connection_id);
		set_endpoint_event(ENDPOINT_EVENT_FINISH, connection_id);
		break;
	}
}

/*=================================================================================*/
/*!	\brief Indication handler reistered through VAPI_OpenDevice()
 *	\section Description
 *	
 *	This function handles indications from the Comcerto device.\n
 *	This function is registered as call back for event.\n
 *	
 *	\section Inputs-Outputs
 *		See VAPI documentation.\n
 *
 *	\section Execution
 *	This function is called by VAPI every time a event occurs on the Comcerto device.\n
 *	It just prints:.\n
 *		- the tone number if the event reported is  eVAPI_TONE_DETECT_EVENT. 
 *		- the new SSRC  if the event reported is  eVAPI_SSRC_CHANGE_EVENT. 
 *	For the eVAPI_CALLER_ID_CMPLT_EVENT it sets the endpoint to the
 *	ENDPOINT_STATE_CID_CMPLT state to verify the indication  of the Caller Id competed is received.\n
 *	For the eVAPI_PT_CHANGE_EVENT it increments a counter to be able to verify the PTCHNG indications have been  received.
 *	This is used when the option -o is entered.\n
 *	In a real system it should set some endpoint events such as CED detected to proceed a T38 switch. 
*/
/*=================================================================================*/
void comcerto_indication_handler(EEventCode eEventCode, void *pvData)
{
	SToneDetectEventParams *tone_detected;
	SSsrcChangeEventParams *ssrc_changed;
	SUndefinedEventParams *this_event;
	SCallerIdCmpltEventParams *this_cid_complete;
	SPtChangeEventParams *this_ptchng_ind;
	SAlrtIndParams *this_alert;
	int i;

	/* FIXME: The parser below need to be improved */

	switch (eEventCode)
	{
	case eVAPI_TONE_DETECT_EVENT:
		tone_detected = (SToneDetectEventParams *) pvData;

		PDEBUG(SANITY_FUNCTION, "Tone %d detected on Connection ID %d",
		       tone_detected->usDetectedTone, tone_detected->ConId);

		break;

	case eVAPI_SSRC_CHANGE_EVENT:
		ssrc_changed = (SSsrcChangeEventParams *) pvData;
		PDEBUG(SANITY_FUNCTION, "SSRC changed 0x%08x on Connection ID %d",
		       ssrc_changed->uNewSSRC, ssrc_changed->ConId);

		break;

	case eVAPI_CALLER_ID_CMPLT_EVENT:
		this_cid_complete = (SCallerIdCmpltEventParams *) pvData;
		PDEBUG(SANITY_FUNCTION, "event eVAPI_CALLER_ID_CMPLT_EVENT received on Connection ID %d",
			this_cid_complete->ConId);
		set_endpoint_state(ENDPOINT_STATE_CID_CMPLT, this_cid_complete->ConId);

		break;

	case eVAPI_PT_CHANGE_EVENT:
		this_ptchng_ind = (SPtChangeEventParams *) pvData;

		if(endpoints[this_ptchng_ind->ConId].ptchng_count == 0)
			PDEBUG(SANITY_FUNCTION, "event eVAPI_PT_CHANGE_EVENT received on Connection ID %d", this_ptchng_ind->ConId);

		endpoints[this_ptchng_ind->ConId].ptchng_count++;

		break;

	case eVAPI_ALERT_IND:
		this_alert = (SAlrtIndParams *) pvData;
		alert_received = True;

		PDEBUG(SANITY_ERROR, ">>>>>>>>>>>>>>>> Alert received <<<<<<<<<<<<<<<<<<<\n");

		{
			PDEBUG(1, "ConID:     %d", this_alert->ConId);
			PDEBUG(1, "DevID:     %d", this_alert->DevId);
			PDEBUG(1, "Type:      %d", this_alert->usAlrtType);
			PDEBUG(1, "CPU:       %d", this_alert->ucCPUNo);
			PDEBUG(1, "Unique ID: %d", this_alert->usUniqueId);
			
			for (i = 0; i< this_alert->ucAlertRawDataSize; i ++)
				PDEBUG(SANITY_INFO, " 0x%02x ", this_alert->aucAlertRawData[i]);

			PDEBUG(SANITY_INFO, "\n");

			/* set all endpoints inactive*/
			for (i = 0; i < MAX_ENDPOINTS; i++)
			{
				set_endpoint_event(ENDPOINT_EVENT_NONE, i);
				set_endpoint_state(ENDPOINT_STATE_IDLE, i);
			}

		}

		break;

	case eVAPI_SPI_EVENT:
	case eVAPI_TONE_GEN_CMPLT_EVENT:
	case eVAPI_SSRC_VIOLATION_EVENT:
	case eVAPI_NTE_TRANSMIT_COMPLETE_EVENT:
	case eVAPI_NTE_RECVD_EVENT:
	case eVAPI_G711_CONCURRENT_DECODER_EVENT:
	case eVAPI_PASSTHRU_CONCURRENT_DECODER_EVENT:
	case eVAPI_CALLER_ID_DETECTED_EVENT:
	case eVAPI_REMOTE_DETECT_EVENT:
	case eVAPI_FAX_SWITCH_CMPLT_EVENT:
		break;


	case eVAPI_UNDEFINED_EVENT:
		PDEBUG(SANITY_FUNCTION, "event %d received", eEventCode);
		this_event = (SUndefinedEventParams *) pvData;

		if (this_event->ucCmdType == CMD_TYPE_INDICATION)
			PDEBUG(SANITY_FUNCTION, "indication 0x%04x received %d params", this_event->usFnCode, this_event->usNoOfParams);

		for (i=0 ; i < this_event->usNoOfParams; i ++)
			printf ("0x%04x ", this_event->ausParams[i]);
		printf("\n");

		break;
	default:
		break;
	}

	return;
}

/*=================================================================================*/
/*!	\brief State machine for VAPI API connection level test. 
 *	\section Description
 *	This function handles the states of the endpoints.\n
 *	
 *	\section Inputs-Outputs
 *		None
 *	
 *	\section Execution
 *	A finite while loop is running to check the state of the endpoints array.\n
 *	For each state a limited number of event are handled.\n
 *	The events are set by the comcerto_response_handler or the comcerto_indication_handler.
 *	If the event is handled in the current state, an action is taken accordingly for the current endpoint.
 *	Else the endpoint keep the same state.
 *	The while loop stops when all the endpoints are in the ENDPOINT_STATE_FINISH state
*/
/*=================================================================================*/
void *state_machine_handler(void *none)
{
	int i, k;
	SRequest request;
	VSTATUS status;
	U8 on_going = 1;
	U8 stop_mask = 0;

	PDEBUG(SANITY_INFO, "State machine started");
	while (on_going)
	{
		if (alert_received == True)
			break;

		stop_mask = ENDPOINT_STATE_FINISH;
		/* Parse all the endpoints in the global endpoints array */
		for (i = 0; i < MAX_ENDPOINTS; i++)
		{
			/* If the state is IDLE we should receive only OFFHOOK or RING events */
			switch (endpoints[i].state)
			{
			case ENDPOINT_STATE_IDLE:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
						endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_VOIP_CHANNEL_CREATED:
					PDEBUG(SANITY_INFO, "VAPI_PlayTone(%d) SYNC TDM side;\n", endpoints[i].index);
					status = VAPI_PlayTone(endpoints[i].index, eDIALTONE, eDirToTDM, NULL, 0, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_StopTone(%d) SYNC TDM side;\n", endpoints[i].index);
					status = VAPI_StopTone(endpoints[i].index, 100, 1, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			case ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_VOIP_CHANNEL_CREATED:
					/* The VoIP channel is created we set now the IP parameters on both sides */
					set_endpoint_state(ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK,
							   endpoints[i].index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					set_ip_udp_parameters(endpoints[i].index, ASYNC,
							      ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK);

					set_endpoint_state(ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK,
							   endpoints[i].peer_index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].peer_index);
					set_ip_udp_parameters(endpoints[i].peer_index, ASYNC,
							      ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK);

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			/* This endpoint has a VoIP channel associated, the Command to set the IP params been posted. 
			 * We are waiting for the response.
			 * In case of originate side the next action is to stqrt the packect processing
			 */
			case ENDPOINT_STATE_WAIT_IP_PARAMETERS_ORIGINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_IP_PARAMETERS_SET:
					set_endpoint_state(ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK,
							   endpoints[i].index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					start_voip_endpoint(endpoints[i].index, ASYNC,
							    ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK);

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}

				break;

			/* This endpoint has a VoIP channel associated, the Command to set the IP params been posted. 
			 * We are waiting for the response.
			 * In case of terminate side the next action is to stqrt the packect processing
			 */
			case ENDPOINT_STATE_WAIT_IP_PARAMETERS_TERMINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_IP_PARAMETERS_SET:
					set_endpoint_state(ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK,
							   endpoints[i].index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					start_voip_endpoint(endpoints[i].index, ASYNC,
							    ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK);

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			/* This endpoint has a VoIP channel started, the VoIP conenction is now running. */
			case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_ORIGINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_CONNECTION_ENABLED:
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

					if(test_level > TEST_LEVEL_NO_CFG)
					{
						PDEBUG(SANITY_INFO, "Originate side running some API in SYNC mode \n");
	
						PDEBUG(SANITY_INFO, "VAPI_SetEthMac(%d) SYNC;\n", endpoints[i].index);
						status =
						VAPI_SetEthMac(endpoints[i].index, CMD_LEVEL_CONN, device_mac_addr,
								host_mac_addr, NULL);
						if (status != SUCCESS)
							PRINT_ERROR;
	
						if (skip_complex == 0)
						{
							PDEBUG(SANITY_INFO, "VAPI_SetPayloadType(%d) eG726_32 ,0x44, eBoth SYNC;\n", endpoints[i].index);
							status =
							VAPI_SetPayloadType(endpoints[i].index, eG726_32, 0x44, eBoth,
										NULL);
							if (status != SUCCESS)
								PRINT_ERROR;
	
							PDEBUG(SANITY_INFO, "VAPI_SetCodecType(%d) eG726_32  SYNC;\n", endpoints[i].index);
							status = VAPI_SetCodecType(endpoints[i].index, eG726_32_ID, NULL);
							if (status != SUCCESS)
								PRINT_ERROR;
						}

					PDEBUG(SANITY_INFO, "VAPI_PlayTone(%d) SYNC TDM side;\n", endpoints[i].index);
					status = VAPI_PlayTone(endpoints[i].index, eDIALTONE, eDirToTDM, NULL, 0, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_PlayTone(%d) SYNC IP side;\n", endpoints[i].index);
					status = VAPI_PlayTone(endpoints[i].index, eDIALTONE, eDirToIP, NULL, 0, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_StopTone(%d) SYNC TDM side;\n", endpoints[i].index);
					status = VAPI_StopTone(endpoints[i].index, 100, 1, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_StopTone(%d) SYNC IP side;\n", endpoints[i].index);
					status = VAPI_StopTone(endpoints[i].index, 100, 2, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_PlayTone(%d) SYNC TDM side;\n", endpoints[i].index);
					status = VAPI_PlayTone(endpoints[i].index, eDIALTONE, eDirToTDM, NULL, 0, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_PlayTone(%d) SYNC IP side;\n", endpoints[i].index);
					status = VAPI_PlayTone(endpoints[i].index, eDIALTONE, eDirToIP, NULL, 0, NULL);
					if (status != SUCCESS)

					PDEBUG(SANITY_INFO, "VAPI_StopTone(%d) SYNC Both sides;\n", endpoints[i].index);
					status = VAPI_StopTone(endpoints[i].index, 100, 0, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_SetPacketInterval(%d) 40 SYNC;\n", endpoints[i].index);
					status = VAPI_SetPacketInterval(endpoints[i].index, 40, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_EchoCancellerReset(%d) SYNC;\n", endpoints[i].index);
					status = VAPI_EchoCancellerReset(endpoints[i].index, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_SetRtpSsrc(%d) SYNC;\n", endpoints[i].index);
					status = VAPI_SetRtpSsrcHeader(endpoints[i].index, 0xaabbccdd, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_SendNteEvent(%d) SYNC;\n", endpoints[i].index);
					status = VAPI_SendNteEvent(endpoints[i].index, 0x20000bb8, 0, 0, 0, 0, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					} /* end of TEST_LEVEL_NO_CFG */

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			/* This endpoint has a VoIP channel started, the VoIP conenction is now running. */
			case ENDPOINT_STATE_WAIT_ENABLE_CONNECTION_TERMINATE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);
					break;

				case ENDPOINT_EVENT_CONNECTION_ENABLED:
					set_endpoint_state(ENDPOINT_STATE_WAIT_TIMED_CONNECTION, endpoints[i].index);

					if(endpoints[i].timeout == 0)
					{
						set_endpoint_event(ENDPOINT_EVENT_CONNECTION_TIMED, endpoints[i].index);
					}
					else
					{
						set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
						UT_TimerStart(endpoints[i].this_timer, endpoints[i].timeout, timed_connection, (void *)(&endpoints[i]));
						/* change the codec type to generates PTCHNG indications */
						PDEBUG(SANITY_INFO, "Set codec g729(%d) run connection for %d seconds\n", endpoints[i].index, endpoints[i].timeout);
						status = VAPI_SetCodecType(endpoints[i].index, eG729_A, NULL);
						if (status != SUCCESS)
							PRINT_ERROR;

					}
					break;
					case ENDPOINT_EVENT_NONE:
					break;
				}
				break;

			/* This few commands have been sent to this endpoint the last one is a switch to T38. */
			case ENDPOINT_STATE_WAIT_TIMED_CONNECTION:
				switch (endpoints[i].event)
				{
					case ENDPOINT_EVENT_CONNECTION_TIMED:
						set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
	
						if(test_level > TEST_LEVEL_NO_CFG)
						{
							PDEBUG(SANITY_INFO, "Termintate side to run some API in ASYNC mode \n");
							request.uiReqId = 0XFFFF;
							request.pfnIoCompCallback = &comcerto_response_handler;
		
							PDEBUG(SANITY_INFO, "VAPI_SetEthMac(%d) ASYNC;\n", endpoints[i].index);
							status =
							VAPI_SetEthMac(endpoints[i].index, CMD_LEVEL_CONN, device_mac_addr,
									host_mac_addr, &request);
							if (status != VAPI_ERR_PENDING)
								PRINT_ERROR;
		
							if (skip_complex == 0)
							{
								PDEBUG(SANITY_INFO, "VAPI_SetPayloadType(%d) eG726_32 ,0x44,eBoth ASYNC;\n", endpoints[i].index);
								status =
								VAPI_SetPayloadType(endpoints[i].index, eG726_32, 0x44, eBoth,
											&request);
								if (status != VAPI_ERR_PENDING)
									PRINT_ERROR;
		
								PDEBUG(SANITY_INFO, "VAPI_SetCodecType(%d) eG726_32 ASYNC;\n", endpoints[i].index);
								status = VAPI_SetCodecType(endpoints[i].index, eG726_32_ID, &request);
								if (status != VAPI_ERR_PENDING)
									PRINT_ERROR;
							}
		
							PDEBUG(SANITY_INFO, "VAPI_SetPacketInterval(%d) 40 ASYNC;\n", endpoints[i].index);
							status = VAPI_SetPacketInterval(endpoints[i].index, 40, &request);
							if (status != VAPI_ERR_PENDING)
								PRINT_ERROR;
		
							PDEBUG(SANITY_INFO, "VAPI_SetRTpSsrc(%d) ASYNC;\n", endpoints[i].index);
							status = VAPI_SetRtpSsrcHeader(endpoints[i].index, 0xaabbccdd, &request);
							if (status != VAPI_ERR_PENDING)
								PRINT_ERROR;
		
							PDEBUG(SANITY_INFO, "VAPI_SendNteEvent(%d) ASYNC;\n", endpoints[i].index);
		
							status =
							VAPI_SendNteEvent(endpoints[i].index, 0x20000bb8, 0, 0, 0, 0, &request);
							if (status != VAPI_ERR_PENDING)
								PRINT_ERROR;
		
							PDEBUG(SANITY_INFO, "VAPI_EchoCancellerReset(%d) ASYNC;\n", endpoints[i].index);
							status = VAPI_EchoCancellerReset(endpoints[i].index, &request);
							if (status != VAPI_ERR_PENDING)
								PRINT_ERROR;
		
							if (skip_t38 == 0)
							{
								PDEBUG(SANITY_INFO, "VAPI_T38 switch(%d) ASYNC;\n", endpoints[i].index);
								status = VAPI_SwitchToT38(endpoints[i].index, NULL, &request);
								if (status != VAPI_ERR_PENDING)
									PRINT_ERROR;
							}
		
							PDEBUG(SANITY_INFO, "VAPI_ConfigureT38Options(%d) ASYNC;\n", endpoints[i].index);
							request.uiReqId = ENDPOINT_STATE_WAIT_T38_OPTIONS_ACK;
							request.pfnIoCompCallback = &comcerto_response_handler;
							set_endpoint_state(ENDPOINT_STATE_WAIT_T38_OPTIONS_ACK, endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

							status =  VAPI_ConfigureT38Options(endpoints[i].index, &g_fax_configuration,
										&request);
							if (status != VAPI_ERR_PENDING)
								PRINT_ERROR;

						}
						else
						{
							PDEBUG(SANITY_INFO, "VAPI_DestroyConnection(%d) ASYNC;\n", endpoints[i].index);
							set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
							set_endpoint_state(ENDPOINT_STATE_FINISH, endpoints[i].index);
							status = destroy_endpoint(endpoints[i].index, ASYNC, ENDPOINT_STATE_FINISH);
							if (status != VAPI_ERR_PENDING)
								PRINT_ERROR;

							PDEBUG(SANITY_INFO, "VAPI_ConfigureT38Options(%d) SYNC;\n", endpoints[i].peer_index);
							status = VAPI_ConfigureT38Options(endpoints[i].peer_index, &g_fax_configuration, NULL);
							if (status != SUCCESS)
								PRINT_ERROR;
		
							if (skip_t38 == 0)
							{
								PDEBUG(SANITY_INFO, "VAPI_T38 switch(%d) SYNC;\n", endpoints[i].peer_index);
								status = VAPI_SwitchToT38(endpoints[i].peer_index, NULL, NULL);
								if (status != SUCCESS)
									PRINT_ERROR;
							}

							PDEBUG(SANITY_INFO, "VAPI_DestroyConnection(%d) SYNC;\n", endpoints[i].peer_index);
							status = destroy_endpoint(endpoints[i].peer_index, SYNC, 0xFFFF);
							set_endpoint_state(ENDPOINT_STATE_FINISH, endpoints[i].peer_index);
							if (status != SUCCESS)
								PRINT_ERROR;

						}

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			/* This few commands have been sent to this endpoint the last one is a switch to T38. */
			case ENDPOINT_STATE_WAIT_T38_OPTIONS_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_T38_OPTIONS_ACK:
					PDEBUG(SANITY_INFO, "VAPI_DestroyConnection(%d) ASYNC\n", endpoints[i].index);
					set_endpoint_state(ENDPOINT_STATE_FINISH, endpoints[i].index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					status = destroy_endpoint(endpoints[i].index, ASYNC, ENDPOINT_STATE_FINISH);

					if (status != VAPI_ERR_PENDING)
						PRINT_ERROR;

					PDEBUG(SANITY_INFO, "VAPI_ConfigureT38Options(%d) SYNC;\n", endpoints[i].peer_index);
					status = VAPI_ConfigureT38Options(endpoints[i].peer_index, &g_fax_configuration, NULL);
					if (status != SUCCESS)
						PRINT_ERROR;

					if (skip_t38 == 0)
					{
						PDEBUG(SANITY_INFO, "VAPI_T38 switch(%d) SYNC;\n", endpoints[i].peer_index);
						status = VAPI_SwitchToT38(endpoints[i].peer_index, NULL, NULL);
						if (status != SUCCESS)
							PRINT_ERROR;
					}

					PDEBUG(SANITY_INFO, "VAPI_DestroyConnection(%d) SYNC;\n", endpoints[i].peer_index);
					status = destroy_endpoint(endpoints[i].peer_index, SYNC, 0xFFFF);
					set_endpoint_state(ENDPOINT_STATE_FINISH, endpoints[i].peer_index);
					if (status != SUCCESS)
						PRINT_ERROR;
					PDEBUG(SANITY_FUNCTION, "Endpoint %d, %d ptchng indication received", endpoints[i].peer_index,
						 endpoints[i].ptchng_count);

					break;

				case ENDPOINT_EVENT_FINISH:
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					set_endpoint_state(ENDPOINT_STATE_FINISH, endpoints[i].index);
					PDEBUG(SANITY_FUNCTION, "Endpoint %d, %d ptchng indication received", endpoints[i].index, endpoints[i].ptchng_count);

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			/*  */
			case ENDPOINT_STATE_FINISH:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_FINISH:
					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			default:
				PDEBUG(SANITY_ERROR, "Endpoint %d: Wrong state %d",i, endpoints[i].state);

			}
		}

		for (k = 0; k < number_of_endpoints; k++)
		{
			stop_mask &= endpoints[k].state;
		}

		if (stop_mask == ENDPOINT_STATE_FINISH)
		{
			PDEBUG(SANITY_INFO, "state_machine_handler: All endpoints are on hook");
			on_going = 0;
		}

#if !defined(_VXWORKS_)
		usleep(100);
#else
		taskDelay(1);
#endif
	}

	if (alert_received == True)
	{
#if !defined(_VXWORKS_)
		core_dump(device_id);
#endif
		main_exit(SUCCESS);
	}

	PDEBUG(SANITY_INFO, "End test connection\n");
	return NULL;
}

/*=================================================================================*/
/*!	\brief State machine for VAPI API conference level test. 
 *	\section Description
 *	This function handles the states of the endpoints for conferecing operation.\n
 *	
 *	\section Inputs-Outputs
 *		None
 *	
 *	\section Execution
 *	A finite while loop is running to check the state of the endpoints array.\n
 *	For each state a limited number of event are handled.\n
 *	The events are set by the comcerto_response_handler or the comcerto_indication_handler.
 *	If the event is handled in the current state, an action is taken accordingly for the current endpoint.
 *	Else the endpoint keep the same state.
 *	The while loop stops when all the endpoints are in the ENDPOINT_STATE_CONFERENCE_FINISH state
*/
/*=================================================================================*/
void *conference_state_machine_handler(void *none)
{
	int i, k;
	SRequest request;
	VSTATUS status;
	U8 on_going = 1;
	U8 stop_mask;
	/* We should have here a signal handler to interrupt the while loop
	 * For now loop forever, CTRL-C to be used to stop.
	 */
	PDEBUG(SANITY_INFO, "conference state machine handler");
	while (on_going)
	{
		if (alert_received == True)
			break;

		stop_mask = ENDPOINT_STATE_CONFERENCE_FINISH;
		/* Parse all the endpoints in the global endpoints array */
		for (i = 0; i < MAX_ENDPOINTS; i++)
		{
			/* If the state is IDLE we should receive only OFFHOOK or RING events */
			switch (endpoints[i].state)
			{
			case ENDPOINT_STATE_IDLE:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			case ENDPOINT_STATE_WAIT_CREATE_TDM_PARTICIPANT_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_TDM_PARTICIPANT_CREATED:
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

					if ( test_level > TEST_LEVEL_NO_CFG)
					{
						PDEBUG(SANITY_INFO, "Enable Participant(%d) SYNC;\n",endpoints[i].index);
						status = VAPI_EnableParticipant(endpoints[i].index, NULL);
						if (status != SUCCESS)
							PRINT_ERROR;
	
						PDEBUG(SANITY_INFO, "Participant VAPI_SetPacketInterval(%d) SYNC;\n",endpoints[i].index);
						status = VAPI_SetPacketInterval(endpoints[i].index, 20, NULL);
						if (status != SUCCESS)
							PRINT_ERROR;
	
						PDEBUG(SANITY_INFO, "Mute Participant(%d) SYNC;\n",endpoints[i].index);
						status = VAPI_MuteParticipant(endpoints[i].index, 1, NULL);
						if (status != SUCCESS)
							PRINT_ERROR;
	
						PDEBUG(SANITY_INFO, "UnMute Participant(%d) SYNC;\n",endpoints[i].index);
						status = VAPI_MuteParticipant(endpoints[i].index, 0, NULL);
						if (status != SUCCESS)
							PRINT_ERROR;
	
						PDEBUG(SANITY_INFO, "Hold Participant(%d) SYNC;\n",endpoints[i].index);
						status = VAPI_PutParticipantOnHold(endpoints[i].index, eUNHOLD, 0, NULL);
						if (status != SUCCESS)
							PRINT_ERROR;
	
						PDEBUG(SANITY_INFO, "UnHold Participant(%d) SYNC;\n",endpoints[i].index);
						status = VAPI_PutParticipantOnHold(endpoints[i].index, eSIMPLEHOLD, 0, NULL);
						if (status != SUCCESS)
							PRINT_ERROR;
					}

					PDEBUG(SANITY_INFO, "Destroy participant(%d) SYNC;\n",endpoints[i].index);
					status = VAPI_DestroyParticipant(endpoints[i].index, NULL);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					set_endpoint_state(ENDPOINT_STATE_CONFERENCE_FINISH, endpoints[i].index);
					if (status != SUCCESS)
						PRINT_ERROR;
					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			case ENDPOINT_STATE_WAIT_CREATE_IP_PARTICIPANT_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_IP_PARTICIPANT_CREATED:
					set_endpoint_state(ENDPOINT_STATE_WAIT_IP_PARAMETERS_IP_PARTICIPANT_ACK,
							   endpoints[i].index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					set_ip_udp_parameters(endpoints[i].index, ASYNC,
							      ENDPOINT_STATE_WAIT_IP_PARAMETERS_IP_PARTICIPANT_ACK);

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			/* This endpoint has a VoIP channel associated, the Command to set the IP params been posted. 
			 * We are waiting for the response.
			 * In case of originate side the next action is to stqrt the packect processing
			 */
			case ENDPOINT_STATE_WAIT_IP_PARAMETERS_IP_PARTICIPANT_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_IP_PARAM_IP_PARTICIPANT_CONFIGURED:
					set_endpoint_state(ENDPOINT_STATE_WAIT_IP_PARTICIPANT_ENABLE_ACK,
							   endpoints[i].index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

					request.uiReqId = ENDPOINT_STATE_WAIT_IP_PARTICIPANT_ENABLE_ACK;
					request.pfnIoCompCallback = &comcerto_response_handler;

					status = VAPI_EnableParticipant(endpoints[i].index, &request);
					if (status != VAPI_ERR_PENDING)
						PRINT_ERROR;

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}

				break;

			/* This endpoint has a VoIP channel started, the VoIP conenction is now running. */
			case ENDPOINT_STATE_WAIT_IP_PARTICIPANT_ENABLE_ACK:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_IP_PARTICIPANT_ENABLED:
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);

					if ( test_level > TEST_LEVEL_NO_CFG)
					{
						request.uiReqId = 0XFFFF;
						request.pfnIoCompCallback = &comcerto_response_handler;

						PDEBUG(SANITY_INFO, "Enable Participant(%d) ASYNC;\n",endpoints[i].index);
						status = VAPI_EnableParticipant(endpoints[i].index, &request);
						if (status != VAPI_ERR_PENDING)
							PRINT_ERROR;
	
						PDEBUG(SANITY_INFO, "Participant VAPI_SetPacketInterval(%d) ASYNC;\n",endpoints[i].index);
						status = VAPI_SetPacketInterval(endpoints[i].index, 20, &request);
						if (status != VAPI_ERR_PENDING)
							PRINT_ERROR;
	
						PDEBUG(SANITY_INFO, "Mute Participant(%d) ASYNC;\n",endpoints[i].index);
						status = VAPI_MuteParticipant(endpoints[i].index, 1, &request);
						if (status != VAPI_ERR_PENDING)
							PRINT_ERROR;
	
						PDEBUG(SANITY_INFO, "UnMute Participant(%d) ASYNC;\n",endpoints[i].index);
						status = VAPI_MuteParticipant(endpoints[i].index, 0, &request);
						if (status != VAPI_ERR_PENDING)
							PRINT_ERROR;
	
						PDEBUG(SANITY_INFO, "Hold Participant(%d) ASYNC;\n",endpoints[i].index);
						status = VAPI_PutParticipantOnHold(endpoints[i].index, eUNHOLD, 0, &request);
						if (status != VAPI_ERR_PENDING)
							PRINT_ERROR;
	
						PDEBUG(SANITY_INFO, "UnHold Participant(%d) ASYNC;\n",endpoints[i].index);
						status = VAPI_PutParticipantOnHold(endpoints[i].index, eSIMPLEHOLD, 0, 
							&request);
	
						if (status != VAPI_ERR_PENDING)
							PRINT_ERROR;
					}

					PDEBUG(SANITY_INFO, "Destroy participant(%d) ASYNC (%d)\n", endpoints[i].index, number_of_participants);

					request.uiReqId = ENDPOINT_STATE_CONFERENCE_FINISH;
					request.pfnIoCompCallback = &comcerto_response_handler;

					status = VAPI_DestroyParticipant(endpoints[i].index, &request);

					if (status != VAPI_ERR_PENDING)
						PRINT_ERROR;
					break;

				case ENDPOINT_EVENT_FINISH:
					set_endpoint_state(ENDPOINT_STATE_CONFERENCE_FINISH, endpoints[i].index);
					set_endpoint_event(ENDPOINT_EVENT_NONE, endpoints[i].index);
					break;

				}

				break;

			/*  */
			case ENDPOINT_STATE_CONFERENCE_FINISH:
				switch (endpoints[i].event)
				{
				case ENDPOINT_EVENT_ON_HOOK:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d on hook", endpoints[i].index);
					onhook_handler(endpoints[i].index);

					break;

				case ENDPOINT_EVENT_NONE:
					break;

				default:
					PDEBUG(DEBUG_FUNCTION, "Endpoint %d: Wrong event %d in state %d",
					       endpoints[i].index, endpoints[i].event, endpoints[i].state);
				}
				break;

			default:
				PDEBUG(SANITY_ERROR, "Endpoint %d: Wrong state %d",i ,endpoints[i].state);
			}
		}

		for (k = 0; k < number_of_participants; k++)
		{
			stop_mask &= endpoints[k].state;
		}
		
		if (stop_mask == ENDPOINT_STATE_CONFERENCE_FINISH)
		{
			PDEBUG(SANITY_INFO, "conference_state_machine_handler: All participants are on hook");
			on_going = 0;
		}

#if !defined(_VXWORKS_)
		usleep(100);
#else
		taskDelay(1);
#endif
	}

	if (alert_received == True)
	{
#if !defined(_VXWORKS_)
		core_dump(device_id);
#endif
		main_exit(SUCCESS);
	}

	PDEBUG(SANITY_INFO, "End test conference\n");
	return NULL;
}

#if !defined(_VXWORKS_)
static int
get_int(const char *s, int *value)
{
	int i;
	char *endptr = NULL;

	if (!s || !*s)
		goto err;

	i = strtol(s, &endptr, 0);
	if (!i && s == endptr)
		goto err;

	if (value)
		*value = i;

	return 0;

err:
	return -1;
}

/*=================================================================================*/
/*!	\brief command line option parser
 *	\section Description
 *	This function is the command line option parser.\n
 */	
/*=================================================================================*/
static int get_opt(int argc, char *argv[])
{

	int c;
	int status = SUCCESS;
	
	while ((c = getopt_long(argc, argv, "a:d:e:v:x:p:f:F:c:l:m:t:o:n:g:r:", long_options, NULL)) != - 1) {
		switch (c) {
		case 'a':
			get_int(optarg, (int *) &connection_create);
			break;

		case 'd':
			get_int(optarg, (int *) &device_itf);
			break;

		case 'e':
			get_int(optarg, (int *) &number_of_endpoints);

			if (number_of_endpoints > MAX_ENDPOINTS)
				number_of_endpoints = MAX_ENDPOINTS;

			skip_complex = 0;
			skip_t38 = 0;
			break;

		case 'v':
			get_int(optarg, (int *) &number_of_endpoints);

			if (number_of_endpoints > MAX_ENDPOINTS)
				number_of_endpoints = MAX_ENDPOINTS;

			skip_complex = 1;
			skip_t38 = 1;
			break;

		case 'x':
			get_int(optarg, (int *) &number_of_endpoints);

			if (number_of_endpoints > MAX_ENDPOINTS)
				number_of_endpoints = MAX_ENDPOINTS;

			skip_complex = 0;
			skip_t38 = 1;
			break;

		case 'p':
			get_int(optarg, (int *) &number_of_participants);

			if (number_of_participants > MAX_PARTICIPANTS)
				number_of_participants = MAX_PARTICIPANTS;
			break;

		case 'f':
			strncpy(firmware, optarg, MAX_STRING);
			boot_option_file = 0;
			break;

		case 'F':
			strncpy(firmware, optarg, MAX_STRING);
			boot_option_file = 1;
			break;

		case 'c':
			strncpy(config_file, optarg, MAX_STRING);
			break;

		case 'l':
			get_int(optarg, (int *) &number_of_loops);
			break;

		case 'm':
			get_int(optarg, (int *) &debug_level);
			break;

		case 't':
			get_int(optarg, (int *) &test_level);
			break;

		case 'o':
			get_int(optarg, (int *) &ontime_connection);
			break;

		case 'n':
			get_int(optarg, (int *) &dual_echo_canceller);
			break;

		case 'g':
			get_int(optarg, (int *) &diag_function_code);
			break;

		case 'r':
			get_int(optarg, (int *) &recover_option);
			break;

		case -2:
			show_help();
			exit(0);
			break;

		case -3:
			show_version();
			exit(0);
			break;

		default:
			status = -1;
		}
	}

	return status;
}

#else
/*=================================================================================*/
/*!	\brief command line option parser
 *	\section Description
 *	This function is the command line option parser.\n
 *	
 *	\section Inputs-Outputs
 *		\param key this is the option (i.e -e)
 *		\param char this is he value of the current option
 *		\state 
 *	
 *	\section Execution
 *	
*/
/*=================================================================================*/
error_t parser(int key, char *arg, struct argp_state *state)
{
	switch (key)
	{
	case 'a':
		connection_create = strtoul(arg, NULL, 0);
		break;

	case 'd':
		device_itf = strtoul(arg, NULL, 0);
		break;

	case 'e':
		number_of_endpoints = strtoul(arg, NULL, 0);

		if (number_of_endpoints > MAX_ENDPOINTS)
			number_of_endpoints = MAX_ENDPOINTS;

		skip_complex = 0;
		skip_t38 = 0;
		break;

	case 'v':
		number_of_endpoints = strtoul(arg, NULL, 0);

		if (number_of_endpoints > MAX_ENDPOINTS)
			number_of_endpoints = MAX_ENDPOINTS;

		skip_complex = 1;
		skip_t38 = 1;
		break;

	case 'x':
		number_of_endpoints = strtoul(arg, NULL, 0);

		if (number_of_endpoints > MAX_ENDPOINTS)
			number_of_endpoints = MAX_ENDPOINTS;

		skip_complex = 0;
		skip_t38 = 1;
		break;

	case 'p':
		number_of_participants = strtoul(arg, NULL, 0);

		if (number_of_participants > MAX_PARTICIPANTS)
			number_of_participants = MAX_PARTICIPANTS;
		break;

	case 'f':
		strncpy(firmware, arg, MAX_STRING);
		boot_option_file = 0;
		break;

	case 'F':
		strncpy(firmware, arg, MAX_STRING);
		boot_option_file = 1;
		break;

	case 'c':
		strncpy(config_file, arg, MAX_STRING);
		break;

	case 'l':
		number_of_loops = strtoul(arg, NULL, 0);
		break;

	case 'm':
		debug_level = strtoul(arg, NULL, 0);
		break;

	case 't':
		test_level = strtoul(arg, NULL, 0);
		break;

	case 'o':
		ontime_connection = strtoul(arg, NULL, 0);
		break;

	case 'n':
		dual_echo_canceller = strtoul(arg, NULL, 0);
		break;

	case 'g':
		diag_function_code = strtoul(arg, NULL, 0);
		break;

	case 'r':
		recover_option = strtoul(arg, NULL, 0);
		break;

	default:
		return ARGP_ERR_UNKNOWN;
	}

	return 0;
}

/* parser option structure */
static struct argp argp = { options, parser, 0, doc, };
#endif

/***************************************************************************
 * main
 ***************************************************************************/
/*! \brief Application entry.

 *  \section Description
	This is the main function of the application.\n
	It does:\n
	 - endpoints array initialization
	 - command line arguments options parsing (device ID, firmware filename, configuration file, etc ...)
	 - then it:
		- Initialises the setup configuration using parameters from the configration file (Device type, Control interface, Slave/Master mode , etc)
		- Prints the VAPI verson (VAPI_GetVersion)
		- Configures the GTL layer (set_device_configuration)
		- Initialises VAPI (VAPI_Init)
		- Open the Device (VAPI_OpenDevice) 
		- Boot the Device (VAPI_AssignBootMAC for csme, VAPI_BootDevice) if in Slave mode.
		- Initialise the TDM buses using parameters from the configration file (VAPI_SetTDMParams).
		- Using VAPI_InitDevice set the SPU feature and enable the Multicommand mode.
		- Set the Network Parameters (MAC, IP addresses) using parameters from the configration file (VAPI_SetEthMac, VAPI_SetDeviceIPAddr).
		- Display device informations (ARM Version, DSP version, Device type) using the VAPI_Passthru command.
		- Starts the connection state machine
		- Starts the conference state machine
		- Shutdown the application on test completion
		
	All the device level VAPI APIs are called in Synchronous mode.
	The connection level VAPI APIs are called in Asynchronous mode.

*/

#if !defined(_VXWORKS_)
int main(int argc, char *argv[])
#else
int vapi_sanity(char *args)
#endif
{
	int status;
	int i, j, k;
	SRequest local_request;
	SConfParams conference_parameters;
	SConfDTSParams dominant_talkers;
	U8 on_going;
	U8 stop_mask = 0;
	struct redirect_info diag_redirect;

	loop_counter = 0;

	alert_received = False;

	/* Initialise the endpoint info array */
	memset(&endpoints, 0, sizeof(endpoints));

	g_fax_configuration.bUseExistingOpts = False;
	g_fax_configuration.ucFaxTxLevel = 9;
	g_fax_configuration.bEnableFaxAutoSwitch = False;
	g_fax_configuration.usSwitchoverEventMask = 0;
	g_fax_configuration.stFaxOpts.ucT38PktLossConcealment = 1;
	g_fax_configuration.stFaxOpts.ucECMDisable = 1;
	g_fax_configuration.stFaxOpts.ucFaxDataRedundancyCnt = 3;
	g_fax_configuration.stFaxOpts.ucT30RedundancyCnt = 7;
	g_fax_configuration.stFaxOpts.ucFaxConnSpeedLimit = 6;
	g_fax_configuration.stFaxOpts.ucErrRcvyMethod = 0;
	g_fax_configuration.stFaxOpts.ucTCFProcedure = 0;
	g_fax_configuration.stFaxOpts.ucNumFEC = 0;
	g_fax_configuration.stFaxOpts.ucNumIFPs = 4;

#if !defined(_VXWORKS_)
	if (argc < 2)
	{
		PDEBUG(SANITY_ERROR, "Please enter at least 1 argument");
		return -1;
	}

	/* returns value < 0 on error, == 0 if all's Ok */
	status = get_opt(argc, argv);

	if (status < 0)
		return -1;
#else
	if (!args || !*args)
	{
		PDEBUG(SANITY_ERROR, "Please enter at least 1 argument");
		return -1;
	}

	/* returns value < 0 on error, > 1 on help or version request, == 0 if all's Ok */
	status = argp_parse(&argp, args);

	if (status > 0)
		return 0;
	if (status < 0)
		return -1;
#endif

	printf("===============================================================\n");
	printf("VAPI SANITY: Compiled at Date %s, Time %s \nVAPI version: %s\n",
				(char *)&__DATE__, (char *)&__TIME__, VAPI_GetVersion());
	printf("===============================================================\n");
	printf("Command line:");
#if !defined(_VXWORKS_)
	for(i = 0; i < argc; i++)
		printf("%s ", argv[i]);
#else
	printf("%s ", args);
#endif
	printf("\n");
	printf("===============================================================\n");
	printf("connection_create option = %d\n", connection_create);
	printf("device_itf = %d\n", device_itf);
	printf("number_of_endpoints = %d\n", number_of_endpoints);
	printf("number_of_participants = %d\n", number_of_participants);
	printf("firmware = %s\n", firmware);
	printf("config_file = %s\n", config_file);
	printf("number_of_loops = %d\n", number_of_loops);
	printf("debug_level = %d\n", debug_level);
	printf("test_level = %d\n", test_level);
	printf("ontime_connection = %d\n", ontime_connection);
	printf("dual_echo_canceller = %d\n", dual_echo_canceller);
	printf("diag_function_code = %d\n", diag_function_code);
	printf("recover_option = %d\n", recover_option);
	printf("===============================================================\n");

	sprintf(error_string, "SUCCESS");

	for (i = 0; i < MAX_ENDPOINTS; i++)	/* there are only 2 endpoints handled */
	{
		endpoints[i].index = i;
		/* timeslot used by create_endpoint */
		endpoints[i].timeslot = i;
		/* give a different port number to each endpoints in the array */
		/* the RTP port must be even */
		endpoints[i].rtp_port = DEFAULT_UDP_PORT + i * 2;
		/* the RTCP port must be odd */
		endpoints[i].rtcp_port = DEFAULT_UDP_PORT + i;
		PDEBUG(DEBUG_FUNCTION, "endpoints[%d].rtp_port %d", i, endpoints[i].rtp_port);

		/* Cross the endpoints 2 by 2
		 * (endpoint 0 -> endpoint 1
		 * (endpoint 1 -> endpoint 0
		 */
		if (i % 2)
		{
			endpoints[i].peer_index = i - 1;
			endpoints[i - 1].peer_index = i;
		}
		endpoints[i].this_timer = UT_TimerCreate();
		endpoints[i].timeout = ontime_connection;
	}

	if (device_itf == eCSM_ITF)
	{
		status = set_device_configuration(config_file, &gtl_csme_config, device_itf);
		gtl_device_configuration = (SDeviceHeader *) &gtl_csme_config;
	}
	else if (device_itf == ePCI_ITF)
	{
		status = set_device_configuration(config_file, &gtl_pci_config, device_itf);
		gtl_device_configuration = (SDeviceHeader *) &gtl_pci_config;
	}
	else
	{
		fprintf(stderr, "Control interface %d not supported\n", device_itf);
		status = FAILURE;
	}
	if (status != SUCCESS)
	{
		/* since init failed we can't use VAPI logging facilities */
		fprintf(stderr, "set_device_configuration Failed, status: %d: Exit\n", status);
		goto error_exit;
	}

	device_id = gtl_device_configuration->uiDevId;

	status = VAPI_Init(gtl_device_configuration);
	if (status != SUCCESS)
	{
		/* since init failed we can't use VAPI logging facilities */
		fprintf(stderr, "VAPI_Init Failed, status: %d\n", status);
		goto error_exit;
	}

#if defined(_VXWORKS_)
	taskPriorityGet(taskIdSelf(), &i);
	taskPrioritySet(taskIdSelf(), i+1);
#endif

	status = VAPI_OpenDevice(device_id, comcerto_indication_handler);

	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "VAPI_OpenDevice %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error VAPI_OpenDevice");

	if ((gtl_device_configuration->eDevMode != eMASTER) && (recover_option != True))
	{
		PDEBUG(SANITY_INIT, "Starting device boot procedure");

		status = boot_device(firmware);

		if (status == SUCCESS)
			PDEBUG(SANITY_INFO, "Boot Device %d SUCCESS\n", device_id);
		else
			ERROR_EXIT(">>>>>>>>>>> error Boot Device");
	}

	if(recover_option == True)
	{
		status = VAPI_InitDevice(device_id,
					VAPI_DEV_OPMODE_RECOVER,
					(dual_echo_canceller == True) ? VAPI_DEV_PROF_SPU_DFECAN_ENABLE : VAPI_DEV_PROF_DEFAULT,
					NULL, NULL);
	}
	else
	{
		status = VAPI_InitDevice(device_id,
					VAPI_DEV_OPMODE_DEFAULT,
					(dual_echo_canceller == True) ? VAPI_DEV_PROF_SPU_DFECAN_ENABLE : VAPI_DEV_PROF_DEFAULT,
					NULL, NULL);
	}

	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "VAPI_InitDevice %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error VAPI_InitDevice");

	/* Set the Network Timming generator for the M823xx device in master mode */
	if ((gtl_device_configuration->ucDevType == DEV_TYPE_M823XX) || 
		(gtl_device_configuration->ucDevType == DEV_TYPE_M823XX_2))
	{
		status = set_ntg_config(device_id);
		if (status == SUCCESS)
			PDEBUG(SANITY_INFO, "set_ntg_config %d SUCCESS\n", device_id);
		else
			ERROR_EXIT(">>>>>>>>>>> error set_ntg_config");
	}

	status = tdm_parameters_initialisation(config_file);
	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "tdm_parameters_initialisation %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error tdm_parameters_initialisation");

	/*For now init MAC address only */
	status = network_parameters_initialisation(config_file, 0);

	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "network_parameters_initialisation %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error network_parameters_initialisation (NO IP)");

	if ((gtl_device_configuration->ucDevType != DEV_TYPE_M821XX) &&
		(gtl_device_configuration->ucDevType != DEV_TYPE_M83XXX) &&
		(gtl_device_configuration->eDevMode == eSLAVE))
	{
		/*Perform multiple IP SRC source */
		status = check_multiple_ip(device_id);
		if (status == SUCCESS)
			PDEBUG(SANITY_INFO, "check_multiple_ip %d SUCCESS\n", device_id);
		else
			ERROR_EXIT(">>>>>>>>>>> error check_multiple_ip");
	}

	/*Now init MAC + IP address */
	status = network_parameters_initialisation(config_file, 1);

	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "network_parameters_initialisation %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error network_parameters_initialisation");

	if (diag_function_code != 0)
	{
		diag_redirect.type = FC_REDIRECT_CSM;
		memcpy(&diag_redirect.dst_mac, host_mac_addr, 6);
	
		status = configure_diag_redirect(&diag_redirect);
		if (status == SUCCESS)
			PDEBUG(SANITY_INFO, "configure_diag_redirect %d SUCCESS\n", device_id);
		else
			ERROR_EXIT(">>>>>>>>>>> error configure_diag_redirect");
	
		status = set_diagnostics(0xffff, CMD_CLASS_OPEN_DIAG, CMD_TYPE_DIAG_MON_LIVE_CTRL, diag_function_code, 1);
		if (status == SUCCESS)
			PDEBUG(SANITY_INFO, "set_diagnostics %d SUCCESS\n", device_id);
		else
			ERROR_EXIT(">>>>>>>>>>> error set_diagnostics");
	}
	display_firmware_version();

	local_request.uiReqId = REQUEST_ID_GENERIC;
	local_request.pfnIoCompCallback = &comcerto_response_handler;

	conference_parameters.bTranscoding = False;
	conference_parameters.bBlockDTMF = False;

	dominant_talkers.bDTSOnOff = True;
	dominant_talkers.usAttackConst = 60;
	dominant_talkers.usNTH1 = 0xFE48;
	dominant_talkers.usNTH2 = 0xFEC0;
	dominant_talkers.usHoldConst = 1000;
	dominant_talkers.usN1 = 1;
	dominant_talkers.usG1 = 0x0000;
	dominant_talkers.usN2 = 2;
	dominant_talkers.usG2 = 0x0064;
	dominant_talkers.usN3 = 3;
	dominant_talkers.usG3 = 0x00C8;

	PDEBUG(SANITY_INFO, "The device has been initialized using VAPI in blocking mode (Synchronous)\n");

	if ((gtl_device_configuration->ucDevType != DEV_TYPE_M821XX) &&
		(gtl_device_configuration->ucDevType != DEV_TYPE_M83XXX))
	{
		status = check_passthru();
		if (status == SUCCESS)
			PDEBUG(SANITY_INFO, "check_passthru %d SUCCESS\n", device_id);
		else
			ERROR_EXIT(">>>>>>>>>>> error check_passthru");
		status = check_cesopsn();
		if (status == SUCCESS)
			PDEBUG(SANITY_INFO, "check_cesopsn %d SUCCESS\n", device_id);
		else
			ERROR_EXIT(">>>>>>>>>>> error check_cesopsn");

		status = check_cmd_class_type();
		if (status == SUCCESS)
			PDEBUG(SANITY_INFO, "check_cmd_class_type %d SUCCESS\n", device_id);
		else
			ERROR_EXIT(">>>>>>>>>>> error check_cmd_class_type");

	}

	status = check_resource_manager();
	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "check_resource_manager %d SUCCESS\n", device_id);
printf("check_resource_manager %d \n", status);
	if (test_level == TEST_LEVEL_DEVICE)
		main_exit(SUCCESS);

	PDEBUG(SANITY_INFO, "Test starts for %d loops with test level %d on: \n -%d endpoints\n -%d participants\n ", number_of_loops, test_level, number_of_endpoints, number_of_participants);

	/*Make sure these Caller ID generation info blocks are not configured*/
	VAPI_SetCidGenInfo (eRemove, eSASToneInfoType, 0);
	VAPI_SetCidGenInfo (eRemove, eNTTInfoType, 0);
	VAPI_SetCidGenInfo (eRemove, eStopBitsInfoType, 0);
	VAPI_SetCidGenInfo (eRemove, ePacketProcessingInfoType, 0);

main_loop:

	for (i = 0; i < MAX_ENDPOINTS; i++)
	{
		endpoints[i].state = ENDPOINT_STATE_IDLE;
		endpoints[i].ptchng_count = 0;
	}

	if (number_of_endpoints == 0)
		goto create_conf;

	/* make sure we have a even number of endpoints */
	if (number_of_endpoints % 2)
		number_of_endpoints++;

	j = number_of_endpoints / 2;

	status = check_play_tone();
	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "check_play_tone %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error check_play_tone");

	status = check_change_conn_type();
	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "check_change_conn_type %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error check_change_conn_type");
	
	status = check_modify_conn();
	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "check_modify_conn %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error check_modify_conn");

	status = check_recover(comcerto_indication_handler);
	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "check_recover %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error check_recover");

	/* always return SUCCESS*/
	VAPI_RegisterReset(0, comcerto_reset_handler, NULL);

	for (i = 0; i < j; i++)
	{
		PDEBUG(SANITY_INFO, "VAPI_CreateConnection(%d) SYNC;\n" ,i);
		status = VAPI_CreateConnection(device_id, i,	/* connection Id */
					       eVOIP, i,	/* timeslot */
					       U_LAW, NULL, NULL);

		if (status != SUCCESS)
			PRINT_ERROR;

		PDEBUG(SANITY_INFO, "VAPI_CreateConnection(%d) SYNC;\n",i + j);
		status = VAPI_CreateConnection(device_id, i + j,/* connection Id */
					       eVOIP, i + j,	/* timeslot */
					       U_LAW, NULL, NULL);

		if (status != SUCCESS)
			PRINT_ERROR;

		if (test_level > TEST_LEVEL_NO_CID)
		{

			PDEBUG(SANITY_INFO, "VAPI_SetCidDetection(%d) SYNC;\n",i);
			status = VAPI_SetCidDetection (i, &CidDetectionCtrl, NULL);
			if (status != SUCCESS)
				PRINT_ERROR;

			PDEBUG(SANITY_INFO, "VAPI_PlayCid(%d) SYNC;\n",i);
			status = VAPI_PlayCid(i, &CallerIdData, NULL);
			if (status != SUCCESS)
				PRINT_ERROR;

			status = connection_rtcp_statistics(i);
			if (status != SUCCESS)
				PRINT_ERROR;
	
			PDEBUG(SANITY_INFO, "VAPI_StartCallerId(%d) ASYNC;\n",i);
			clid_message.bIsOnHook = True;
			status = VAPI_StartCallerId(i + j, &clid_message, &local_request);
			if (status != VAPI_ERR_PENDING)
				PRINT_ERROR;
	
			status = connection_query_rtcpstat(i + j);
			if (status != SUCCESS)
				PRINT_ERROR;
		}
	}

	if (test_level > TEST_LEVEL_NO_CID)
	{
		PDEBUG(SANITY_INFO, "Wait to check if the CallerId indication is received\n");

		on_going = 1;
		while (on_going)
		{
			stop_mask = ENDPOINT_STATE_CID_CMPLT;
			for (k = 0; k < number_of_endpoints; k++)
			{
				stop_mask &= endpoints[k].state;
			}
	
			if (stop_mask == ENDPOINT_STATE_CID_CMPLT)
			{
				PDEBUG(SANITY_INFO, "All endpoints have received CID complete ind");
				on_going = 0;
			}
		}
	}

	for (i = 0; i < j; i++)
	{
		if (test_level > TEST_LEVEL_NO_LB)
		{
			PDEBUG(SANITY_INFO, "VAPI_SetIPParams(%d) SYNC;\n", i);
			status = set_ip_udp_parameters(i, SYNC, 0);
			if (status != SUCCESS)
				PRINT_ERROR;
	
			PDEBUG(SANITY_INFO, "VAPI_SetIPParams(%d) SYNC;\n", i + j);
			status = set_ip_udp_parameters(i + j, SYNC, 0);
			if (status != SUCCESS)
				PRINT_ERROR;
	
			PDEBUG(SANITY_INFO, "VAPI_EnableConnection(%d) SYNC;\n",i);
			status = VAPI_EnableConnection(i, NULL);
			if (status != SUCCESS)
				PRINT_ERROR;
	
			PDEBUG(SANITY_INFO, "VAPI_EnableConnection(%d) ASYNC;\n",i + j);
			status = VAPI_EnableConnection(i + j, &local_request);
			if (status != VAPI_ERR_PENDING)
				PRINT_ERROR;

			/* Transcoding operation not supported by the M28xx or M821xx devices */
			if ((gtl_device_configuration->ucDevType != DEV_TYPE_M828XX) && (gtl_device_configuration->ucDevType != DEV_TYPE_M821XX) && (gtl_device_configuration->ucDevType != DEV_TYPE_M83XXX))
			{
				STranscodingOption transcoding_option = {
					.bAction = 1,
					.ucDSPMode = 1,
				};

				PDEBUG(SANITY_INFO, "VAPI_TranscodingSession Enable(%d) ASYNC;\n",i);
				status = VAPI_TranscodingSession(i, i + j, &transcoding_option, &local_request);
				if (status != VAPI_ERR_PENDING)
					PRINT_ERROR;
	
				PDEBUG(SANITY_INFO, "VAPI_SetPacketInterval(%d) ASYNC;\n",i);
				status = VAPI_SetPacketInterval(i, 40, &local_request);
				if (status != VAPI_ERR_PENDING)
					PRINT_ERROR;

				transcoding_option.bAction = 0;
				PDEBUG(SANITY_INFO, "VAPI_TranscodingSession Disable(%d) SYNC;\n",i);
				status = VAPI_TranscodingSession(i, i + j, &transcoding_option, NULL);
				if (status != SUCCESS)
					PRINT_ERROR;

				transcoding_option.ucVirtualTranscoding = 1;
				transcoding_option.bAction = 1;
				PDEBUG(SANITY_INFO, "VAPI_TranscodingSession Enable Virtual Transcoding (%d) ASYNC;\n",i);
				status = VAPI_TranscodingSession(i, i + j, &transcoding_option, &local_request);
				if (status != VAPI_ERR_PENDING)
					PRINT_ERROR;

				transcoding_option.bAction = 0;
				PDEBUG(SANITY_INFO, "VAPI_TranscodingSession Disable(%d) SYNC;\n",i);
				status = VAPI_TranscodingSession(i, i + j, &transcoding_option, NULL);
				if (status != SUCCESS)
					PRINT_ERROR;

			}
	
			PDEBUG(SANITY_INFO, "VAPI_EnableConnection(%d) ASYNC;\n",i);
			status = VAPI_EnableConnection(i, &local_request);
			if (status != VAPI_ERR_PENDING)
				PRINT_ERROR;
	
			PDEBUG(SANITY_INFO, "VAPI_EnableConnection(%d) ASYNC;\n",i + j);
			status = VAPI_EnableConnection(i + j, &local_request);
			if (status != VAPI_ERR_PENDING)
				PRINT_ERROR;
	
			PDEBUG(SANITY_INFO, "Enable inter channel loopback(%d) SYNC;\n",i);
			status = VAPI_Loopback(i, i + j, eINTER_CHNL_POST_ENCAPS, NULL);
			if (status != SUCCESS)
				PRINT_ERROR;
	
			PDEBUG(SANITY_INFO, "Disable inter channel loopback(%d) SYNC;\n",i);
			status = VAPI_Loopback(i, i + j, eREMOVE_INTER_CHNL_POSTENCAPS_LOOPBACK, NULL);
			if (status != SUCCESS)
				PRINT_ERROR;
	
			PDEBUG(SANITY_INFO, "Enable THC loopback(%d) SYNC;\n",i);
			status = VAPI_Loopback(i, i + j, eTHC, NULL);
			if (status != SUCCESS)
				PRINT_ERROR;
	
			PDEBUG(SANITY_INFO, "Disable THC loopback(%d) SYNC;\n",i);
			status = VAPI_Loopback(i, i + j, eREMOVE_THC_LOOPBACK, NULL);
			if (status != SUCCESS)
				PRINT_ERROR;
		}

		PDEBUG(SANITY_INFO, "destroy_endpoint(%d) SYNC;\n", i);
		set_endpoint_state(ENDPOINT_STATE_ON_HOOK, i);
		status = destroy_endpoint(i, SYNC, 0);
		if (status != SUCCESS)
			PRINT_ERROR;
	
		PDEBUG(SANITY_INFO, "destroy_endpoint(%d) ASYNC;\n",i + j);
		status = destroy_endpoint(i + j, ASYNC, REQUEST_ID_DESTROY_LOOPBACK);
		if (status != VAPI_ERR_PENDING)
			PRINT_ERROR;
	}

	PDEBUG(SANITY_INFO, "WAIT for all endpoints on hook");
	on_going = 1;
	while (on_going)
	{
		stop_mask = ENDPOINT_STATE_ON_HOOK;
		for (k = 0; k < number_of_endpoints; k++)
		{
			stop_mask &= endpoints[k].state;
		}

		if (stop_mask == ENDPOINT_STATE_ON_HOOK)
		{
			PDEBUG(SANITY_INFO, "All endpoints are on hook");
			on_going = 0;
		}
	}

	/* make sure we have a even number of endpoints */
	if (number_of_endpoints % 2)
		number_of_endpoints++;

	if (connection_create == 1)
	{
		for (i = 0; i < number_of_endpoints; i++)
		{
			/* odds endpoints are terminate side, even endpoints are originate side */
			if (i % 2)
				set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK, i);
			else
				set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK, i);
	
			PDEBUG(SANITY_INFO, "VAPI_AllocateConnection(%d) SYNC;\n",i + j);
			status = VAPI_AllocateConnection(device_id, i, eVOIP, eNarrowBand, 1, &endpoints[i].timeslot, NULL, NULL);

			if (status != SUCCESS)
			{ 
				PDEBUG(SANITY_ERROR, "Endpoint %d >>>>>>>>>>> error %d", endpoints[i].index, status); 
				sprintf(error_string, "Endpoint %d >>>>>>>>>>> error %d", endpoints[i].index, status);
			}
			else
				set_endpoint_event(ENDPOINT_EVENT_VOIP_CHANNEL_CREATED, i);
			/* make sure the connection is active to generate tones */
			status = VAPI_SetConnectionState(i, eTdmActive, NULL);
			if (status != VAPI_ERR_PENDING)
				PRINT_ERROR;
		}

		state_machine_handler(NULL);
	}
	else
	{
		for (i = 0; i < number_of_endpoints; i++)
		{
			/* odds endpoints are terminate side, even endpoints are originate side */
			if (i % 2)
			{
				set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK, i);
				status = create_voip_endpoint(i, ASYNC, ENDPOINT_STATE_WAIT_CREATE_CONNECTION_TERMINATE_ACK);
				if (status != VAPI_ERR_PENDING)
					PRINT_ERROR;
			}
			else
			{
				set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK, i);
				status = create_voip_endpoint(i, ASYNC, ENDPOINT_STATE_WAIT_CREATE_CONNECTION_ORIGINATE_ACK);
				if (status != VAPI_ERR_PENDING)
					PRINT_ERROR;
			}
		}

		state_machine_handler(NULL);
	}

create_conf:
	status = check_allocate_participant();
	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "check_allocate_participant %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error check_allocate_participant");

	status = check_conf_transcoding();
	if (status == SUCCESS)
		PDEBUG(SANITY_INFO, "check_conf_transcoding %d SUCCESS\n", device_id);
	else
		ERROR_EXIT(">>>>>>>>>>> error check_conf_transcoding");

	if (number_of_participants == 0)
		goto end_of_loop;

	PDEBUG(SANITY_INFO, "Start conferencing testing\n");
	for (i = 0; i < MAX_ENDPOINTS; i++)
	{
		endpoints[i].state = ENDPOINT_STATE_IDLE;
		endpoints[i].ptchng_count = 0;
	}

	status = VAPI_CreateConference(device_id, g_conference_id, &conference_parameters, True, &dominant_talkers, NULL);

	/* get at least 4 participants */
	if (number_of_participants < 4)
		number_of_participants = 4;

	for (i = 0; i < number_of_participants; i++)
	{
		/* odds participants are TDM side, even participants are IP side */
		if (i % 2)
		{
			set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_IP_PARTICIPANT_ACK, i);
			status = create_participant(i, ASYNC, ENDPOINT_STATE_WAIT_CREATE_IP_PARTICIPANT_ACK, g_conference_id, IP_SIDE);
			if (status != VAPI_ERR_PENDING)
				PRINT_ERROR;

		}
		else
		{
			set_endpoint_state(ENDPOINT_STATE_WAIT_CREATE_TDM_PARTICIPANT_ACK, i);
			status = create_participant(i, ASYNC, ENDPOINT_STATE_WAIT_CREATE_TDM_PARTICIPANT_ACK, g_conference_id, TDM_SIDE);
			if (status != VAPI_ERR_PENDING)
				PRINT_ERROR;
		}
	}

	conference_state_machine_handler(NULL);

	status = VAPI_DestroyConference(g_conference_id, NULL);

end_of_loop:
	number_of_loops--;
	loop_counter++;
	printf("Loops passed: %d\n", loop_counter);

	if (strcmp(error_string, "SUCCESS") != 0)
		main_exit(FAILURE);

	if (number_of_loops)
		goto main_loop;

	printf("GLOBAL error string = %s\n", error_string);
	/*if the global error string has been over written --> error*/
	if (strcmp(error_string, "SUCCESS") != 0)
		main_exit(FAILURE);
	else
		main_exit(SUCCESS);

error_exit:
	main_exit(status);

	return status;
}
