/*! 	\file readcfg.c
	\brief This file contains the helper functions to read the configuration file (-c option)  */
/* Copyright © 2004-2010 Mindspeed Technologies, Inc.
 * Mindspeed Confidential.
 * All rights reserved.
 *
 * This file is a component of the Mindspeed® VAPI software ("VAPI") and is
 * distributed under the Mindspeed Software License Agreement (the "Agreement").
 * Before using this file, you must agree to be bound by the the terms and conditions of 
 * the Agreement.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "readcfg.h"
#include "vapi_san_type.h"

#define _DEBUG_READ_CFG_	1
#undef	_DEBUG_READ_CFG_

int cfg_parse_line(int *line_n, char *line, struct _CFG **conf, struct _CFG **clast, struct _KEY **klast)
{
	struct _CFG *cnode;
	struct _KEY *knode;
	char *temp, *temp2, *temp3;

	(*line_n)++;
	temp = line + 1;

	if (*temp == '#')
		return 0;

	temp = (char *)strchr(temp, '#');
	if (temp)
		*temp = 0;

	temp = line + 1;
	while ((*temp == ' ') || (*temp == '\t'))
		temp++;

	temp2 = temp + strlen(temp) - 1;
	while ((*temp2 == ' ') || (*temp2 == '\t') || (*temp2 == '\n') || (*temp2 == '\r'))
		*temp2-- = 0;

	if (!*temp)		/* ignore blanks */
		return 0;

	if (*temp == '[')
	{
		temp++;

		temp2 = (char *)strchr(temp, ']');
		if (!temp2)
		{
			info("Config line #%i missing ], %s", *line_n, line);
			return -1;
		}

		*temp2-- = 0;
		if (*(temp2 + 2))
		{
			info("Junk after ] in line #%i, %s", *line_n, line);
			return -1;
		}

		while ((*temp2 == ' ') || (*temp2 == '\t'))
			*temp2-- = 0;

		if (!*temp)
		{
			info("Missing APP on line #%i, %s", *line_n, line);
			return -1;
		}

		for (cnode = *conf; cnode; cnode = cnode->next)
			if (strcasecmp(cnode->app, temp) == 0)
			{
				info("Duplicate APP values in line #%i, %s", *line_n, line);
				return -1;
			}

		cnode = (struct _CFG *)malloc(sizeof(struct _CFG));
		memset(cnode, 0, sizeof(struct _CFG));

		cnode->app = (char *)malloc(strlen(temp) + 1);
		strcpy(cnode->app, temp);

#ifdef _DEBUG_READ_CFG_
		info("%s", temp);
#endif
		if (!*clast)
			*conf = *clast = cnode;
		else
		{
			(*clast)->next = cnode;
			*clast = cnode;
		}

		*klast = knode = NULL;
	}
	else if ((temp2 = (char *)strchr(temp, '=')) != NULL)
	{
		if (!(*clast))
		{
			info("No APP value specified by line #%i, %s", *line_n, line);
			return -1;
		}

		temp3 = temp2 - 1;
		while ((*temp3 == ' ') || (*temp3 == '\t'))
			*temp3-- = 0;

		*temp2++ = 0;
		while ((*temp2 == ' ') || (*temp2 == '\t'))
			temp2++;

		if (!*temp)
		{
			info("Missing KEY on line #%i, %s", *line_n, line);
			return -1;
		}

		for (knode = (*clast)->key; knode; knode = knode->next)
			if (strcasecmp(knode->name, temp) == 0)
			{
				info("Duplicate KEY on line #%i, %s", *line_n, line);
				return -1;
			}

		knode = (struct _KEY *)malloc(sizeof(struct _KEY));
		memset(knode, 0, sizeof(struct _KEY));

		knode->name = (char *)malloc(strlen(temp) + 1);
		strcpy(knode->name, temp);
#ifdef _DEBUG_READ_CFG_
		info("%s ", temp);
#endif
		if (!*temp2)
		{
			info("Missing VALUE on line #%i, %s", *line_n, line);
			knode->value = (char *)malloc(strlen("0") + 1);
			sprintf(knode->value, "0");
		}
		else
		{
			knode->value = (char *)malloc(strlen(temp2) + 1);
			strcpy(knode->value, temp2);
#ifdef _DEBUG_READ_CFG_
			info("%s", knode->value);
#endif
		}

		if (!*klast)
			(*clast)->key = *klast = knode;
		else
		{
			(*klast)->next = knode;
			*klast = knode;
		}
	}
	else
	{
		info("Invalid config line #%i, %s", *line_n, line);
		return -1;
	}

	return 0;
}

/* Reads the config file and constructs a tree */
struct _CFG *cfg_read(const char *filename, int print_on)
{
	FILE *cf;
	struct _CFG *conf, *clast;
	struct _KEY *klast;
	char line[MAX_LINE_SIZE];
	int line_n;

	conf = clast = NULL;
	klast = NULL;

	cf = fopen(filename, "r");
	if (!cf)
	{
		if (print_on == 1)
		{
			info("Can't find config file \"%s\"", filename);
		}

		return (NULL);
	}

	line_n = 0;
	line[0] = '*';		/* this stops the space searchers from going too far */

	while (fgets(&line[1], MAX_LINE_SIZE - 1, cf) != NULL)
	{
		if (cfg_parse_line(&line_n, line, &conf, &clast, &klast) < 0)
		{
			conf = NULL;
			break;
		}
	}

	if (!conf)
		info("Invalid config file %s.", filename);

	fclose(cf);

	return (conf);
}

/* Reads the config file and constructs a tree */
struct _CFG *cfg_read_mem(char *app, char *data)
{
	struct _CFG *conf, *clast;
	struct _KEY *klast;
	char *data_copy;
	char *str_token;
	char line[MAX_LINE_SIZE];
	int line_n;

	conf = clast = NULL;
	klast = NULL;

	line_n = 0;
	line[0] = '*';		/* this stops the space searchers from going too far */

	/* workaround, add the section name here */
	UT_Snprintf(line + 1, MAX_LINE_SIZE - 1, "[%s]", app);

	if (cfg_parse_line(&line_n, line, &conf, &clast, &klast) < 0)
	{
		conf = NULL;
		goto out;
	}

	data_copy = strdup(data);

	str_token = strtok(data_copy, "\n");

	while (str_token != NULL)
	{
		strncpy(line + 1, str_token, MAX_LINE_SIZE - 1);

		if (cfg_parse_line(&line_n, line, &conf, &clast, &klast) < 0)
		{
			conf = NULL;
			break;
		}

		str_token = strtok(NULL, "\n");
	}

	free(data_copy);

      out:
	if (!conf)
		info("Invalid config record");

	return (conf);
}

struct _KEY *cfg_get_app(struct _CFG *cnode, const char *app)
{
	while (cnode)
	{

#ifdef _DEBUG_READ_CFG_
		info("cnode: %s %s", cnode->app, app);
#endif

		if (!strcmp(app, cnode->app))
			return cnode->key;

		cnode = cnode->next;
	}

	return NULL;
}

char *cfg_get_val_with_knode(struct _KEY *knode, const char *key_name)
{
	while (knode)
	{
#ifdef _DEBUG_READ_CFG_
		info("knode: %s %s", knode->name, key_name);
#endif

		if (!strcmp(key_name, knode->name))
			return knode->value;

		knode = knode->next;
	}

	return NULL;
}

char *cfg_get_val(struct _CFG *cfg, const char *app, const char *key_name)
{
	struct _KEY *knode;

	knode = cfg_get_app(cfg, app);
	if (knode == NULL)
		return NULL;

	return cfg_get_val_with_knode(knode, key_name);
}

int cfg_get_val_int(struct _KEY *knode, const char *key_name, int min_val, int max_val)
{
	char *value;
	int val;

	value = cfg_get_val_with_knode(knode, key_name);
	if (value != NULL)
	{
		val = strtol(value, NULL, 0);

		if ((val < min_val || val > max_val) && val != 0)
		{
			info("%s = %d out of range [%d; %d] => using min value %d",
			     key_name, val, min_val, max_val, min_val);
			return min_val;
		}
		else
			return val;
	}

	info("Missing %s => use min value: %d", key_name, min_val);

	return min_val;
}

void cfg_get_val_ip(struct _KEY *knode, const char *key_name, U8 * ip_addr, const char *default_ip)
{
	const char *value;
	char *ip_ascii;
	char *str_token;
	int i;

	value = cfg_get_val_with_knode(knode, key_name);
	if (value == NULL)
	{
		info("Missing %s => use default value: %s", key_name, default_ip);
		value = default_ip;
	}

	ip_ascii = strdup(value);
	str_token = strtok(ip_ascii, ".");
	for (i = 0; i < 4; i++)
	{
		ip_addr[i] = strtol(str_token, NULL, 10);
		str_token = strtok(NULL, ".");
	}

	free(ip_ascii);
}

void cfg_get_val_mac(struct _KEY *knode, const char *key_name, U8 * mac_addr, U8 * default_mac_addr)
{
	char *value;
	char *mac_ascii;
	char *str_token;
	int i;

	value = cfg_get_val_with_knode(knode, key_name);
	if (value == NULL)
	{
		info("Missing %s => use default value", key_name);
		memcpy(mac_addr, default_mac_addr, 6);
	}
	else
	{
		mac_ascii = strdup(value);
		str_token = strtok(mac_ascii, ":");

		for (i = 0; i < 6; i++)
		{
			mac_addr[i] = 0;
			mac_addr[i] |= strtol(str_token, NULL, 16);
			str_token = strtok(NULL, ":");
		}
	}
}

void cfg_print(struct _CFG *conf)
{
	struct _CFG *cnode;
	struct _KEY *knode;

	while (conf)
	{
		info("[%s]", conf->app);
		while (conf->key)
		{
			knode = conf->key;

			info("%s=%s", knode->name, knode->value);

			conf->key = knode->next;
		}

		info("");

		cnode = conf;
		conf = conf->next;
	}
}

/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ CFGFILE Library
 *  cfg_clean
 *
 *  Purpose :	
 *  Input   :	
 *  Output  :	
 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 */
void cfg_clean(struct _CFG *conf)
{
	struct _CFG *cnode;
	struct _KEY *knode;

	while (conf)
	{
		while (conf->key)
		{
			knode = conf->key;
			conf->key = knode->next;
			free(knode->value);
			free(knode->name);
			free(knode);
		}

		cnode = conf;
		conf = conf->next;
		free(cnode->app);
		free(cnode);
	}
}

/**
 * print_ip_addr -
 *
 *
 */
void print_ip_addr(const char *prefix, U8 * ip_addr)
{
	info("%s%u.%u.%u.%u", prefix ? prefix : "", ip_addr[0], ip_addr[1], ip_addr[2], ip_addr[3]);
}

/**
 * print_mac_addr -
 *
 *
 */
void print_mac_addr(const char *prefix, U8 * mac_addr)
{
	info("%s%02x:%02x:%02x:%02x:%02x:%02x",
	     prefix ? prefix : "", mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5]);
}
